/*
 * This file is part of the Carpet REMS Addition project, licensed under the
 * GNU Lesser General Public License v3.0
 *
 * Copyright (C) 2025 A Minecraft Server and contributors
 *
 * Carpet REMS Addition is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Carpet REMS Addition is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Carpet REMS Addition. If not, see <https://www.gnu.org/licenses/>.
 */

package rems.carpet.mixins.SignCommand;

import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.ParseResults;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import rems.carpet.REMSServer;
import rems.carpet.REMSSettings;
import rems.carpet.utils.SignCommand;

import java.util.*;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1657;
import net.minecraft.class_1802;
import net.minecraft.class_1937;
import net.minecraft.class_2165;
import net.minecraft.class_2168;
import net.minecraft.class_2338;
import net.minecraft.class_2478;
import net.minecraft.class_2561;
import net.minecraft.class_2625;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3965;
import net.minecraft.class_8242;

@Mixin(class_2478.class)
public class AbstractSignBlockMixin {
    @Unique
    private static final Set<String> ALLOWED_COMMANDS = new HashSet<>(Arrays.asList(
            "say",
            "player",
            "tick"
    ));

    @Inject(
            method = "onUse",
            at = @At("HEAD"),
            cancellable = true
    )
    private void onUse(class_2680 state, class_1937 world, class_2338 pos, class_1657 player, class_1268 hand, class_3965 hit, CallbackInfoReturnable<class_1269> ci) {
        if (REMSSettings.SignCommand) {
            if (world.method_8321(pos) instanceof class_2625 signBlockEntity) {
                // 获取过滤标识（关键新增参数）
                boolean filtered = player.method_33793();

                // 获取正确的 SignText 对象
                boolean isFront = signBlockEntity.method_49834(player);
                class_8242 signText = isFront ?
                        signBlockEntity.method_49853() :  // 使用明确方法获取
                        signBlockEntity.method_49854();

                // 传递两个参数（关键修复）
                String fullCommand = processSignText(signText, filtered);

                // 第二步：验证命令格式
                if (!fullCommand.startsWith("/")) {
                    return;
                }

                // 第三步：白名单验证
                String actualCommand = fullCommand.substring(1);
                if (!isCommandAllowed(actualCommand)) {
                    player.method_7353(class_2561.method_43470("§c该指令未被允许通过告示牌执行"), false);
                    ci.setReturnValue(class_1269.method_29236(true));
                    return;
                }

                // 第四步：执行条件检查
                if (player.method_6047().method_31574(class_1802.field_8162) && !player.method_5715()) {
                    ci.setReturnValue(class_1269.method_29236(true));
                    executeValidatedCommand(player, actualCommand);
                }
            }
        }
    }

    // 文本处理核心方法
    @Unique
    private String processSignText(class_8242 signText, boolean filtered) {
        StringBuilder commandBuilder = new StringBuilder();

        // 获取过滤后的文本列表
        List<class_2561> textLines = List.of(signText.method_49877(filtered));

        for (int i = 0; i < textLines.size(); i++) {
            String line = textLines.get(i).getString()
                    .replaceAll("§.", "")       // 移除所有颜色代码
                    .replaceAll("[\\x00-\\x1F]", "") // 过滤控制字符
                    .trim();

            // 处理空行
            if (line.isEmpty()) continue;

            // 处理续行符（行末的\）
            if (line.endsWith("\\")) {
                commandBuilder.append(line, 0, line.length() - 1);
            } else {
                commandBuilder.append(line);
                // 在行尾添加空格（最后一行除外）
                if (i != textLines.size() - 1) commandBuilder.append(" ");
            }
        }

        return commandBuilder.toString()
                .replaceAll("\\s+", " ") // 合并连续空格
                .trim();
    }

    // 白名单验证逻辑
    @Unique
    private boolean isCommandAllowed(String rawCommand) {
        // 提取基础命令
        String[] parts = rawCommand.split(" ", 2);
        String baseCommand = parts[0].toLowerCase();

        // 处理命名空间（如 minecraft:give → give）
        int colonIndex = baseCommand.indexOf(':');
        if (colonIndex != -1) {
            baseCommand = baseCommand.substring(colonIndex + 1);
        }

        return ALLOWED_COMMANDS.contains(baseCommand);
    }

    // 命令执行方法
    @Unique
    private void executeValidatedCommand(class_1657 player, String command) {
        class_3218 world = Objects.requireNonNull(player.method_5682()).method_30002();;
        class_2168 commandSource = new class_2168(
                class_2165.field_17395,
                player.method_19538(),
                player.method_5802(),
                world,
                4,
                player.method_5477().getString(),
                player.method_5476(),
                world.method_8503(),
                player
        );

        player.method_5682().execute(() -> {
            try {
                CommandDispatcher<class_2168> dispatcher = commandSource.method_9211().method_3734().method_9235();
                ParseResults<class_2168> results = dispatcher.parse(command, commandSource);

                if (results.getExceptions().isEmpty()) {
                    dispatcher.execute(results);
                } else {
                    player.method_7353(class_2561.method_43470("§c指令语法错误"), false);
                }
            } catch (CommandSyntaxException e) {
                player.method_7353(class_2561.method_43470("§c执行失败: " + e.getMessage()), false);
            }
        });
    }
}

