package net.typho.vibrancy;

import com.google.gson.*;
import net.minecraft.core.BlockPos;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.Vec3;
import net.typho.vibrancy.light.AbstractPointLight;
import net.typho.vibrancy.light.BlockPointLight;
import org.joml.Vector3f;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;

public record DynamicLightInfo(Vector3f color, BlockStateFunction<Optional<Float>> radius, BlockStateFunction<Optional<Float>> brightness, BlockStateFunction<Optional<Float>> flicker, BlockStateFunction<Optional<Vec3>> offset) {
    public static final Map<Predicate<BlockState>, Function<BlockState, DynamicLightInfo>> MAP = new LinkedHashMap<>();

    public static DynamicLightInfo get(BlockState state) {
        return MAP.entrySet().stream()
                .filter(entry -> entry.getKey().test(state))
                .findAny()
                .map(entry -> entry.getValue().apply(state))
                .orElse(null);
    }

    public BlockPointLight createBlockLight(BlockPos pos, BlockState state) {
        return (BlockPointLight) initLight(new BlockPointLight(
                pos
        ).setFlicker(flicker().apply(state).orElse(0f)), state);
    }

    public void addBlockLight(BlockPos pos, BlockState state) {
        BlockPointLight old = Vibrancy.BLOCK_LIGHTS.get(pos);

        if (old == null) {
            Vibrancy.BLOCK_LIGHTS.put(pos, createBlockLight(pos, state));
        }
    }

    public <L extends AbstractPointLight> L initLight(L light, BlockState state) {
        float b = brightness.apply(state).orElse(1f);
        light.setColor(color().mul(b, new Vector3f()))
                .setRadius(radius.apply(state).orElse((float) state.getLightEmission()));
        return light;
    }

    public static class Builder {
        public Vector3f color;
        public BlockStateFunction<Optional<Float>> radius = state -> Optional.empty(), brightness = state -> Optional.empty(), flicker = state -> Optional.empty();
        public BlockStateFunction<Optional<Vec3>> offset = state -> Optional.empty();

        public Builder color(Vector3f color) {
            this.color = color;
            return this;
        }

        public Builder radius(BlockStateFunction<Optional<Float>> radius) {
            BlockStateFunction<Optional<Float>> radius1 = this.radius;
            this.radius = state -> radius.apply(state).or(() -> radius1.apply(state));
            return this;
        }

        public Builder brightness(BlockStateFunction<Optional<Float>> brightness) {
            BlockStateFunction<Optional<Float>> brightness1 = this.brightness;
            this.brightness = state -> brightness.apply(state).or(() -> brightness1.apply(state));
            return this;
        }

        public Builder flicker(BlockStateFunction<Optional<Float>> flicker) {
            BlockStateFunction<Optional<Float>> flicker1 = this.flicker;
            this.flicker = state -> flicker.apply(state).or(() -> flicker1.apply(state));
            return this;
        }

        public Builder offset(BlockStateFunction<Optional<Vec3>> offset) {
            BlockStateFunction<Optional<Vec3>> offset1 = this.offset;
            this.offset = state -> offset.apply(state).or(() -> offset1.apply(state));
            return this;
        }

        public Builder copy(DynamicLightInfo info) {
            return color(info.color)
                    .radius(info.radius)
                    .brightness(info.brightness)
                    .flicker(info.flicker)
                    .offset(info.offset);
        }

        public Builder then(Block block, String key, JsonElement value) {
            return switch (key) {
                case "color" -> {
                    if (value.isJsonArray()) {
                        JsonArray array = value.getAsJsonArray();

                        if (array.size() != 3) {
                            throw new JsonParseException("Expected a 3-element array for color while parsing dynamic light info while parsing " + block + ", got " + array.size() + " elements");
                        }

                        yield color(new Vector3f(
                                array.get(0).getAsJsonPrimitive().getAsFloat(),
                                array.get(1).getAsJsonPrimitive().getAsFloat(),
                                array.get(2).getAsJsonPrimitive().getAsFloat()
                        ));
                    } else {
                        throw new JsonParseException("Expected a 3-element array for color while parsing dynamic light info while parsing " + block + ", got " + value);
                    }
                }
                case "offset" -> {
                    if (value.isJsonNull()) {
                        offset = state -> Optional.empty();
                        yield this;
                    } else if (value.isJsonObject() || value.isJsonArray()) {
                        yield offset(BlockStateFunction.parseJson(block, value, json -> {
                            if (json.isJsonArray()) {
                                JsonArray array = json.getAsJsonArray();

                                if (array.size() != 3) {
                                    throw new JsonParseException("Expected a 3-element array for offset while parsing dynamic light info while parsing " + block + ", got " + array.size() + " elements");
                                }

                                return Optional.of(new Vec3(
                                        array.get(0).getAsJsonPrimitive().getAsDouble(),
                                        array.get(1).getAsJsonPrimitive().getAsDouble(),
                                        array.get(2).getAsJsonPrimitive().getAsDouble()
                                ));
                            } else {
                                throw new JsonParseException("Expected a 3-element array for offset while parsing dynamic light info while parsing " + block + ", got " + json);
                            }
                        }, Optional::empty));
                    } else {
                        throw new JsonParseException("Expected a 3-element array for offset while parsing dynamic light info while parsing " + block + ", got " + value);
                    }
                }
                case "radius" -> {
                    if (value.isJsonNull()) {
                        radius = state -> Optional.empty();
                        yield this;
                    } else if (value.isJsonObject() || value.isJsonPrimitive()) {
                        yield radius(BlockStateFunction.parseJson(block, value, json -> {
                            if (json.isJsonPrimitive()) {
                                return Optional.of(json.getAsJsonPrimitive().getAsFloat());
                            } else {
                                throw new JsonParseException("Expected a float for radius while parsing dynamic light info while parsing " + block + ", got " + json);
                            }
                        }, Optional::empty));
                    } else {
                        throw new JsonParseException("Expected a float for radius while parsing dynamic light info while parsing " + block + ", got " + value);
                    }
                }
                case "brightness" -> {
                    if (value.isJsonNull()) {
                        brightness = state -> Optional.empty();
                        yield this;
                    } else if (value.isJsonObject() || value.isJsonPrimitive()) {
                        yield brightness(BlockStateFunction.parseJson(block, value, json -> {
                            if (json.isJsonPrimitive()) {
                                return Optional.of(json.getAsJsonPrimitive().getAsFloat());
                            } else {
                                throw new JsonParseException("Expected a float for brightness while parsing dynamic light info while parsing " + block + ", got " + json);
                            }
                        }, Optional::empty));
                    } else {
                        throw new JsonParseException("Expected a float for brightness while parsing dynamic light info while parsing " + block + ", got " + value);
                    }
                }
                case "flicker" -> {
                    if (value.isJsonNull()) {
                        flicker = state -> Optional.empty();
                        yield this;
                    } else if (value.isJsonObject() || value.isJsonPrimitive()) {
                        yield flicker(BlockStateFunction.parseJson(block, value, json -> {
                            if (json.isJsonPrimitive()) {
                                return Optional.of(json.getAsJsonPrimitive().getAsFloat());
                            } else {
                                throw new JsonParseException("Expected a float for flicker while parsing dynamic light info while parsing " + block + ", got " + json);
                            }
                        }, Optional::empty));
                    } else {
                        throw new JsonParseException("Expected a float for flicker while parsing dynamic light info while parsing " + block + ", got " + value);
                    }
                }
                default -> this;
            };
        }

        public Builder load(Block block, JsonElement json) {
            if (json.isJsonObject()) {
                JsonObject jsonObj = json.getAsJsonObject();

                JsonElement copy = jsonObj.get("copy");

                if (copy != null) {
                    if (copy.isJsonPrimitive()) {
                        JsonPrimitive primitive = copy.getAsJsonPrimitive();

                        if (primitive.isString()) {
                            String s = primitive.getAsString();

                            if (s.startsWith("#")) {
                                throw new IllegalArgumentException("Can't copy the dynamic light info of a tag (for technical reasons)");
                            }

                            copy(get(BuiltInRegistries.BLOCK.get(ResourceLocation.parse(s)).defaultBlockState()));
                        } else {
                            throw new JsonParseException("Expected a string for copy while parsing dynamic light info for \"" + block + "\", got " + primitive);
                        }
                    } else {
                        throw new JsonParseException("Expected a string for copy while parsing dynamic light info for \"" + block + "\", got " + copy);
                    }
                }

                jsonObj.asMap().forEach((key1, value) -> then(block, key1, value));
                return this;
            } else if (json.isJsonPrimitive()) {
                JsonPrimitive primitive = json.getAsJsonPrimitive();

                if (primitive.isString()) {
                    String s = primitive.getAsString();

                    if (s.startsWith("#")) {
                        throw new IllegalArgumentException("Can't copy the dynamic light info of a tag (for technical reasons)");
                    }

                    return copy(get(BuiltInRegistries.BLOCK.get(ResourceLocation.parse(s)).defaultBlockState()));
                }
            }

            throw new JsonParseException("Expected either a string or object while parsing dynamic light info for \"" + block + "\", got " + json);
        }

        public DynamicLightInfo build() {
            return new DynamicLightInfo(color, radius, brightness, flicker, offset);
        }
    }
}
