package net.anawesomguy.musicbox.item;

import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.Lifecycle;
import com.mojang.serialization.ListBuilder;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import it.unimi.dsi.fastutil.ints.IntList;
import it.unimi.dsi.fastutil.shorts.ShortArrayList;
import it.unimi.dsi.fastutil.shorts.ShortList;
import org.apache.commons.lang3.StringUtils;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Stream;
import net.minecraft.class_124;
import net.minecraft.class_1792;
import net.minecraft.class_1836;
import net.minecraft.class_2561;
import net.minecraft.class_5699;
import net.minecraft.class_9299;
import net.minecraft.class_9473;

public final class MusicBoxDataComponent implements class_9299 {
    public static final Codec<short[]> NOTES_CODEC = Codec.of(MusicBoxDataComponent::encodeNotes,
                                                              MusicBoxDataComponent::decodeNotes);
    public static final Codec<MusicBoxDataComponent> CODEC = RecordCodecBuilder.create(instance -> instance.group(
        Codec.INT.optionalFieldOf("key_offset", 0).forGetter(MusicBoxDataComponent::getKeyOffset),
        Codec.BOOL.optionalFieldOf("minor", Boolean.FALSE).forGetter(MusicBoxDataComponent::isInMinor),
        class_5699.field_33442.fieldOf("ticks_per_beat").forGetter(MusicBoxDataComponent::getTicksPerBeat),
        Codec.STRING.optionalFieldOf("song_artist", "").forGetter(MusicBoxDataComponent::getArtist),
        Codec.STRING.fieldOf("song_name").forGetter(MusicBoxDataComponent::getSongName),
        NOTES_CODEC.fieldOf("notes").forGetter(MusicBoxDataComponent::getNotes)
    ).apply(instance, MusicBoxDataComponent::new));

    public static final int TOTAL_BEATS = 36; // length of the music
    public static final int NOTES_RANGE = 15; // the amount of notes each value in `notes` represents

    private static final int[] MAJOR_OFFSETS = {0, 2, 4, 5, 7, 9, 11}; // semi-tone offsets in a major key
    private static final int[] MINOR_OFFSETS = {0, 2, 3, 5, 7, 8, 10}; // semi-tone offsets in a minor key
    private static final int SCALE_LENGTH = 7; // MAJOR_OFFSETS.length

    private final int keyOffset;
    private final boolean minor;
    private final int ticksPerBeat; // the music box is always in x/4 tempo so one beat is a quarter note
    private final String artist;
    private final String songName;
    /**
     * An array of all the notes. The first element of the array would be played on the first {@link #ticksPerNote}, the second on the second, and so on.
     * <p>
     * For example, if {@link #keyOffset} was 0, then the first bit would represent C4, the second bit D4, the eighth C5, and so on.
     * <p>
     * The music box can only play 2 and a half octaves, so the sign bit is ignored (shorts have 16 bits)
     */
    private final short[] notes;
    private final int ticksPerNote; // ticks per each value in `notes`

    public MusicBoxDataComponent(int keyOffset, boolean minor, int ticksPerBeat, String artist, String song) {
        this.keyOffset = keyOffset;
        this.minor = minor;
        if (ticksPerBeat <= 0)
            throw new IllegalArgumentException("ticksPerBeat is not positive");
        this.ticksPerBeat = ticksPerBeat;
        this.artist = Objects.requireNonNull(artist);
        this.songName = Objects.requireNonNull(song);
        // basically, if it's divisible by 2, then ticksPerNote = ticksPerBeat / 2,
        // if it's divisible by 4, then it's ticksPerBeat / 4,
        // and the same for 8
        // otherwise ticksPerNote = ticksPerBeat (and only quarter notes are usable)
        int ticksPerNote = this.ticksPerNote = (ticksPerBeat % 2 == 0) ? (ticksPerBeat / (ticksPerBeat % 4 == 0 ? (ticksPerBeat % 8 == 0 ? 8 : 4) : 2)) : ticksPerBeat;
        this.notes = new short[TOTAL_BEATS * (ticksPerBeat / ticksPerNote)];
    }

    public MusicBoxDataComponent(int keyOffset, boolean minor, int ticksPerBeat, String artist, String song, short[] notes) {
        this(keyOffset, minor, ticksPerBeat, artist, song);
        System.arraycopy(notes, 0, this.notes, 0, Math.min(notes.length, this.notes.length));
    }

    public void getSemitones(int index, IntList output) {
        output.clear();
        short notes = this.notes[index];
        if (notes == 0 || notes == -32768)
            return;
        int i = NOTES_RANGE;
        while (i-- > 0) { // 14, 13, ... 1, 0
            if ((notes & 1) == 1) {
                int semitoneOffset = (minor ? MINOR_OFFSETS : MAJOR_OFFSETS)[i % SCALE_LENGTH] + keyOffset;
                output.add(semitoneOffset);
            }
            notes >>>= 1;
        }
    }

    public String getArtist() {
        return artist;
    }

    public String getSongName() {
        return songName;
    }

    // it's ok to modify if you know what you're doing, actually
    public short[] getNotes() {
        return notes;
    }

    public int getKeyOffset() {
        return keyOffset;
    }

    public boolean isInMinor() {
        return minor;
    }

    public int getTicksPerBeat() {
        return ticksPerBeat;
    }

    public int getTicksPerNote() {
        return ticksPerNote;
    }

    public static <T> DataResult<T> encodeNotes(short[] notes, DynamicOps<T> ops, T prefix) {
        ListBuilder<T> list = ops.listBuilder();
        for (int i = 0, j = 0, len = notes.length; i < len; i++) {
            short note = notes[i];
            if (note == 0 || note == -32768) {
                j++;
            } else {
                if (j > 0) {
                    list.add(ops.createInt(j));
                    j = 0;
                }
                list.add(ops.createString(
                    StringUtils.leftPad(
                        StringUtils.substring(Integer.toBinaryString(note), -NOTES_RANGE),
                        NOTES_RANGE,
                        '0'
                    )));
            }
        }
        return list.build(prefix);
    }

    public static <T> DataResult<Pair<short[], T>> decodeNotes(DynamicOps<T> ops, T input) {
        return ops.getList(input).setLifecycle(Lifecycle.stable()).map(stream -> {
            ShortList list = new ShortArrayList(72);
            Stream.Builder<T> failed = Stream.builder();
            stream.accept(t -> {
                DataResult<String> oString = ops.getStringValue(t);
                if (oString.isSuccess()) {
                    String str = oString.getOrThrow();
                    short s;
                    try {
                        s = Short.parseShort(String.format("+%.15s", str), 2);
                    } catch (NumberFormatException e) {
                        failed.add(ops.createString("Invalid binary input, replacing with 0: " + str));
                        s = 0;
                    }
                    list.add(s);
                    return;
                }

                DataResult<Number> oNum = ops.getNumberValue(t);
                if (oNum.isSuccess()) {
                    int s = oNum.getOrThrow().intValue();
                    while (s-- > 0) {
                        list.add((short)0);
                    }
                }

            });
            return Pair.of(list.toShortArray(), ops.createList(failed.build()));
        });
    }

    @Override
    public void method_57409(class_1792.class_9635 context, Consumer<class_2561> textConsumer, class_1836 type, class_9473 components) {
        if (!songName.isEmpty()) {
            class_2561 songText = class_2561.method_43470(songName).method_27695(class_124.field_1056, class_124.field_1080);
            class_2561 text = artist.isEmpty() ?
                songText :
                class_2561.method_43470(artist + " - ").method_27692(class_124.field_1080).method_10852(songText);
            textConsumer.accept(text);
        }
    }
}
