package cc.cassian.raspberry.blocks;

import com.google.common.collect.ImmutableMap;
import net.minecraft.Util;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.world.item.context.BlockPlaceContext;
import net.minecraft.world.level.BlockGetter;
import net.minecraft.world.level.LevelAccessor;
import net.minecraft.world.level.LevelReader;
import net.minecraft.world.level.block.*;
import net.minecraft.world.level.block.state.BlockBehaviour;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.StateDefinition;
import net.minecraft.world.level.block.state.properties.*;
import net.minecraft.world.phys.shapes.*;

import java.awt.*;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

public class FlowerGarlandBlock extends Block {
    public static final BooleanProperty NORTH = BlockStateProperties.f_61368_;
    public static final BooleanProperty EAST = BlockStateProperties.f_61369_;
    public static final BooleanProperty SOUTH = BlockStateProperties.f_61370_;
    public static final BooleanProperty WEST = BlockStateProperties.f_61371_;
    public static final Map<Direction, BooleanProperty> PROPERTY_BY_DIRECTION = PipeBlock.f_55154_.entrySet().stream().filter((entry) -> (entry.getKey() != Direction.DOWN && entry.getKey() != Direction.UP)).collect(Util.m_137448_());;
    private static final VoxelShape WEST_AABB = Block.m_49796_(0.0F, 9.0F, 0.0F, 1.0F, 15.0F, 16.0F);
    private static final VoxelShape EAST_AABB = Block.m_49796_(15.0F, 9.0F, 0.0F, 16.0F, 15.0F, 16.0F);
    private static final VoxelShape NORTH_AABB = Block.m_49796_(0.0F, 9.0F, 0.0F, 16.0F, 15.0F, 1.0F);
    private static final VoxelShape SOUTH_AABB = Block.m_49796_(0.0F, 9.0F, 15.0F, 16.0F, 15.0F, 16.0F);
    private static final VoxelShape SUPPORT_SHAPE = Block.m_49796_(0.0F, 13.0F, 0.0F, 16.0F, 16.0F, 16.0F);
    private final Map<BlockState, VoxelShape> shapesCache;

    public FlowerGarlandBlock(BlockBehaviour.Properties properties) {
        super(properties);
        this.m_49959_(this.f_49792_.m_61090_()
                .m_61124_(NORTH, false)
                .m_61124_(EAST, false)
                .m_61124_(SOUTH, false)
                .m_61124_(WEST, false));
        this.shapesCache = ImmutableMap.copyOf(this.f_49792_.m_61056_().stream().collect(Collectors.toMap(Function.identity(), cc.cassian.raspberry.blocks.FlowerGarlandBlock::calculateShape)));
    }

    private static VoxelShape calculateShape(BlockState state) {
        VoxelShape voxelshape = Shapes.m_83040_();
        if (state.m_61143_(NORTH)) {
            voxelshape = Shapes.m_83110_(voxelshape, NORTH_AABB);
        }

        if (state.m_61143_(SOUTH)) {
            voxelshape = Shapes.m_83110_(voxelshape, SOUTH_AABB);
        }

        if (state.m_61143_(EAST)) {
            voxelshape = Shapes.m_83110_(voxelshape, EAST_AABB);
        }

        if (state.m_61143_(WEST)) {
            voxelshape = Shapes.m_83110_(voxelshape, WEST_AABB);
        }

        return voxelshape.m_83281_() ? Shapes.m_83144_() : voxelshape;
    }

    @Override
    public VoxelShape m_5940_(BlockState state, BlockGetter level, BlockPos pos, CollisionContext context) {
        return this.shapesCache.get(state);
    }

    @Override
    public boolean m_7420_(BlockState state, BlockGetter level, BlockPos pos) {
        return true;
    }

    @Override
    public boolean m_7898_(BlockState state, LevelReader level, BlockPos pos) {
        return this.hasFaces(this.getUpdatedState(state, level, pos));
    }

    private boolean hasFaces(BlockState state) {
        return this.countFaces(state) > 0;
    }

    private int countFaces(BlockState state) {
        int i = 0;

        for(BooleanProperty booleanproperty : PROPERTY_BY_DIRECTION.values()) {
            if (state.m_61143_(booleanproperty)) {
                ++i;
            }
        }

        return i;
    }

    private boolean canSupportAtFace(BlockGetter level, BlockPos pos, Direction direction) {
        if (direction.m_122434_() == Direction.Axis.Y) {
            return false;
        } else {
            BlockPos blockpos = pos.m_121945_(direction);
            return isAcceptableNeighbour(level, blockpos, direction);
        }
    }

    public static boolean isAcceptableNeighbour(BlockGetter blockReader, BlockPos neighborPos, Direction attachedFace) {
        BlockState state = blockReader.m_8055_(neighborPos);

        VoxelShape neighborSupportFaceShape = state.m_60816_(blockReader, neighborPos).m_83263_(attachedFace.m_122424_());
        VoxelShape neighborCollisionFaceShape = state.m_60812_(blockReader, neighborPos).m_83263_(attachedFace.m_122424_());

        // Check that neighbour shape covers support shape
        return !Shapes.m_83157_(neighborCollisionFaceShape, SUPPORT_SHAPE, BooleanOp.f_82683_) || !Shapes.m_83157_(neighborSupportFaceShape, SUPPORT_SHAPE, BooleanOp.f_82683_);
    }

    private BlockState getUpdatedState(BlockState state, BlockGetter level, BlockPos pos) {
        BlockPos blockpos = pos.m_7494_();
        BlockState blockstate = null;

        for(Direction direction : Direction.Plane.HORIZONTAL) {
            BooleanProperty booleanproperty = getPropertyForFace(direction);
            if (state.m_61143_(booleanproperty)) {
                boolean flag = this.canSupportAtFace(level, pos, direction);
                if (!flag) {
                    if (blockstate == null) {
                        blockstate = level.m_8055_(blockpos);
                    }

                    flag = blockstate.m_60713_(this) && blockstate.m_61143_(booleanproperty);
                }

                state = state.m_61124_(booleanproperty, flag);
            }
        }

        return state;
    }

    @Override
    public BlockState m_7417_(BlockState state, Direction direction, BlockState neighborState, LevelAccessor level, BlockPos currentPos, BlockPos neighborPos) {
        if (direction.m_122434_() == Direction.Axis.Y) {
            return super.m_7417_(state, direction, neighborState, level, currentPos, neighborPos);
        } else {
            BlockState blockstate = this.getUpdatedState(state, level, currentPos);
            return !this.hasFaces(blockstate) ? Blocks.f_50016_.m_49966_() : blockstate;
        }
    }

    @Override
    public boolean m_6864_(BlockState blockState, BlockPlaceContext blockPlaceContext) {
        if (!blockPlaceContext.m_7078_() && blockPlaceContext.m_43722_().m_150930_(this.m_5456_())) {
            return this.countFaces(blockState) < PROPERTY_BY_DIRECTION.size();
        } else {
            return super.m_6864_(blockState, blockPlaceContext);
        }
    }

    @Override
    public BlockState m_5573_(BlockPlaceContext context) {
        BlockState blockstate = context.m_43725_().m_8055_(context.m_8083_());
        boolean flag = blockstate.m_60713_(this);
        BlockState blockstate1 = flag ? blockstate : this.m_49966_();

        for(Direction direction : context.m_6232_()) {
            if (direction.m_122434_() != Direction.Axis.Y) {
                BooleanProperty booleanproperty = getPropertyForFace(direction);
                boolean flag1 = flag && blockstate.m_61143_(booleanproperty);
                if (!flag1 && this.canSupportAtFace(context.m_43725_(), context.m_8083_(), direction)) {
                    return blockstate1.m_61124_(booleanproperty, true);
                }
            }
        }

        return flag ? blockstate1 : null;
    }

    @Override
    protected void m_7926_(StateDefinition.Builder<Block, BlockState> builder) {
        builder.m_61104_(NORTH, EAST, SOUTH, WEST);
    }

    @Override
    public BlockState m_6843_(BlockState state, Rotation rotation) {
        switch (rotation) {
            case CLOCKWISE_180 -> {
                return (((state.m_61124_(NORTH, state.m_61143_(SOUTH))).m_61124_(EAST, state.m_61143_(WEST))).m_61124_(SOUTH, state.m_61143_(NORTH))).m_61124_(WEST, state.m_61143_(EAST));
            }
            case COUNTERCLOCKWISE_90 -> {
                return (((state.m_61124_(NORTH, state.m_61143_(EAST))).m_61124_(EAST, state.m_61143_(SOUTH))).m_61124_(SOUTH, state.m_61143_(WEST))).m_61124_(WEST, state.m_61143_(NORTH));
            }
            case CLOCKWISE_90 -> {
                return (((state.m_61124_(NORTH, state.m_61143_(WEST))).m_61124_(EAST, state.m_61143_(NORTH))).m_61124_(SOUTH, state.m_61143_(EAST))).m_61124_(WEST, state.m_61143_(SOUTH));
            }
            default -> {
                return state;
            }
        }
    }

    @Override
    public BlockState m_6943_(BlockState state, Mirror mirror) {
        switch (mirror) {
            case LEFT_RIGHT -> {
                return (state.m_61124_(NORTH, state.m_61143_(SOUTH))).m_61124_(SOUTH, state.m_61143_(NORTH));
            }
            case FRONT_BACK -> {
                return (state.m_61124_(EAST, state.m_61143_(WEST))).m_61124_(WEST, state.m_61143_(EAST));
            }
            default -> {
                return super.m_6943_(state, mirror);
            }
        }
    }

    public static BooleanProperty getPropertyForFace(Direction face) {
        return PROPERTY_BY_DIRECTION.get(face);
    }
}
