/* The MIT License (MIT)

Copyright (c) 2025 

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */

package cc.cassian.raspberry.compat.vanillabackport.leash;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import cc.cassian.raspberry.mixin.minecraft.EntityAccessor;
import cc.cassian.raspberry.mixin.minecraft.PathfinderMobAccessor;
import net.minecraft.Util;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.sounds.SoundSource;
import net.minecraft.world.entity.*;
import net.minecraft.world.entity.ai.goal.Goal;
import net.minecraft.world.entity.animal.horse.AbstractChestedHorse;
import net.minecraft.world.entity.animal.horse.AbstractHorse;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.Vec3;

import java.util.*;
import java.util.function.Function;
import java.util.function.Predicate;

public interface Leashable {
    Map<Predicate<Entity>, Function<Entity, Vec3[]>> QUAD_LEASH_OFFSETS = Util.m_137537_(() -> {
        ImmutableMap.Builder<Predicate<Entity>, Function<Entity, Vec3[]>> offsets = new ImmutableMap.Builder<>();
        offsets.put(entity -> entity instanceof AbstractHorse, entity -> Leashable.createQuadLeashOffsets(entity, 0.04, 0.52, 0.23, 0.87));
        offsets.put(entity -> entity instanceof AbstractChestedHorse, entity -> Leashable.createQuadLeashOffsets(entity, 0.04, 0.41, 0.18, 0.73));
        return offsets.build();
    });

    Vec3 AXIS_SPECIFIC_ELASTICITY = new Vec3(0.8, 0.2, 0.8);
    List<Vec3> ENTITY_ATTACHMENT_POINT = ImmutableList.of(new Vec3(0.0, 0.5, 0.5));
    List<Vec3> LEASHER_ATTACHMENT_POINT = ImmutableList.of(new Vec3(0.0, 0.5, 0.0));
    List<Vec3> SHARED_QUAD_ATTACHMENT_POINTS = ImmutableList.of(
        new Vec3(-0.5, 0.5, 0.5),
        new Vec3(-0.5, 0.5, -0.5),
        new Vec3(0.5, 0.5, -0.5),
        new Vec3(0.5, 0.5, 0.5)
    );

    default Vec3 raspberry$getLeashOffset(float partialTick) {
        return new Vec3(0, ((Entity)this).m_20192_() * 0.8F, 0); 
    }

    default boolean raspberry$isLeashed() {
        if (this instanceof Mob mob) return mob.m_21523_();
        return false;
    }

    default boolean canHaveALeashAttachedTo(Entity target) {
        if (this == target) {
            return false;
        } else {
            return this.leashDistanceTo(target) <= this.leashSnapDistance() && this.canBeLeashed(target instanceof Player player ? player : null);
        }
    }

    default void raspberry$onLeashRemoved() {
    }

    default void raspberry$setDelayedLeashHolderId(int id) {}

    default void raspberry$setLeashedTo(Entity entity, boolean sendAttachPacket) {
        if (this instanceof Mob mob) mob.m_21463_(entity, sendAttachPacket);
    }

    default double leashDistanceTo(Entity entity) {
        return entity.m_20191_().m_82399_().m_82554_(((Entity) this).m_20191_().m_82399_());
    }

    default boolean canBeLeashed(Player entity) {
        if (this instanceof Mob mob) return mob.m_6573_(entity);
        return true;
    }

    default void raspberry$dropLeash(boolean broadcast, boolean dropItem) {
        if (this instanceof Mob mob) {
            Entity holder = mob.m_21524_();

            mob.m_21455_(broadcast, dropItem);

            if (holder instanceof Leashable leashableHolder) {
                leashableHolder.raspberry$onLeashRemoved();
            }
        }
    }

    static <E extends Entity & Leashable> void tickLeash(E entity) {
        Entity holder = entity.raspberry$getLeashHolder();
        if (holder != null && holder.f_19853_ == entity.f_19853_) {
            double leashDistance = entity.leashDistanceTo(holder);

            if (entity instanceof TamableAnimal pet && pet.m_21827_()) {
                if (leashDistance > entity.leashSnapDistance()) {
                    entity.raspberry$dropLeash(true, true);
                }
                return;
            }

            entity.whenLeashedTo(holder);
            if (leashDistance > entity.leashSnapDistance()) {
                entity.f_19853_.m_6269_(null, holder, SoundEvents.f_12033_, SoundSource.NEUTRAL, 1.0F, 1.0F);
                entity.leashTooFarBehaviour();
            } else if (leashDistance > entity.leashElasticDistance() - (double) holder.m_20205_() - (double) entity.m_20205_() && entity.checkElasticInteractions(holder)) {
                entity.onElasticLeashPull(holder);
            } else {
                entity.closeRangeLeashBehavior(holder);
            }

            entity.m_146922_((float) (entity.m_146908_() - entity.raspberry$angularMomentum()));
            entity.setRaspberry$angularMomentum(entity.raspberry$angularMomentum() * (double) angularFriction(entity));
        }
    }

    default void onElasticLeashPull(Entity entity) {
        if (this instanceof PathfinderMob mob) {
            ((PathfinderMobAccessor) this).callOnLeashDistance(mob.m_20270_(entity));
        }
        
        Entity self = (Entity) this;
        if (self.m_20184_().f_82480_ > -0.5D) {
            self.f_19789_ = 1.0F;
        }
    }

    default double leashSnapDistance() {
        return 12.0;
    }

    default double leashElasticDistance() {
        return 6.0;
    }

    static <E extends Entity & Leashable> float angularFriction(E entity) {
        if (entity.m_20096_()) {
            return entity.f_19853_.m_8055_(((EntityAccessor) entity).callGetBlockPosBelowThatAffectsMyMovement()).m_60734_().m_49958_() * 0.91F;
        } else {
            return (entity.m_20069_() || entity.m_20077_()) ? 0.8F : 0.91F;
        }
    }

    default void whenLeashedTo(Entity entity) {
        if (this instanceof PathfinderMob mob) mob.m_21446_(entity.m_20183_(), (int) this.leashElasticDistance() - 1);
        if (entity instanceof Leashable ext) ext.notifyLeashHolder();
    }

    default void notifyLeashHolder() {}

    default void leashTooFarBehaviour() {
        if (this instanceof PathfinderMob mob) mob.f_21345_.m_25355_(Goal.Flag.MOVE);
        this.raspberry$dropLeash(true, true);
    }

    default void closeRangeLeashBehavior(Entity entity) {
        if (this instanceof PathfinderMob mob) {
            if (((PathfinderMobAccessor) this).callShouldStayCloseToLeashHolder()) {
                mob.f_21345_.m_25374_(Goal.Flag.MOVE);
                float distanceFromHolder = mob.m_20270_(entity);
                Vec3 movement = new Vec3(entity.m_20185_() - mob.m_20185_(), entity.m_20186_() - mob.m_20186_(), entity.m_20189_() - mob.m_20189_()).m_82541_().m_82490_(Math.max(distanceFromHolder - 2.0F, 0.0F));
                mob.m_21573_().m_26519_(mob.m_20185_() + movement.f_82479_, mob.m_20186_() + movement.f_82480_, mob.m_20189_() + movement.f_82481_, ((PathfinderMobAccessor) this).callFollowLeashSpeed());
            }
        }
    }

    default boolean checkElasticInteractions(Entity entity) {
        if (((Entity) this).m_6688_() instanceof Player) return false;

        boolean supportQuadLeash = entity instanceof Leashable holder && holder.supportQuadLeashAsHolder() && this.supportQuadLeash();
        List<Wrench> wrenches = computeElasticInteraction(
            (Entity & Leashable) this,
            entity,
            supportQuadLeash ? SHARED_QUAD_ATTACHMENT_POINTS : ENTITY_ATTACHMENT_POINT,
            supportQuadLeash ? SHARED_QUAD_ATTACHMENT_POINTS : LEASHER_ATTACHMENT_POINT
        );

        if (wrenches.isEmpty()) {
            return false;
        } else {
            Wrench wrench = Wrench.accumulate(wrenches).scale(supportQuadLeash ? 0.25 : 1.0);
            this.setRaspberry$angularMomentum(this.raspberry$angularMomentum() + 10.0 * wrench.torque());
            Vec3 offset = getHolderMovement(entity).m_82546_(getKnownMovement((Entity) this));
            
            Vec3 currentMotion = ((Entity) this).m_20184_();
            Vec3 deltaMovement = wrench.force().m_82559_(AXIS_SPECIFIC_ELASTICITY).m_82549_(offset.m_82490_(0.11));
            ((Entity) this).m_20256_(currentMotion.m_82549_(deltaMovement));
            
            return true;
        }
    }

    static Vec3 getHolderMovement(Entity entity) {
        return entity instanceof Mob mob && mob.m_21525_() ? Vec3.f_82478_ : getKnownMovement(entity);
    }

    static Vec3 getKnownMovement(Entity entity) {
        Entity passenger = entity.m_6688_();
        if (passenger instanceof Player player) {
            if (entity.m_6084_()) {
                return player.m_20184_();
            }
        }
        return entity.m_20184_();
    }

    static <E extends Entity & Leashable> List<Wrench> computeElasticInteraction(E entity, Entity holder, List<Vec3> attachmentPoints, List<Vec3> holderAttachmentPoints) {
        double elasticDistance = entity.leashElasticDistance();
        Vec3 entityMovement = getHolderMovement(entity);
        float entityYaw = entity.m_146908_() * (float) (Math.PI / 180.0);
        Vec3 entityDimensions = new Vec3(entity.m_20205_(), entity.m_20206_(), entity.m_20205_());
        float holderYaw = holder.m_146908_() * (float) (Math.PI / 180.0);
        Vec3 holderDimensions = new Vec3(holder.m_20205_(), holder.m_20206_(), holder.m_20205_());
        List<Wrench> wrenches = new ArrayList<>();

        for (int i = 0; i < attachmentPoints.size(); i++) {
            Vec3 entityOffset = attachmentPoints.get(i).m_82559_(entityDimensions).m_82524_(-entityYaw);
            Vec3 entityPosition = entity.m_20182_().m_82549_(entityOffset);
            Vec3 holderOffset = holderAttachmentPoints.get(i).m_82559_(holderDimensions).m_82524_(-holderYaw);
            Vec3 holderPosition = holder.m_20182_().m_82549_(holderOffset);
            computeDampenedSpringInteraction(holderPosition, entityPosition, elasticDistance, entityMovement, entityOffset).ifPresent(wrenches::add);
        }

        return wrenches;
    }

    private static Optional<Wrench> computeDampenedSpringInteraction(Vec3 holderPos, Vec3 entityPos, double threshold, Vec3 movement, Vec3 offset) {
        double distance = entityPos.m_82554_(holderPos);
        if (distance < threshold) {
            return Optional.empty();
        } else {
            Vec3 force = holderPos.m_82546_(entityPos).m_82541_().m_82490_(distance - threshold);
            double torque = Wrench.torqueFromForce(offset, force);
            boolean movingWithForce = movement.m_82526_(force) >= 0.0;
            if (movingWithForce) {
                force = force.m_82490_(0.3F);
            }
            return Optional.of(new Wrench(force, torque));
        }
    }

    default boolean supportQuadLeash() {
        Entity entity = (Entity) this;
        for (Predicate<Entity> filter : QUAD_LEASH_OFFSETS.keySet()) {
            if (filter.test(entity)) {
                return true;
            }
        }
        return false;
    }

    default boolean supportQuadLeashAsHolder() {
        return false;
    }

    static Vec3[] createQuadLeashOffsets(Entity entity, double forwardOffset, double sideOffset, double widthOffset, double heightOffset) {
        float entityWidth = entity.m_20205_();
        double forward = forwardOffset * (double) entityWidth;
        double side = sideOffset * (double) entityWidth;
        double width = widthOffset * (double) entityWidth;
        double height = heightOffset * (double) entity.m_20206_();

        return new Vec3[] {
            new Vec3(-width, height, side + forward),
            new Vec3(-width, height, -side + forward),
            new Vec3(width, height, -side + forward),
            new Vec3(width, height, side + forward)
        };
    }

    default Entity raspberry$getLeashHolder() {
        if (this instanceof Mob mob) return mob.m_21524_();
        return null;
    }

    static List<Leashable> leashableLeashedTo(Entity entity) {
        return leashableInArea(entity.f_19853_, entity.m_20182_(), l -> l.raspberry$getLeashHolder() == entity);
    }

    static List<Leashable> leashableInArea(Level level, Vec3 pos, Predicate<Leashable> filter) {
        AABB area = AABB.m_165882_(pos, 32.0, 32.0, 32.0);
        return level.m_6443_(Entity.class, area, entity -> entity instanceof Leashable leashable && filter.test(leashable))
            .stream()
            .map(Leashable.class::cast)
            .toList();
    }

    default double raspberry$angularMomentum() {
        return 0.0;
    }

    default void setRaspberry$angularMomentum(double raspberry$angularMomentum) { }

    record Wrench(Vec3 force, double torque) {
        static final Wrench ZERO = new Wrench(Vec3.f_82478_, 0.0);
        static double torqueFromForce(Vec3 position, Vec3 force) {
            return position.f_82481_ * force.f_82479_ - position.f_82479_ * force.f_82481_;
        }
        public static Wrench accumulate(List<Wrench> wrenches) {
            if (wrenches.isEmpty()) return ZERO;
            double x = 0.0;
            double y = 0.0;
            double z = 0.0;
            double torque = 0.0;
            for (Wrench wrench : wrenches) {
                Vec3 force = wrench.force;
                x += force.f_82479_;
                y += force.f_82480_;
                z += force.f_82481_;
                torque += wrench.torque;
            }
            return new Wrench(new Vec3(x, y, z), torque);
        }
        public Wrench scale(double factor) {
            return new Wrench(this.force.m_82490_(factor), this.torque * factor);
        }
    }
}