/* MIT License

Copyright (c) 2025 Martin Kadlec

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

package cc.cassian.raspberry.compat.vanillabackport.leash;

import cc.cassian.raspberry.compat.vanillabackport.leash.network.KnotConnectionSyncPacket;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.decoration.LeashFenceKnotEntity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.LeadItem;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.ShearsItem;
import net.minecraft.world.level.gameevent.GameEvent;

import java.util.List;

public class KnotInteractionHelper {

    public static void syncKnots(LeashFenceKnotEntity knot) {
        if (knot.f_19853_.f_46443_) return;

        KnotConnectionManager manager = KnotConnectionManager.getManager(knot);
        KnotConnectionSyncPacket packet = new KnotConnectionSyncPacket(knot.m_19879_(), manager.getConnectedUuids());

        if (knot.f_19853_ instanceof net.minecraft.server.level.ServerLevel serverLevel) {
            for (net.minecraft.server.level.ServerPlayer player : serverLevel.m_7726_().f_8325_.m_183262_(
                    new net.minecraft.world.level.ChunkPos(knot.m_20183_()), false)) {
                packet.sendTo(player);
            }
        }
    }

    public static class HeldEntities {
        public final List<Leashable> all;
        public final List<Leashable> mobs;
        public final List<LeashFenceKnotEntity> knots;
        public final boolean hasMobs;
        public final boolean hasKnots;

        public HeldEntities(Entity holder) {
            this.all = Leashable.leashableInArea(holder.f_19853_, holder.m_20182_(), l -> l.raspberry$getLeashHolder() == holder);
            this.mobs = all.stream().filter(l -> !(l instanceof LeashFenceKnotEntity)).toList();
            this.knots = all.stream()
                    .filter(l -> l instanceof LeashFenceKnotEntity)
                    .map(l -> (LeashFenceKnotEntity) l).toList();
            this.hasMobs = !mobs.isEmpty();
            this.hasKnots = !knots.isEmpty();
        }

        public boolean isEmpty() { return all.isEmpty(); }
    }

    public static InteractionResult handleKnotInteraction(Player player, InteractionHand hand, LeashFenceKnotEntity knot) {
        HeldEntities held = new HeldEntities(player);
        boolean isHoldingThisKnot = held.all.stream().anyMatch(l -> l == knot);

        if (!held.isEmpty() && !(held.all.size() == 1 && isHoldingThisKnot)) {
            boolean created = false;

            for (LeashFenceKnotEntity heldKnot : held.knots) {
                if (heldKnot == knot) continue;

                if (heldKnot.m_20270_(knot) > 12.0) continue;

                if (KnotConnectionManager.createConnection(heldKnot, knot)) {
                    created = true;
                    ((Leashable)heldKnot).raspberry$dropLeash(true, false);
                    syncKnots(heldKnot);
                }
            }

            for (Leashable mob : held.mobs) {
                if (((Leashable)knot).raspberry$getLeashHolder() == mob) {
                    continue;
                }

                mob.raspberry$setLeashedTo(knot, true);
                created = true;
            }

            if (created) {
                syncKnots(knot);
                knot.m_5496_(SoundEvents.f_12087_, 1.0f, 1.0f);
            }

            return InteractionResult.SUCCESS;
        }

        HeldEntities heldByKnot = new HeldEntities(knot);
        boolean isKnotLeashed = ((Leashable)knot).raspberry$isLeashed();
        boolean hasCustom = KnotConnectionManager.getManager(knot).hasConnections();
        boolean isShears = player.m_21120_(hand).m_41720_() instanceof ShearsItem;

        if (player.m_6144_() || isShears) {
            if (heldByKnot.hasMobs || hasCustom || isKnotLeashed) {
                knot.m_5496_(SoundEvents.f_12033_, 1.0f, 1.0f);

                for (Leashable mob : heldByKnot.mobs) {
                    mob.raspberry$dropLeash(true, false);
                    if (!player.m_150110_().f_35937_) {
                        if (mob instanceof Entity entity) {
                            entity.m_19998_(Items.f_42655_);
                        }
                    }
                }

                if (hasCustom) {
                    discardCustomConnections(knot, player);
                }

                if (isKnotLeashed) {
                    ((Leashable)knot).raspberry$dropLeash(true, true);
                }

                knot.m_146870_();

                return InteractionResult.SUCCESS;
            }
        }

        if (heldByKnot.hasMobs) {
            for (Leashable mob : heldByKnot.mobs) {
                mob.raspberry$setLeashedTo(player, true);
            }
            knot.m_5496_(SoundEvents.f_12033_, 1.0f, 1.0f);
            if (shouldRemoveKnot(knot)) knot.m_146870_();
            return InteractionResult.SUCCESS;

        } else if (isHoldingThisKnot) {
            ((Leashable)knot).raspberry$dropLeash(true, true);
            return InteractionResult.SUCCESS;

        } else if (isKnotLeashed) {
            ((Leashable)knot).raspberry$setLeashedTo(player, true);
            knot.m_5496_(SoundEvents.f_12033_, 1.0f, 1.0f);
            return InteractionResult.SUCCESS;

        } else if (hasLeadItem(player)) {
            ((Leashable)knot).raspberry$setLeashedTo(player, true);
            knot.m_5496_(SoundEvents.f_12087_, 1.0f, 1.0f);
            consumeLead(player);
            return InteractionResult.SUCCESS;

        } else if (hasCustom) {
            pickupCustomConnections(knot, player);
            knot.m_5496_(SoundEvents.f_12033_, 1.0f, 1.0f);
            return InteractionResult.SUCCESS;
        }

        return InteractionResult.PASS;
    }

    public static boolean hasLeadItem(Player player) {
        return player.m_21205_().m_41720_() instanceof LeadItem ||
                player.m_21206_().m_41720_() instanceof LeadItem;
    }

    public static void consumeLead(Player player) {
        if (!player.m_150110_().f_35937_) {
            if (player.m_21205_().m_41720_() instanceof LeadItem) {
                player.m_21205_().m_41774_(1);
            } else if (player.m_21206_().m_41720_() instanceof LeadItem) {
                player.m_21206_().m_41774_(1);
            }
        }
    }

    public static boolean shouldRemoveKnot(LeashFenceKnotEntity knot) {
        boolean hasVanilla = !Leashable.leashableLeashedTo(knot).isEmpty();
        boolean isLeashed = ((Leashable)knot).raspberry$getLeashHolder() != null;
        boolean hasCustom = KnotConnectionManager.getManager(knot).hasConnections();
        return !hasVanilla && !isLeashed && !hasCustom;
    }

    public static void pickupCustomConnections(LeashFenceKnotEntity knot, Player player) {
        KnotConnectionManager manager = KnotConnectionManager.getManager(knot);
        List<LeashFenceKnotEntity> connected = manager.getConnectedKnots(knot);

        for (LeashFenceKnotEntity other : connected) {
            KnotConnectionManager.removeConnection(knot, other);

            if (player.m_20280_(other) <= 100.0) {
                ((Leashable)other).raspberry$setLeashedTo(player, true);
                syncKnots(other);
            } else {
                other.m_19998_(Items.f_42655_);
                if (shouldRemoveKnot(other)) {
                    other.m_146870_();
                } else {
                    syncKnots(other);
                }
            }
            break;
        }

        if (shouldRemoveKnot(knot)) knot.m_146870_();
        else syncKnots(knot);
    }

    public static void discardCustomConnections(LeashFenceKnotEntity knot, Entity breaker) {
        KnotConnectionManager manager = KnotConnectionManager.getManager(knot);
        List<LeashFenceKnotEntity> connected = manager.getConnectedKnots(knot);

        for (LeashFenceKnotEntity other : connected) {
            KnotConnectionManager.removeConnection(knot, other);
            if (shouldRemoveKnot(other)) other.m_146870_();
            else syncKnots(other);

            knot.m_19998_(Items.f_42655_);
        }

        if (shouldRemoveKnot(knot)) knot.m_146870_();
        else syncKnots(knot);

        knot.m_146852_(GameEvent.f_157795_, breaker);
    }
}