/* The MIT License (MIT)

Copyright (c) 2025

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */

package cc.cassian.raspberry.compat.vanillabackport.leash;

import java.util.List;

import cc.cassian.raspberry.RaspberryMod;
import cc.cassian.raspberry.compat.vanillabackport.leash.network.KnotConnectionSyncPacket;
import cc.cassian.raspberry.config.ModConfig;
import net.minecraft.core.BlockPos;
import net.minecraft.sounds.SoundSource;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.tags.BlockTags;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.decoration.LeashFenceKnotEntity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.gameevent.GameEvent;
import net.minecraft.world.phys.AABB;
import net.minecraftforge.common.Tags;
import net.minecraftforge.event.entity.player.PlayerEvent;
import net.minecraftforge.event.entity.player.PlayerInteractEvent;
import net.minecraftforge.event.level.BlockEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.common.Mod;
import org.jetbrains.annotations.Nullable;

@Mod.EventBusSubscriber(modid = RaspberryMod.MOD_ID)
public class LeashEvents {

    @SubscribeEvent
    public static void onEntityInteract(PlayerInteractEvent.EntityInteract event) {
        if (!ModConfig.get().backportLeash) {
            return;
        }

        Player player = event.getEntity();
        Entity target = event.getTarget();
        ItemStack stack = event.getItemStack();

        if (!target.f_19853_.f_46443_ && player.m_36341_() && target instanceof Leashable leashable && leashable.canBeLeashed(player) && target.m_6084_()) {
            if (!(target instanceof LivingEntity living && living.m_6162_())) {
                List<Leashable> nearbyEntities = Leashable.leashableInArea(target.f_19853_, target.m_20182_(), l -> l.raspberry$getLeashHolder() == player);

                if (!nearbyEntities.isEmpty()) {
                    boolean attachedAny = false;

                    for (Leashable sourceEntity : nearbyEntities) {
                        if (sourceEntity instanceof LeashFenceKnotEntity && target instanceof LeashFenceKnotEntity) continue;
                        if (sourceEntity instanceof LeashFenceKnotEntity) continue;
                        if (target instanceof LeashFenceKnotEntity) continue;
                        if (leashable.raspberry$getLeashHolder() == sourceEntity) continue;

                        if (sourceEntity.canHaveALeashAttachedTo(target)) {
                            sourceEntity.raspberry$setLeashedTo(target, true);
                            attachedAny = true;
                        }
                    }

                    if (attachedAny) {
                        target.f_19853_.m_220407_(GameEvent.f_223708_, target.m_20183_(), GameEvent.Context.m_223717_(player));
                        target.m_5496_(SoundEvents.f_12087_, 1.0F, 1.0F);

                        event.setCancellationResult(InteractionResult.SUCCESS);
                        event.setCanceled(true);
                        return;
                    }
                }
            }
        }

        if (stack.m_204117_(Tags.Items.SHEARS) && shearOffAllLeashConnections(target, player)) {
            stack.m_41622_(1, player, p -> p.m_21190_(event.getHand()));
            event.setCancellationResult(InteractionResult.SUCCESS);
            event.setCanceled(true);
            return;
        }

        if (target.m_6084_() && target instanceof Leashable leashable) {
            if (leashable.raspberry$getLeashHolder() == player) {
                if (!target.f_19853_.f_46443_) {
                    leashable.raspberry$dropLeash(true, !player.m_7500_());
                    target.f_19853_.m_214171_(GameEvent.f_223708_, target.m_20182_(), GameEvent.Context.m_223717_(player));
                    target.m_5496_(SoundEvents.f_12033_, 1.0F, 1.0F);
                }
                event.setCancellationResult(InteractionResult.SUCCESS);
                event.setCanceled(true);
                return;
            }

            if (stack.m_150930_(Items.f_42655_) && !(leashable.raspberry$getLeashHolder() instanceof Player)) {
                if (target instanceof LeashFenceKnotEntity) {
                    return;
                }

                if (!target.f_19853_.f_46443_ && leashable.canHaveALeashAttachedTo(player)) {
                    if (leashable.raspberry$isLeashed()) {
                        leashable.raspberry$dropLeash(true, true);
                    }

                    leashable.raspberry$setLeashedTo(player, true);
                    target.m_5496_(SoundEvents.f_12087_, 1.0F, 1.0F);

                    if (!player.m_7500_()) stack.m_41774_(1);
                }
                event.setCancellationResult(InteractionResult.SUCCESS);
                event.setCanceled(true);
            }
        }
    }

    @SubscribeEvent
    public static void onStartTracking(PlayerEvent.StartTracking event) {
        if (!ModConfig.get().backportLeash) return;

        Entity target = event.getTarget();
        if (target instanceof LeashFenceKnotEntity knot && target instanceof KnotConnectionAccess access) {
            KnotConnectionManager manager = access.raspberry$getConnectionManager();

            if (manager.hasConnections()) {
                KnotConnectionSyncPacket packet = new KnotConnectionSyncPacket(knot.m_19879_(), manager.getConnectedUuids());
                packet.sendTo((net.minecraft.server.level.ServerPlayer) event.getEntity());
            }
        }
    }

    @SubscribeEvent
    public static void onBlockBreak(BlockEvent.BreakEvent event) {
        if (!ModConfig.get().backportLeash) return;

        BlockState state = event.getState();
        if (!state.m_204336_(BlockTags.f_13039_)) return;

        Level level = (Level) event.getLevel();
        if (level.f_46443_) return;

        BlockPos pos = event.getPos();

        List<LeashFenceKnotEntity> knots = level.m_6443_(
                LeashFenceKnotEntity.class,
                new AABB(pos),
                knot -> knot.m_31748_().equals(pos)
        );

        for (LeashFenceKnotEntity knot : knots) {
            if (knot instanceof KnotConnectionAccess) {
                KnotInteractionHelper.discardCustomConnections(knot, event.getPlayer());
            }
            knot.m_146870_();
        }
    }

    public static boolean shearOffAllLeashConnections(Entity entity, Player player) {
        boolean sheared = dropAllLeashConnections(entity, player);
        if (sheared && entity.f_19853_ instanceof ServerLevel server) {
            server.m_5594_(null, entity.m_20183_(), SoundEvents.f_12344_, SoundSource.PLAYERS, 1.0f, 1.0f);
        }
        return sheared;
    }

    public static boolean dropAllLeashConnections(Entity entity, @Nullable Player player) {
        List<Leashable> leashed = Leashable.leashableLeashedTo(entity);
        boolean dropConnections = !leashed.isEmpty();

        if (entity instanceof Leashable leashable && leashable.raspberry$isLeashed()) {
            leashable.raspberry$dropLeash(true, true);
            dropConnections = true;
        }

        for (Leashable leashable : leashed) {
            leashable.raspberry$dropLeash(true, true);
        }

        if (entity instanceof LeashFenceKnotEntity knot && entity instanceof KnotConnectionAccess) {
            KnotInteractionHelper.discardCustomConnections(knot, player);
            dropConnections = true;
        }

        if (dropConnections) {
            entity.m_146852_(GameEvent.f_157781_, player);
            if (entity instanceof LeashFenceKnotEntity) {
                entity.m_146870_();
            }
            return true;
        } else {
            return false;
        }
    }
}