/* The MIT License (MIT)

Copyright (c) 2025 

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */


package cc.cassian.raspberry.compat.vanillabackport.leash;

import cc.cassian.raspberry.mixin.minecraft.EntityRendererAccessor;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.blaze3d.vertex.VertexConsumer;
import com.mojang.math.Matrix4f;
import net.minecraft.client.renderer.LightTexture;
import net.minecraft.client.renderer.MultiBufferSource;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.client.renderer.culling.Frustum;
import net.minecraft.client.renderer.entity.EntityRenderDispatcher;
import net.minecraft.core.BlockPos;
import net.minecraft.util.Mth;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.decoration.LeashFenceKnotEntity;
import net.minecraft.world.level.LightLayer;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

public class LeashRenderer<T extends Entity> {
    private static final float LEASH_THICKNESS = 0.075F;
    private final EntityRenderDispatcher dispatcher;
    @Nullable private List<LeashState> leashStates;

    public LeashRenderer(EntityRenderDispatcher dispatcher) {
        this.dispatcher = dispatcher;
    }

    public boolean shouldRender(T entity, Frustum camera, boolean isVisible) {
        if (!isVisible) {
            AABB entityBox = entity.m_6921_().m_82400_(0.5);
            if (Double.isNaN(entityBox.f_82288_) || entityBox.m_82309_() == 0.0) {
                 entityBox = new AABB(entity.m_20185_() - 2.0, entity.m_20186_() - 2.0, entity.m_20189_() - 2.0, 
                                     entity.m_20185_() + 2.0, entity.m_20186_() + 2.0, entity.m_20189_() + 2.0);
            }

            if (camera.m_113029_(entityBox)) {
                return true;
            } 
            
            if (entity instanceof Leashable leashable) {
                Entity holder = leashable.raspberry$getLeashHolder();
                if (holder != null) {
                    AABB holderBox = holder.m_6921_();
                    if (camera.m_113029_(holderBox) || camera.m_113029_(entityBox.m_82367_(holderBox))) {
                        return true;
                    }
                }
            }

            if (entity instanceof KnotConnectionAccess access) {
                List<LeashFenceKnotEntity> connectedKnots = access.raspberry$getConnectionManager()
                    .getConnectedKnots((LeashFenceKnotEntity) entity);
                for (LeashFenceKnotEntity knot : connectedKnots) {
                    AABB knotBox = knot.m_6921_();
                    if (camera.m_113029_(knotBox) || camera.m_113029_(entityBox.m_82367_(knotBox))) {
                        return true;
                    }
                }
            }
        }
        return isVisible;
    }

    public void render(T entity, float partialTick, PoseStack poseStack, MultiBufferSource buffer) {
        this.setupLeashRendering(entity, partialTick);

        if (this.leashStates != null) {
            for (LeashState state : this.leashStates) {
                this.renderLeash(poseStack, buffer, state);
            }
        }
    }

    private void renderLeash(PoseStack stack, MultiBufferSource buffer, LeashState state) {
        float deltaX = (float) (state.end.f_82479_ - state.start.f_82479_);
        float deltaY = (float) (state.end.f_82480_ - state.start.f_82480_);
        float deltaZ = (float) (state.end.f_82481_ - state.start.f_82481_);

        if (state.isKnotToKnot && deltaY == 0.0F) {
            float horizontalDistance = (float)Math.sqrt(deltaX * deltaX + deltaZ * deltaZ);
            float maxDroop = 0.15F;
            float distanceScale = 0.05F;
            state.droopAmount = maxDroop / (1.0F + horizontalDistance * distanceScale);
        }

        float scaleFactor = (1.0F / Mth.m_14116_(deltaX * deltaX + deltaZ * deltaZ)) * 0.05F / 2.0F;

        float offsetZ = deltaZ * scaleFactor;
        float offsetX = deltaX * scaleFactor;

        stack.m_85836_();
        stack.m_85837_(state.offset.f_82479_, state.offset.f_82480_, state.offset.f_82481_);

        VertexConsumer vertices = buffer.m_6299_(RenderType.m_110475_());
        Matrix4f matrices = stack.m_85850_().m_85861_();

        for (int segment = 0; segment <= 24; segment++) {
            addVertexPair(vertices, matrices, deltaX, deltaY, deltaZ, LEASH_THICKNESS,
                         offsetZ, offsetX, segment, false, state);
        }

        for (int segment = 24; segment >= 0; segment--) {
            addVertexPair(vertices, matrices, deltaX, deltaY, deltaZ, 0.0F,
                         offsetZ, offsetX, segment, true, state);
        }

        stack.m_85849_();
    }

    private static void addVertexPair(
        VertexConsumer vertices, Matrix4f matrices,
        float deltaX, float deltaY, float deltaZ,
        float thickness2,
        float offsetZ, float offsetX,
        int segment, boolean isInnerFace, LeashState state
    ) {
        float progress = (float) segment / 24.0f;

        int blockLight = (int) Mth.m_14179_(progress, state.startBlockLight, state.endBlockLight);
        int skyLight = (int) Mth.m_14179_(progress, state.startSkyLight, state.endSkyLight);
        int packedLight = LightTexture.m_109885_(blockLight, skyLight);

        boolean useSecondary = segment % 2 == (isInnerFace ? 1 : 0);

        final float A_R = 112f / 255f;
        final float A_G = 75f / 255f;
        final float A_B = 42f  / 255f;

        final float B_R = 79f / 255f;
        final float B_G = 48f / 255f;
        final float B_B = 26f / 255f;

        float red   = useSecondary ? B_R : A_R;
        float green = useSecondary ? B_G : A_G;
        float blue  = useSecondary ? B_B : A_B;

        float posX = deltaX * progress;
        float posZ = deltaZ * progress;
        float posY;
        
        if (state.slack) {
            posY = (deltaY > 0.0f
                ? deltaY * progress * progress
                : deltaY - deltaY * (1.0f - progress) * (1.0f - progress));
        } else {
            posY = deltaY * progress;
        }

        if (state.isKnotToKnot && state.droopAmount > 0) {
            float droopFactor = progress * (1.0f - progress) * 4.0f;
            posY -= state.droopAmount * droopFactor;
        }

        vertices.m_85982_(matrices, posX - offsetZ, posY + thickness2, posZ + offsetX)
                .m_85950_(red, green, blue, 1.0f).m_85969_(packedLight).m_5752_();
        vertices.m_85982_(matrices, posX + offsetZ, posY + LeashRenderer.LEASH_THICKNESS - thickness2, posZ - offsetX)
                .m_85950_(red, green, blue, 1.0f).m_85969_(packedLight).m_5752_();
    }

    private void setupLeashRendering(T entity, float partialTicks) {
        List<LeashState> collector = new ArrayList<>();

        if (entity instanceof Leashable leashable) {
            Entity leashHolder = leashable.raspberry$getLeashHolder();
            if (leashHolder != null) {
                addLeashStates(entity, leashable, leashHolder, partialTicks, collector, false);
            }
        }

        if (entity instanceof KnotConnectionAccess access && entity instanceof Leashable leashableSelf) {
            List<LeashFenceKnotEntity> connectedKnots = access.raspberry$getConnectionManager()
                .getConnectedKnots((LeashFenceKnotEntity) entity);
            for (LeashFenceKnotEntity targetKnot : connectedKnots) {
                if (entity.m_20148_().compareTo(targetKnot.m_20148_()) < 0) {
                    addLeashStates(entity, leashableSelf, targetKnot, partialTicks, collector, true);
                }
            }
        }

        this.leashStates = collector.isEmpty() ? null : collector;
    }

    private void addLeashStates(T entity, Leashable leashable, Entity target, float partialTicks, 
                                List<LeashState> collector, boolean isKnotToKnot) {
        float entityRotation = Mth.m_14179_(partialTicks, entity.f_19859_, entity.m_146908_()) * ((float) Math.PI / 180);
        Vec3 leashOffset = leashable.raspberry$getLeashOffset(partialTicks);

        BlockPos entityPos = new BlockPos(entity.m_20299_(partialTicks));
        BlockPos holderPos = new BlockPos(target.m_20299_(partialTicks));
        int entityBlockLight = this.getBlockLightLevel(entity, entityPos);
        int holderBlockLight = this.getBlockLightLevel(target, holderPos);
        int entitySkyLight = entity.f_19853_.m_45517_(LightLayer.SKY, entityPos);
        int holderSkyLight = entity.f_19853_.m_45517_(LightLayer.SKY, holderPos);

        Vec3 rotatedOffset = leashOffset.m_82524_(-entityRotation);
        LeashState leashState = new LeashState();
        leashState.offset = rotatedOffset;
        leashState.start = entity.m_20318_(partialTicks).m_82549_(rotatedOffset);
        
        if (isKnotToKnot && target instanceof Leashable targetLeashable) {
             float targetRotation = Mth.m_14179_(partialTicks, target.f_19859_, target.m_146908_()) * ((float) Math.PI / 180);
             Vec3 targetOffset = targetLeashable.raspberry$getLeashOffset(partialTicks).m_82524_(-targetRotation);
             leashState.end = target.m_20318_(partialTicks).m_82549_(targetOffset);
        } else {
             leashState.end = target.m_7398_(partialTicks);
        }
        
        leashState.startBlockLight = entityBlockLight;
        leashState.endBlockLight = holderBlockLight;
        leashState.startSkyLight = entitySkyLight;
        leashState.endSkyLight = holderSkyLight;
        leashState.slack = isKnotToKnot;
        leashState.isKnotToKnot = isKnotToKnot;
        collector.add(leashState);
    }

    private int getBlockLightLevel(Entity entity, BlockPos pos) {
        return ((EntityRendererAccessor) this.dispatcher.m_114382_(entity)).callGetBlockLightLevel(entity, pos);
    }
}