/* The MIT License (MIT)

Copyright (c) 2025 

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */

package cc.cassian.raspberry.mixin.minecraft;

import cc.cassian.raspberry.compat.vanillabackport.leash.InterpolationHandler;
import cc.cassian.raspberry.compat.vanillabackport.leash.Leashable;
import cc.cassian.raspberry.config.ModConfig;
import net.minecraft.core.BlockPos;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.protocol.game.ClientboundSetEntityLinkPacket;
import net.minecraft.network.syncher.EntityDataAccessor;
import net.minecraft.network.syncher.EntityDataSerializers;
import net.minecraft.network.syncher.SynchedEntityData;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.decoration.HangingEntity;
import net.minecraft.world.entity.vehicle.Boat;
import net.minecraft.world.item.Items;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

import java.util.OptionalInt;
import java.util.UUID;

@Mixin(Boat.class)
public abstract class BoatMixin extends Entity implements Leashable {

    @Unique
    private static final EntityDataAccessor<OptionalInt> DATA_ID_LEASH_HOLDER_ID = SynchedEntityData.m_135353_(BoatMixin.class, EntityDataSerializers.f_135044_);

    @Unique
    private int raspberry$delayedLeashHolderId;

    @Unique
    @Nullable
    private Entity raspberry$leashHolder;

    @Unique
    private final InterpolationHandler raspberry$interpolation = new InterpolationHandler(this, 3);

    @Unique
    @Nullable
    private CompoundTag raspberry$leashInfoTag;

    public BoatMixin(EntityType<?> entityType, Level level) {
        super(entityType, level);
    }

    @Inject(method = "defineSynchedData", at = @At("TAIL"))
    private void raspberry$defineLeashData(CallbackInfo ci) {
        this.f_19804_.m_135372_(DATA_ID_LEASH_HOLDER_ID, OptionalInt.empty());
    }

    @Unique
    private void raspberry$restoreLeashFromSave() {
        if (this.raspberry$leashInfoTag != null && this.f_19853_ instanceof ServerLevel serverLevel) {
            if (this.raspberry$leashInfoTag.m_128403_("UUID")) {
                UUID uuid = this.raspberry$leashInfoTag.m_128342_("UUID");
                Entity entity = serverLevel.m_8791_(uuid);
                if (entity != null) {
                    this.raspberry$setLeashedTo(entity, true);
                    return;
                }
            } else if (this.raspberry$leashInfoTag.m_128425_("X", 99) && this.raspberry$leashInfoTag.m_128425_("Y", 99) && this.raspberry$leashInfoTag.m_128425_("Z", 99)) {
                BlockPos pos = net.minecraft.nbt.NbtUtils.m_129239_(this.raspberry$leashInfoTag);
                this.raspberry$setLeashedTo(net.minecraft.world.entity.decoration.LeashFenceKnotEntity.m_31844_(this.f_19853_, pos), true);
                return;
            }

            if (this.f_19797_ > 100) {
                this.m_19998_(Items.f_42655_);
                this.raspberry$leashInfoTag = null;
            }
        }
    }

    @Inject(method = "tick", at = @At("TAIL"))
    private void raspberry$tickLeash(CallbackInfo ci) {
        if (!this.f_19853_.f_46443_ && ModConfig.get().backportLeash) {
            if (this.raspberry$leashInfoTag != null) {
                this.raspberry$restoreLeashFromSave();
            }

            if (this.raspberry$leashHolder != null) {
                if (!this.m_6084_() || !this.raspberry$leashHolder.m_6084_()) {
                    this.raspberry$dropLeash(true, true);
                }
            }

            Leashable.tickLeash(this);
        }
    }

    @Inject(method = "addAdditionalSaveData", at = @At("TAIL"))
    private void raspberry$saveLeash(CompoundTag compound, CallbackInfo ci) {
        if (!ModConfig.get().backportLeash) return;

        if (this.raspberry$leashHolder != null) {
            CompoundTag tag = new CompoundTag();
            if (this.raspberry$leashHolder instanceof LivingEntity) {
                tag.m_128362_("UUID", this.raspberry$leashHolder.m_20148_());
            } else if (this.raspberry$leashHolder instanceof HangingEntity) {
                BlockPos pos = ((HangingEntity) this.raspberry$leashHolder).m_31748_();
                tag.m_128405_("X", pos.m_123341_());
                tag.m_128405_("Y", pos.m_123342_());
                tag.m_128405_("Z", pos.m_123343_());
            }
            compound.m_128365_("Leash", tag);
        } else if (this.raspberry$leashInfoTag != null) {
            compound.m_128365_("Leash", this.raspberry$leashInfoTag.m_6426_());
        }
    }

    @Inject(method = "readAdditionalSaveData", at = @At("TAIL"))
    private void raspberry$readLeash(CompoundTag compound, CallbackInfo ci) {
        if (!ModConfig.get().backportLeash) return;

        if (compound.m_128425_("Leash", 10)) {
            this.raspberry$leashInfoTag = compound.m_128469_("Leash");
        }
    }

    @Override
    public boolean raspberry$isLeashed() {
        return this.f_19804_.m_135370_(DATA_ID_LEASH_HOLDER_ID).isPresent();
    }

    @Nullable
    @Override
    public Entity raspberry$getLeashHolder() {
        if (this.raspberry$leashHolder == null && this.f_19804_.m_135370_(DATA_ID_LEASH_HOLDER_ID).isPresent()) {
            if (this.f_19853_.f_46443_) {
                this.raspberry$leashHolder = this.f_19853_.m_6815_(this.f_19804_.m_135370_(DATA_ID_LEASH_HOLDER_ID).getAsInt());
            }
        }
        return this.raspberry$leashHolder;
    }

    @Override
    public void raspberry$setLeashedTo(Entity entity, boolean sendPacket) {
        this.raspberry$leashHolder = entity;
        this.raspberry$leashInfoTag = null;
        this.raspberry$delayedLeashHolderId = 0;
        this.f_19804_.m_135381_(DATA_ID_LEASH_HOLDER_ID, OptionalInt.of(entity.m_19879_()));

        if (sendPacket && !this.f_19853_.f_46443_ && this.f_19853_ instanceof ServerLevel serverLevel) {
            serverLevel.m_7726_().m_8445_(this, new ClientboundSetEntityLinkPacket(this, entity));
        }
    }

    @Override
    public void raspberry$dropLeash(boolean broadcast, boolean dropItem) {
        boolean wasLeashed = this.raspberry$leashHolder != null;

        this.raspberry$leashHolder = null;
        this.raspberry$leashInfoTag = null;
        this.raspberry$delayedLeashHolderId = 0;
        this.f_19804_.m_135381_(DATA_ID_LEASH_HOLDER_ID, OptionalInt.empty());

        if (!this.f_19853_.f_46443_ && wasLeashed) {
            if (dropItem) {
                this.m_19998_(Items.f_42655_);
            }

            if (broadcast && this.f_19853_ instanceof ServerLevel serverLevel) {
                serverLevel.m_7726_().m_8445_(this, new ClientboundSetEntityLinkPacket(this, null));
            }
        }
    }

    @Override
    public void raspberry$setDelayedLeashHolderId(int id) {
        this.raspberry$delayedLeashHolderId = id;

        this.raspberry$leashHolder = null;
        this.f_19804_.m_135381_(DATA_ID_LEASH_HOLDER_ID, OptionalInt.empty());

        if (id != 0) {
            Entity entity = this.f_19853_.m_6815_(id);
            if (entity != null) {
                this.raspberry$setLeashedTo(entity, false);
                this.raspberry$delayedLeashHolderId = 0;
            }
        }
    }

    @Inject(method = "lerpTo", at = @At("HEAD"), cancellable = true)
    private void raspberry$lerpTo(double x, double y, double z, float yRot, float xRot, int lerpSteps, boolean teleport, CallbackInfo ci) {
        if (ModConfig.get().backportLeash) {
            ci.cancel();
            this.raspberry$interpolation.interpolateTo(new Vec3(x, y, z), yRot, xRot);
        }
    }

    @Inject(method = "tickLerp", at = @At("HEAD"), cancellable = true)
    private void raspberry$tickLerp(CallbackInfo ci) {
        if (ModConfig.get().backportLeash) {
            ci.cancel();
            if (this.m_6109_()) {
                this.raspberry$interpolation.cancel();
                this.m_217006_(this.m_20185_(), this.m_20186_(), this.m_20189_());
            }

            this.raspberry$interpolation.interpolate();
        }
    }

    @Override
    public @NotNull Vec3 m_7939_() {
        return new Vec3(0.0, 0.88F * this.m_20206_(), 0.64F * this.m_20205_());
    }

    @Override
    public void m_6089_() {
        super.m_6089_();
        if (ModConfig.get().backportLeash) {
            this.raspberry$dropLeash(true, false);
        }
    }

    @Inject(method = "tick", at = @At("HEAD"))
    private void raspberry$resolveLeash(CallbackInfo ci) {
        if (this.f_19853_.f_46443_ && this.raspberry$delayedLeashHolderId != 0 && this.raspberry$getLeashHolder() == null) {
            Entity entity = this.f_19853_.m_6815_(this.raspberry$delayedLeashHolderId);
            if (entity != null) {
                this.raspberry$setLeashedTo(entity, false);
                this.raspberry$delayedLeashHolderId = 0;
            }
        }
    }
}