package com.github.wyzzard225.settingsmanager.client;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import dev.isxander.yacl3.api.*;
import dev.isxander.yacl3.api.controller.*;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.awt.*;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.*;
import net.minecraft.class_2561;
import net.minecraft.class_437;

@SuppressWarnings({"UnusedReturnValue", "StringConcatenationArgumentToLogCall", "unused"})
public class Settings {
	 public String modName;
	 public String modId;
	 private transient Logger modConsole = LoggerFactory.getLogger("SettingsManagerLib ERROR: SETTINGS NOT SETUP");

	 //Map<String(id), Object(cur setting)>
	 private HashMap<String, Object> settings = new HashMap<>();

	 //Map<String(page.group.id), Object(info)>
	 //Object[] {id,name,defaultValue,descr,*min,*max};
	 private transient final HashMap<String, HashMap<String, HashMap<String, Object[]>>> configInfo = new HashMap<>();

	 public transient HashMap<String, Object> defaultSettings = this.resetSettings();
	 private transient final HashMap<String, String> configLocs = new HashMap<>();
	 private transient final HashMap<String, List<String>> groupOrders = new HashMap<>();

	 public Settings() {
	 }

	 public HashMap<String, Object> resetSettings() {
			HashMap<String, Object> defaults = new HashMap<>();
			for (String id : settings.keySet()) {
				 String[] idPath = Objects.requireNonNull(configLocs).get(id).split("\\.");
				 defaults.put(id, configInfo.get(idPath[0]).get(idPath[1]).get(idPath[2])[2]);
			}
			return defaults;
	 }

	 public Settings setup(String name, String id) {
			modConsole = LoggerFactory.getLogger(id + " | SettingsManagerLib");
			modName = name;
			modId = id;
			return this;
	 }

	 //call this when you are done creating config options, removes old settings that are no longer in the config
	 public Settings cleanup() {
			String[] refSettings = settings.keySet().toArray(new String[0]);
			for (String page : configInfo.keySet()) {
				 for (String group : configInfo.get(page).keySet()) {
						for (String id : configInfo.get(page).get(group).keySet()) {
							 for (int i = 0; i < refSettings.length; i++) {
									if (id.equals(refSettings[i])) {
										 refSettings[i] = null;
									}
							 }
						}
				 }
			}
			for (String id : refSettings) {
				 if (id != null) {
						settings.remove(id);
				 }
			}
			return this;
	 }

	 public Settings orderGroups(String page, String[] groupOrder) {
			if (!groupOrders.containsKey(page)) {
				 modConsole.error("Error while ordering groups:Page " + page + " does not exist!");
				 return this;
			}
			List<String> finalOrder = new java.util.ArrayList<>(Arrays.stream(groupOrder).toList());
			for (String group : groupOrders.get(page)) {
				 if (!finalOrder.contains(group)) {
						finalOrder.add(group);
				 }
			}
			groupOrders.put(page, finalOrder);
			return this;
	 }

	 private Settings registerValue(String idPath, String name, Object defaultValue, @Nullable String descr, Object min, Object max, Object step, String additionalData) {
			if (descr == null) {
				 descr = "";
			}
			String[] newId = idPath.split("\\.");

			//if the group doesn't exist yet, create it
			if (!configInfo.containsKey(newId[0])) {
				 configInfo.put(newId[0], new HashMap<>());
			}
			if (!configInfo.get(newId[0]).containsKey(newId[1])) {
				 configInfo.get(newId[0]).put(newId[1], new HashMap<>());
			}

			//put the new config information at the location
			configInfo.get(newId[0]).get(newId[1]).put(newId[2], new Object[]{newId[2], name, defaultValue, descr, additionalData, min, max, step});
			if (!settings.containsKey(newId[2])) {
				 settings.put(newId[2], defaultValue);
			}
			configLocs.put(newId[2], idPath);
			if (!groupOrders.containsKey(newId[0])) {groupOrders.put(newId[0], new ArrayList<>());}
			if (!groupOrders.get(newId[0]).contains(newId[1])) {groupOrders.get(newId[0]).add(newId[1]);}
			return this;
	 }

	 private Settings registerValue(String idPath, String name, Object defaultValue, @Nullable String descr, String additionalData) {
			if (descr == null) {
				 descr = "";
			}
			String[] newId = idPath.split("\\.");

			//if the group doesn't exist yet, create it
			if (!configInfo.containsKey(newId[0])) {
				 configInfo.put(newId[0], new HashMap<>());
			}
			if (!configInfo.get(newId[0]).containsKey(newId[1])) {
				 configInfo.get(newId[0]).put(newId[1], new HashMap<>());
			}

			//put the new config information at the location
			configInfo.get(newId[0]).get(newId[1]).put(newId[2], new Object[]{newId[2], name, defaultValue, descr, additionalData});
			if (!settings.containsKey(newId[2])) {
				 settings.put(newId[2], defaultValue);
			}
			configLocs.put(newId[2], idPath);
			if (!groupOrders.containsKey(newId[0])) {groupOrders.put(newId[0], new ArrayList<>());}
			if (!groupOrders.get(newId[0]).contains(newId[1])) {groupOrders.get(newId[0]).add(newId[1]);}
			return this;
	 }

	 @SuppressWarnings("SameParameterValue")
	 private <E extends Enum<E>> Settings registerValue(String idPath, String name, E defaultValue, @Nullable String descr, String additionalData, Class<E> enumClass) {
			if (descr == null) {
				 descr = "";
			}
			String[] newId = idPath.split("\\.");

			//if the group doesn't exist yet, create it
			if (!configInfo.containsKey(newId[0])) {
				 configInfo.put(newId[0], new HashMap<>());
			}
			if (!configInfo.get(newId[0]).containsKey(newId[1])) {
				 configInfo.get(newId[0]).put(newId[1], new HashMap<>());
			}

			//put the new config information at the location
			configInfo.get(newId[0]).get(newId[1]).put(newId[2], new Object[]{newId[2], name, defaultValue, descr, additionalData,enumClass});
			if (!settings.containsKey(newId[2])) {settings.put(newId[2], defaultValue);}
			configLocs.put(newId[2], idPath);
			if (!groupOrders.containsKey(newId[0])) {groupOrders.put(newId[0], new ArrayList<>());}
			if (!groupOrders.get(newId[0]).contains(newId[1])) {groupOrders.get(newId[0]).add(newId[1]);}
			return this;
	 }

	 public Settings add(String id, String page, @Nullable String group, float defaultValue, String name, @Nullable String description, float min, float max, float step) { //float
			registerValue(page + "." + group + "." + id, name, defaultValue, description, min, max, step, "");
			return this;
	 }

	 public Settings add(String id, String page, @Nullable String group, int defaultValue, String name, @Nullable String description, int min, int max, int step) {//int
			registerValue(page + "." + group + "." + id, name, defaultValue, description, min, max, step, "");
			return this;
	 }

	 public Settings addSlider(String id, String page, @Nullable String group, float defaultValue, String name, @Nullable String description, float min, float max, float step) { //float slider
			registerValue(page + "." + group + "." + id, name, defaultValue, description, min, max, step, "isSlider");
			return this;
	 }

	 public Settings addSlider(String id, String page, @Nullable String group, int defaultValue, String name, @Nullable String description, int min, int max, int step) {//int slider
			registerValue(page + "." + group + "." + id, name, defaultValue, description, min, max, step, "isSlider");
			return this;
	 }

	 public enum BooleanFormat {
			YESNO, TRUEFALSE, ONOFF, TICKBOX
	 }

	 public Settings add(String id, String page, @Nullable String group, boolean defaultValue, String name, @Nullable String description) {//boolean
			registerValue(page + "." + group + "." + id, name, defaultValue, description, "");
			return this;
	 }

	 public Settings add(String id, String page, @Nullable String group, boolean defaultValue, String name, @Nullable String description, BooleanFormat visualFormat) {//formatted boolean
			id = page + "." + group + "." + id;
			if (visualFormat == BooleanFormat.TICKBOX) {
				 registerValue(id, name, defaultValue, description, "isTickbox");
			} else if (visualFormat == BooleanFormat.YESNO) {
				 registerValue(id, name, defaultValue, description, "isYesNo");
			} else if (visualFormat == BooleanFormat.ONOFF) {
				 registerValue(id, name, defaultValue, description, "isOnOff");
			} else {
				 registerValue(id, name, defaultValue, description, "");
			}
			return this;
	 }

	 public Settings addColor(String id, String page, @Nullable String group, int defaultValue, String name, @Nullable String description) {//color
			registerValue(page + "." + group + "." + id, name, defaultValue, description, "isColor");
			return this;
	 }

	 public Settings add(String id, String page, @Nullable String group, String defaultValue, String name, @Nullable String description) {//string
			registerValue(page + "." + group + "." + id, name, defaultValue, description, "");
			return this;
	 }

	 public <E extends Enum<E>> Settings add(String id, String page, @Nullable String group, E defaultValue, String name, @Nullable String description, Class<E> enumClass, EnumDisplayType displayType) {//enum
			registerValue(page + "." + group + "." + id, name, defaultValue, description, "isEnum."+displayType.toString(), enumClass);
			return this;
	 }

	 @Nullable
	 public Boolean get(String id, boolean none) {//get boolean
			Object val = settings.get(id);
			if (val instanceof Boolean) {
				 return (boolean) val;
			}
			modConsole.error("Error getting boolean value: " + id + ". Invalid value type: "+val.getClass());
			return null;
	 }

	 @Nullable
	 public Float get(String id, float none) {//get float
			Object val = settings.get(id);
			if (val instanceof Float) {
				 return (float) val;
			} else if (val instanceof Double f) {
				 return f.floatValue();
			}
			modConsole.error("Error getting float value: " + id + ". Invalid value type: "+val.getClass());
			return null;
	 }

	 @Nullable
	 public Integer get(String id, int none) {//get int
			Object val = settings.get(id);
			if (val instanceof Integer) {
				 return (int) val;
			} else if (val instanceof Float f) {
				 return f.intValue();
			} else if (val instanceof Double f) {
				 return f.intValue();
			}
			modConsole.error("Error getting int value: " + id + ". Invalid value type: "+val.getClass());
			return null;
	 }

	 @Nullable
	 public String get(String id, String none) {//get string
			Object val = settings.get(id);
			if (val instanceof String) {
				 return (String) val;
			}
			modConsole.error("Error getting string value: " + id + ". Invalid value type: "+val.getClass());
			return null;
	 }

	 @Nullable
	 public <E extends Enum<E>> E get(String id, Class<E> enumClass) {//get enum
			Object val = settings.get(id);
			try {
			return Enum.valueOf(enumClass, val.toString());
			} catch (IllegalArgumentException e) {
				 modConsole.error("Error getting enum value: " + id + ". Invalid value type: "+val.getClass()+". Value: "+val+". Target enum class: "+enumClass);
				 return null;
			}
	 }

	 public Settings loadFromFile() {
			File CONFIG_FILE = new File("config/" + modId + ".json");
			if (CONFIG_FILE.exists()) {
				 try (FileReader reader = new FileReader(CONFIG_FILE)) {
						Gson gson = new GsonBuilder().create();
						Settings newSettings = gson.fromJson(reader, Settings.class);
						if (newSettings != null) {
							 for (String key : newSettings.settings.keySet()) {
									this.settings.put(key, newSettings.settings.get(key));
							 }
						}
				 } catch (IOException e) {
						modConsole.error("Failed to load config! Error source: " + Arrays.toString(e.getStackTrace()));
				 }
			}
			return this;
	 }

	 public Settings saveToFile() {
			File CONFIG_FILE = new File("config/" + modId + ".json");
			try (FileWriter writer = new FileWriter(CONFIG_FILE)) {
				 Gson gson = new GsonBuilder().create();
				 Settings settings = new Settings().importSettings(this.settings);
				 gson.toJson(settings, writer);
				 modConsole.info("Saved Config!");
			} catch (IOException e) {
				 modConsole.error("Failed to save config! Error source: " + Arrays.toString(e.getStackTrace()));
			}
			return this;
	 }

	 public Settings set(String id, Object value, @Nullable String page, @Nullable String group) {
			settings.put(id, value);
			return this;
	 }

	 public int getChroma() {
			@Nullable Float chromaDuration = get("chromaDuration", 1.0f);
			if (chromaDuration == null) {
				 chromaDuration = 4.0f;
			}
			double chromaPhase = ((System.nanoTime() / 1_000_000_000.0) % chromaDuration) / chromaDuration;
			return Color.HSBtoRGB((float) chromaPhase, 1, 1);
	 }

	 /*
	 @param newSettings HashMap<String, Object> newSettings the values of all settings to import. Should not include any data related to creating the config menu, that should be reinstantiated every time the game runs.
	 Must be called after all the settings are created
	  */
	 public Settings importSettings(HashMap<String, Object> newSettings) {
			this.settings = newSettings;
			return this;
	 }

	 @SuppressWarnings("ForLoopReplaceableByForEach")
	 public class_437 getConfig(class_437 parentScreen) {
			YetAnotherConfigLib.Builder output = YetAnotherConfigLib.createBuilder();
			String[] pages = configInfo.keySet().toArray(new String[0]);
			output.title(Text.literal(modName + " Settings"))
							.save(this::saveToFile);
			for (int i = 0; i < pages.length; i++) {
				 ConfigCategory.Builder category = ConfigCategory.createBuilder();
				 String curPage = pages[i];
				 category.name(Text.literal(curPage));

//				 String[] groups = configInfo.get(curPage).keySet().toArray(new String[0]);
				 for (int j = 0; j < groupOrders.get(curPage).size(); j++) {
						String groupName = groupOrders.get(curPage).get(j);
						OptionGroup.Builder group = OptionGroup.createBuilder();
						group.name(Text.literal(groupName));

						String[] optionInfo = configInfo.get(curPage).get(groupName).keySet().toArray(new String[0]);
						for (int k = 0; k < optionInfo.length; k++) {
							 String optionId = optionInfo[k];

							 Object[] optionData = configInfo.get(curPage).get(groupName).get(optionId);
							 //optionData: Object[]{id, name, defaultValue, descr, additionalData, min, max, step}
							 //optionData: Object[]{id, name, defaultValue, descr, additionalData, enumClass}

							 List<String> additionalData = List.of(((String) optionData[4]).split("\\."));

							 switch (optionData[2]) {
									case Boolean b -> {
										 Option.Builder<Boolean> option = Option.createBuilder();
										 option.name(Text.literal(optionData[1].toString()));
										 option.description(OptionDescription.of(Text.of((String) optionData[3])));
										 if (additionalData.contains("isTickbox")) {
												option.controller(TickBoxControllerBuilder::create);
										 } else if (additionalData.contains("isYesNo")) {
												option.controller(opt -> BooleanControllerBuilder.create(opt).coloured(true).yesNoFormatter());
										 } else if (additionalData.contains("isOnOff")) {
												option.controller(opt -> BooleanControllerBuilder.create(opt).coloured(true).onOffFormatter());
										 } else {
												option.controller(opt -> BooleanControllerBuilder.create(opt).coloured(true).trueFalseFormatter());
										 }
										 option.binding(b, () -> Objects.requireNonNull(get((String) optionData[0], true)), newValue -> set((String) optionData[0], newValue, curPage, groupName));
										 if (groupName.isEmpty() || groupName.equals("none")) {
												category.option(option.build());
										 } else {
												group.option(option.build());
										 }

									}
									case Float v -> {
										 Option.Builder<Float> option = Option.createBuilder();
										 option.name(Text.literal(optionData[1].toString()));
										 option.description(OptionDescription.of(Text.of((String) optionData[3])));
										 if (additionalData.contains("isSlider")) {
												option.controller(opt -> FloatSliderControllerBuilder.create(opt).range((Float) optionData[5], (Float) optionData[6]).step((Float) optionData[7]));
										 } else {
												option.controller(opt -> FloatFieldControllerBuilder.create(opt).range((Float) optionData[5], (Float) optionData[6]));
										 }
										 option.binding(v, () -> Objects.requireNonNull(get((String) optionData[0], 1.0f)), newValue -> set((String) optionData[0], newValue, curPage, groupName));
										 if (groupName.isEmpty() || groupName.equals("none")) {
												category.option(option.build());
										 } else {
												group.option(option.build());
										 }

									}
									case Integer integer -> {
										 if (additionalData.contains("isColor")) {//use color picker
												Option.Builder<Color> option = Option.createBuilder();
												option.name(Text.literal(optionData[1].toString()));
												option.description(OptionDescription.of(Text.of((String) optionData[3])));
												option.controller(opt -> ColorControllerBuilder.create(opt).allowAlpha(true));
												option.binding(new Color(integer, true), () -> new Color(Objects.requireNonNull(get((String) optionData[0], 1)), true), newValue -> set((String) optionData[0], newValue.getRGB(), curPage, groupName));
												if (groupName.isEmpty() || groupName.equals("none")) {
													 category.option(option.build());
												} else {
													 group.option(option.build());
												}
										 } else {//use int slider
												Option.Builder<Integer> option = Option.createBuilder();
												option.name(Text.literal(optionData[1].toString()));
												option.description(OptionDescription.of(Text.of((String) optionData[3])));
												if (additionalData.contains("isSlider")) {
													 option.controller(opt -> IntegerSliderControllerBuilder.create(opt).range((Integer) optionData[5], (Integer) optionData[6]).step((Integer) optionData[7]));
												} else {
													 option.controller(opt -> IntegerFieldControllerBuilder.create(opt).range((Integer) optionData[5], (Integer) optionData[6]));
												}
												option.binding(integer, () -> Objects.requireNonNull(get((String) optionData[0], 1)), newValue -> set((String) optionData[0], newValue, curPage, groupName));
												if (groupName.isEmpty() || groupName.equals("none")) {
													 category.option(option.build());
												} else {
													 group.option(option.build());
												}
										 }
									}
									case Enum<?> anEnum -> {
										 if (groupName.isEmpty() || groupName.equals("none")) {
												if (additionalData.contains("DROPDOWN")) {
													 category.option(createEnumOption(optionData, curPage, groupName, EnumDisplayType.DROPDOWN));
												} else {
													 category.option(createEnumOption(optionData, curPage, groupName, EnumDisplayType.CYCLE));
												}
										 } else {
												if (additionalData.contains("DROPDOWN")) {
													 group.option(createEnumOption(optionData, curPage, groupName, EnumDisplayType.DROPDOWN));
												} else {
													 group.option(createEnumOption(optionData, curPage, groupName, EnumDisplayType.CYCLE));
												}
										 }
									}
									case String s -> {
										 Option.Builder<String> option = Option.createBuilder();
										 option.name(Text.literal(optionData[1].toString()));
										 option.description(OptionDescription.of(Text.of((String) optionData[3])));
										 option.controller(StringControllerBuilder::create);
										 option.binding(s, () -> Objects.requireNonNull(get((String) optionData[0], "")), newValue -> set((String) optionData[0], newValue, curPage, groupName));
										 if (groupName.isEmpty() || groupName.equals("none")) {
												category.option(option.build());
										 } else {
												group.option(option.build());
										 }
									}
									case null, default ->
													modConsole.error("Error creating config option: " + optionData[1] + ". Invalid default value type.");
							 }
						}
						if (!(groupName.isEmpty() || groupName.equals("none"))) {
							 category.group(group.build());
						}
				 }
				 output.category(category.build());
			}
			return output.build().generateScreen(parentScreen);
	 }

	 public enum EnumDisplayType {
			DROPDOWN, CYCLE
	 }

	 @SuppressWarnings("unchecked")
	 private <E extends Enum<E>> Option<E> createEnumOption(Object[] optionData, String curPage, String groupName, EnumDisplayType displayType) {
			Option.Builder<E> option = Option.createBuilder();
			option.name(class_2561.method_43470(optionData[1].toString()));
			if (displayType==EnumDisplayType.DROPDOWN) {
				 option.description(OptionDescription.of(class_2561.method_30163(optionData[3] +"\n\n\n\nTo change the value, delete the current value and click the new value")));
			} else {
				 option.description(OptionDescription.of(class_2561.method_30163((String) optionData[3])));
			}
			if (displayType == EnumDisplayType.DROPDOWN) {
				 option.controller(EnumDropdownControllerBuilder::create);
			} else {
				 option.controller(opt -> EnumControllerBuilder.create(opt).enumClass((Class<E>) optionData[5]));
			}
			option.binding((E) optionData[2], () -> Objects.requireNonNull(get((String) optionData[0], (Class<E>) optionData[5])), newValue -> set((String) optionData[0], newValue, curPage, groupName));
			return option.build();
	 }
}
