package com.tacz.guns.item;

import com.google.common.base.Suppliers;
import com.tacz.guns.GunMod;
import com.tacz.guns.api.DefaultAssets;
import com.tacz.guns.api.GunProperties;
import com.tacz.guns.api.TimelessAPI;
import com.tacz.guns.api.entity.ReloadState;
import com.tacz.guns.api.item.IGun;
import com.tacz.guns.api.item.attachment.AttachmentType;
import com.tacz.guns.api.item.gun.AbstractGunItem;
import com.tacz.guns.api.item.gun.FireMode;
import com.tacz.guns.api.item.nbt.GunItemDataAccessor;
import com.tacz.guns.command.sub.DebugCommand;
import com.tacz.guns.debug.GunMeleeDebug;
import com.tacz.guns.entity.EntityKineticBullet;
import com.tacz.guns.entity.shooter.ShooterDataHolder;
import com.tacz.guns.resource.index.CommonGunIndex;
import com.tacz.guns.resource.pojo.data.attachment.EffectData;
import com.tacz.guns.resource.pojo.data.attachment.MeleeData;
import com.tacz.guns.resource.pojo.data.gun.*;
import com.tacz.guns.util.AllowAttachmentTagMatcher;
import org.joml.Vector2d;
import org.luaj.vm2.*;
import org.luaj.vm2.lib.jse.CoerceJavaToLua;
import org.luaj.vm2.lib.jse.CoerceLuaToJava;
import org.slf4j.MarkerFactory;

import javax.annotation.Nullable;
import net.minecraft.class_1282;
import net.minecraft.class_1291;
import net.minecraft.class_1293;
import net.minecraft.class_1309;
import net.minecraft.class_1322;
import net.minecraft.class_1657;
import net.minecraft.class_1676;
import net.minecraft.class_1799;
import net.minecraft.class_1890;
import net.minecraft.class_2398;
import net.minecraft.class_243;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_5134;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.DoubleFunction;
import java.util.function.Supplier;

/**
 * 现代枪的逻辑实现
 */
public class ModernKineticGunItem extends AbstractGunItem implements GunItemDataAccessor {
    public static final String TYPE_NAME = "modern_kinetic";

    private static final DoubleFunction<class_1322> AM_FACTORY = amount -> new class_1322(
            class_2960.method_60655(GunMod.MOD_ID, "melee_damage"),
            amount, class_1322.class_1323.field_6328
    );

    public ModernKineticGunItem() {
        super(new class_1793().method_7889(1));
    }

    @Override
    public boolean startBolt(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return false;
        }
        return Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("start_bolt")))
                .map(func -> func.call(CoerceJavaToLua.coerce(api)).checkboolean())
                .orElse(true);
    }

    @Override
    public boolean tickBolt(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return false;
        }
        return Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("tick_bolt")))
                .map(func -> func.call(CoerceJavaToLua.coerce(api)).checkboolean())
                .orElseGet(() -> defaultTickBolt(api));
    }

    @Override
    public void shoot(ShooterDataHolder dataHolder, class_1799 gunItem, Supplier<Float> pitch, Supplier<Float> yaw, class_1309 shooter) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);
        api.setPitchSupplier(pitch);
        api.setYawSupplier(yaw);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return;
        }

        Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("shoot")))
                .ifPresentOrElse(
                        func -> func.call(CoerceJavaToLua.coerce(api)),
                        () -> api.shootOnce(api.isShootingNeedConsumeAmmo()));
    }

    @Override
    public boolean startReload(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return false;
        }
        return Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("start_reload")))
                .map(func -> func.call(CoerceJavaToLua.coerce(api)).checkboolean())
                .orElse(true);
    }

    @Override
    public ReloadState tickReload(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return new ReloadState();
        }
        return Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("tick_reload")))
                .map(func -> {
                    ReloadState reloadState = new ReloadState();
                    Varargs varargs = func.invoke(CoerceJavaToLua.coerce(api));
                    int typeOrdinary = varargs.arg(1).checkint();
                    long countDown = varargs.arg(2).checklong();
                    reloadState.setStateType(ReloadState.StateType.values()[typeOrdinary]);
                    reloadState.setCountDown(countDown);
                    return reloadState;
                })
                .orElseGet(() -> defaultTickReload(api));
    }

    @Override
    public void interruptReload(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return;
        }
        Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("interrupt_reload")))
                .ifPresent(func -> func.call(CoerceJavaToLua.coerce(api)));
    }


    @Override
    public void melee(ShooterDataHolder dataHolder, class_1309 user, class_1799 gunItem) {
        class_2960 gunId = this.getGunId(gunItem);
        TimelessAPI.getCommonGunIndex(gunId).ifPresent(gunIndex -> {
            GunMeleeData meleeData = gunIndex.getGunData().getMeleeData();
            float distance = meleeData.getDistance();

            class_2960 muzzleId = this.getAttachmentId(gunItem, AttachmentType.MUZZLE);
            MeleeData muzzleData = getMeleeData(muzzleId);
            if (muzzleData != null) {
                doMelee(user, distance, muzzleData.getDistance(), muzzleData.getRangeAngle(), muzzleData.getKnockback(), muzzleData.getDamage(), muzzleData.getEffects());
                return;
            }

            class_2960 stockId = this.getAttachmentId(gunItem, AttachmentType.STOCK);
            MeleeData stockData = getMeleeData(stockId);
            if (stockData != null) {
                doMelee(user, distance, stockData.getDistance(), stockData.getRangeAngle(), stockData.getKnockback(), stockData.getDamage(), stockData.getEffects());
                return;
            }

            GunDefaultMeleeData defaultData = meleeData.getDefaultMeleeData();
            if (defaultData == null) {
                return;
            }
            doMelee(user, distance, defaultData.getDistance(), defaultData.getRangeAngle(), defaultData.getKnockback(), defaultData.getDamage(), Collections.emptyList());
        });
    }

    @Override
    public void tickHeat(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        long heatTimestamp = dataHolder.heatTimestamp;
        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return;
        }
        Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("tick_heat")))
                .ifPresentOrElse(
                        func -> func.call(CoerceJavaToLua.coerce(api), LuaValue.valueOf(heatTimestamp)),
                        () -> defaultTickHeat(heatTimestamp, gunItem)
                );
    }

    private void defaultTickHeat(long heatTimestamp, class_1799 gunItem) {
        var iGun = IGun.getIGunOrNull(gunItem);
        if (iGun == null) return;
        TimelessAPI.getCommonGunIndex(iGun.getGunId(gunItem))
                .map(index -> index.getGunData().getHeatData())
                .ifPresent(heatData -> {
                    if (iGun.getHeatAmount(gunItem) <= 0) return;
                    if (iGun.isOverheatLocked(gunItem)) {
                        tickLocked(iGun, gunItem, heatData, heatTimestamp);
                    } else {
                        tickNormal(iGun, gunItem, heatData, heatTimestamp);
                    }
                });
    }

    public void tickLocked(IGun iGun, class_1799 gunStack, GunHeatData heatData, long heatTimestamp) {
        if (System.currentTimeMillis() - heatTimestamp >= heatData.getOverHeatTime()) {
            float heatAmount = iGun.getHeatAmount(gunStack)
                    - ((float) (System.currentTimeMillis() - heatTimestamp) / 10000f)
                    * heatData.getCoolingMultiplier();

            iGun.setHeatAmount(gunStack, heatAmount);
            if (heatAmount <= 0) {
                iGun.setOverheatLocked(gunStack, false);
            }
        }
    }

    public void tickNormal(IGun iGun, class_1799 gunStack, GunHeatData heatData, long heatTimestamp) {
        if (System.currentTimeMillis() - heatTimestamp >= heatData.getCoolingDelay()) {
            float heatAmount = iGun.getHeatAmount(gunStack)
                    - ((float) (System.currentTimeMillis() - heatTimestamp) / 10000f)
                    * heatData.getCoolingMultiplier();

            iGun.setHeatAmount(gunStack, heatAmount);
        }
    }

    public <T> T modifyProperty(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter,
                                String luaMethodName, String id, Class<T> type, T original) {
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return original;
        }

        var afterDefaultModification = defaultPropertyModification.modify(gunItem, shooter, gunIndex, id, original);

        try {
            return Optional.ofNullable(gunIndex.getScript())
                    .map(script -> checkFunction(script.get(luaMethodName)))
                    .map(func -> func.call(CoerceJavaToLua.coerce(api), LuaValue.valueOf(id), CoerceJavaToLua.coerce(afterDefaultModification)))
                    .map(luaValue -> type.cast(CoerceLuaToJava.coerce(luaValue, type)))
                    .orElse(afterDefaultModification);
        } catch (Exception exception) {
            GunMod.LOGGER.warn(MarkerFactory.getMarker("Gun Script"), "Failed to modify gun property {}", id, exception);
            return afterDefaultModification;
        }
    }

    public final DefaultPropertyModification defaultPropertyModification = new DefaultPropertyModification();

    public class DefaultPropertyModification {
        public static final class_2960 SLUGS = class_2960.method_60655(GunMod.MOD_ID, "intrinsic/slug");

        @SuppressWarnings("unchecked")
        public <T> T modify(class_1799 gunItem, class_1309 shooter, CommonGunIndex gunIndex,
                            String id, T original) {
            if (GunProperties.RuntimeOnly.BULLET_AMOUNT.equals(id)) {
                if (AllowAttachmentTagMatcher.matchTag(SLUGS, getAttachmentId(gunItem, AttachmentType.EXTENDED_MAG))) {
                    return (T) Integer.valueOf(1);
                }
            }
            return original;
        }
    }

    @Override
    public void doBulletSpread(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter, class_1676 projectile,
                               int bulletCnt, float processedSpeed, float inaccuracy, float pitch, float yaw) {
        if (!(projectile instanceof EntityKineticBullet bullet)) {
            return;
        }
        ModernKineticGunScriptAPI api = new ModernKineticGunScriptAPI();
        api.setItemStack(gunItem);
        api.setShooter(shooter);
        api.setDataHolder(dataHolder);

        CommonGunIndex gunIndex = api.getGunIndex();
        if (gunIndex == null) {
            return;
        }
        Optional.ofNullable(gunIndex.getScript())
                .map(script -> checkFunction(script.get("calcSpread")))
                .map(func -> func.call(CoerceJavaToLua.coerce(api), LuaValue.valueOf(bulletCnt), LuaValue.valueOf(inaccuracy)))
                .map(luaValue -> {
                    if (luaValue.istable()) {
                        LuaTable table = luaValue.checktable();
                        return new Vector2d(table.get(1).checkdouble(), table.get(2).checkdouble());
                    }
                    return null;
                }).ifPresentOrElse(vector2d -> {
                    bullet.shootFromRotation(shooter, pitch, yaw, 0.0F, processedSpeed, vector2d);
                }, () -> {
                    bullet.method_24919(shooter, pitch, yaw, 0.0F, processedSpeed, inaccuracy);
                });
    }

    private boolean defaultTickBolt(ModernKineticGunScriptAPI api) {
        GunData gunData = api.getGunIndex().getGunData();
        long boltActionTime = (long) (gunData.getBoltActionTime() * 1000);
        float rawBoltFeedTime = gunData.getBoltFeedTime();
        long boltFeedTime = rawBoltFeedTime == -1 ? boltActionTime : (long) (gunData.getBoltFeedTime() * 1000);
        if (api.getBoltTime() < boltFeedTime) {
            return true;
        }
        if (!api.hasAmmoInBarrel()) {
            // 如果是背包直读则检测消耗背包弹药
            if (api.useInventoryAmmo()) {
                if (api.consumeAmmoFromPlayer(1) == 1) {
                    api.setAmmoInBarrel(true);
                }
            } else if (api.removeAmmoFromMagazine(1) != 0) {
                api.setAmmoInBarrel(true);
            }
        }
        return api.getBoltTime() < boltActionTime;
    }

    private ReloadState defaultTickReload(ModernKineticGunScriptAPI api) {
        CommonGunIndex gunIndex = api.getGunIndex();
        // 获取 ReloadData
        GunData gunData = gunIndex.getGunData();
        GunReloadData reloadData = gunData.getReloadData();
        // 计算新的 stateType 和 countDown
        long countDown;
        ReloadState.StateType stateType;
        ReloadState.StateType oldStateType = ReloadState.StateType.values()[api.getReloadStateType()];
        long progressTime = api.getReloadTime();
        if (oldStateType.isReloadingEmpty()) {
            long feedTime = (long) (reloadData.getFeed().getEmptyTime() * 1000);
            long finishingTime = (long) (reloadData.getCooldown().getEmptyTime() * 1000);
            if (progressTime < feedTime) {
                stateType = ReloadState.StateType.EMPTY_RELOAD_FEEDING;
                countDown = feedTime - progressTime;
            } else if (progressTime < finishingTime) {
                stateType = ReloadState.StateType.EMPTY_RELOAD_FINISHING;
                countDown = finishingTime - progressTime;
            } else {
                stateType = ReloadState.StateType.NOT_RELOADING;
                countDown = ReloadState.NOT_RELOADING_COUNTDOWN;
            }
        } else if (oldStateType.isReloadingTactical()) {
            long feedTime = (long) (reloadData.getFeed().getTacticalTime() * 1000);
            long finishingTime = (long) (reloadData.getCooldown().getTacticalTime() * 1000);
            if (progressTime < feedTime) {
                stateType = ReloadState.StateType.TACTICAL_RELOAD_FEEDING;
                countDown = feedTime - progressTime;
            } else if (progressTime < finishingTime) {
                stateType = ReloadState.StateType.TACTICAL_RELOAD_FINISHING;
                countDown = finishingTime - progressTime;
            } else {
                stateType = ReloadState.StateType.NOT_RELOADING;
                countDown = ReloadState.NOT_RELOADING_COUNTDOWN;
            }
        } else {
            stateType = ReloadState.StateType.NOT_RELOADING;
            countDown = ReloadState.NOT_RELOADING_COUNTDOWN;
        }
        // 如果换弹状态发生 装填 -> 收尾 的变化，则需要调用补弹
        if (oldStateType == ReloadState.StateType.EMPTY_RELOAD_FEEDING && oldStateType != stateType) {
            this.defaultReloadFinishing(api, false);
        }
        if (oldStateType == ReloadState.StateType.TACTICAL_RELOAD_FEEDING && oldStateType != stateType) {
            this.defaultReloadFinishing(api, true);
        }
        // 返回 tick 结果
        ReloadState reloadState = new ReloadState();
        reloadState.setStateType(stateType);
        reloadState.setCountDown(countDown);
        return reloadState;
    }

    private void defaultReloadFinishing(ModernKineticGunScriptAPI api, boolean isTactical) {
        GunData data = api.getGunIndex().getGunData();
        int needAmmoCount = api.getNeededAmmoAmount();
        boolean needConsumeAmmo = api.isReloadingNeedConsumeAmmo();
        boolean infinite = data.getReloadData().isInfinite();
        needConsumeAmmo = needConsumeAmmo || infinite;
        switch (data.getReloadData().getType()) {
            case MAGAZINE -> {
                if (needConsumeAmmo) {
                    int consumedAmount = api.consumeAmmoFromPlayer(needAmmoCount);
                    api.putAmmoInMagazine(consumedAmount);
                } else {
                    api.putAmmoInMagazine(needAmmoCount);
                }
            }
            case FUEL -> {
                if (needConsumeAmmo) {
                    int consumedAmount = api.consumeAmmoFromPlayer(1);
                    api.putAmmoInMagazine(needAmmoCount * consumedAmount);
                } else {
                    api.putAmmoInMagazine(needAmmoCount);
                }
            }
            default -> {
                // 未实现
            }
        }
        // 如果不是战术换弹，需要将弹匣中的一枚子弹放到枪膛中
        Bolt boltType = api.getGunIndex().getGunData().getBolt();
        if (!isTactical && (boltType == Bolt.MANUAL_ACTION || boltType == Bolt.CLOSED_BOLT)) {
            int i = api.removeAmmoFromMagazine(1);
            if (i != 0) {
                api.setAmmoInBarrel(true);
            }
        }
    }

    private void doMelee(class_1309 user, float gunDistance, float meleeDistance, float rangeAngle, float knockback, float damage, List<EffectData> effects) {
        // 枪长 + 刺刀长 = 总长
        double distance = gunDistance + meleeDistance;
        float xRot = (float) Math.toRadians(-user.method_36455());
        float yRot = (float) Math.toRadians(-user.method_36454());
        // 视角向量
        class_243 eyeVec = new class_243(0, 0, 1).method_1037(xRot).method_1024(yRot).method_1029().method_1021(distance);
        // 球心坐标
        class_243 centrePos = user.method_33571().method_1020(eyeVec);
        // 先获取范围内所有的实体
        List<class_1309> entityList = user.method_37908().method_18467(class_1309.class, user.method_5829().method_1014(distance));
        Supplier<Float> realDamage = Suppliers.memoize(() -> {
            var instance = user.method_5996(class_5134.field_23721);
            if (instance == null) {
                return damage;
            }
            var oldBase = instance.method_6201();
            var modifier = AM_FACTORY.apply(damage);
            try {
                instance.method_6192(0);
                instance.method_26835(modifier);
                return (float) instance.method_6194();
            } finally {
                instance.method_6192(oldBase);
                instance.method_6202(modifier);
            }
        });
        // 而后检查是否在锥形范围内
        for (class_1309 living : entityList) {
            // 先计算出球心->目标向量
            class_243 targetVec = living.method_33571().method_1020(centrePos);
            // 目标到球心距离
            double targetLength = targetVec.method_1033();
            // 距离在一倍距离之内的，在玩家背后，不进行伤害
            if (targetLength < distance) {
                continue;
            }
            // 计算出向量夹角
            double degree = Math.toDegrees(Math.acos(targetVec.method_1026(eyeVec) / (targetLength * distance)));
            // 向量夹角在范围内的，才能进行伤害
            if (degree < (rangeAngle / 2)) {
                // 判断实体和玩家之间是否有阻隔
                if (user.method_6057(living)) {
                    doPerLivingHurt(user, living, knockback, realDamage.get(), effects);
                }
            }
        }

        // 玩家扣饱食度
        if (user instanceof class_1657 player) {
            player.method_7322(0.1F);
        }

        // Debug 模式
        if (DebugCommand.DEBUG) {
            GunMeleeDebug.showRange(user, (int) Math.round(distance), centrePos, eyeVec, rangeAngle);
        }
    }

    private static void doPerLivingHurt(class_1309 user, class_1309 target, float knockback, float damage, List<EffectData> effects) {
        if (target.equals(user)) {
            return;
        }
        target.method_6005(knockback, (float) Math.sin(Math.toRadians(user.method_36454())), (float) -Math.cos(Math.toRadians(user.method_36454())));
        class_1282 source;
        if (user instanceof class_1657 player) {
            source = user.method_48923().method_48802(player);
        } else {
            source = user.method_48923().method_48812(user);
        }
        target.method_5643(source, damage);
        if (target.method_37908() instanceof class_3218 serverLevel) {
            class_1890.method_60107(serverLevel, target, source);
        }
        if (!target.method_5805()) {
            return;
        }
        for (EffectData data : effects) {
            class_6880<class_1291> mobEffect = class_7923.field_41174.method_55841(data.getEffectId()).orElse(null);
            if (mobEffect == null) {
                continue;
            }
            int time = Math.max(0, data.getTime() * 20);
            int amplifier = Math.max(0, data.getAmplifier());
            class_1293 effectInstance = new class_1293(mobEffect, time, amplifier, false, data.isHideParticles());
            target.method_6092(effectInstance);
        }
        if (user.method_37908() instanceof class_3218 serverLevel) {
            int count = (int) (damage * 0.5);
            serverLevel.method_14199(class_2398.field_11209, target.method_23317(), target.method_23323(0.5), target.method_23321(), count, 0.1, 0, 0.1, 0.2);
        }
    }

    @Nullable
    private MeleeData getMeleeData(class_2960 attachmentId) {
        if (DefaultAssets.isEmptyAttachmentId(attachmentId)) {
            return null;
        }
        return TimelessAPI.getCommonAttachmentIndex(attachmentId).map(index -> index.getData().getMeleeData()).orElse(null);
    }

    private LuaFunction checkFunction(LuaValue luaValue) {
        if (luaValue.isfunction()) {
            return (LuaFunction) luaValue;
        } else if (luaValue.isnil()) {
            return null;
        } else {
            throw new LuaError("bad argument: function or nil expected, got " + luaValue.typename());
        }
    }

    @Override
    public void fireSelect(ShooterDataHolder dataHolder, class_1799 gunItem) {
        class_2960 gunId = this.getGunId(gunItem);
        TimelessAPI.getCommonGunIndex(gunId).map(gunIndex -> {
            FireMode fireMode = this.getFireMode(gunItem);
            List<FireMode> fireModeSet = gunIndex.getGunData().getFireModeSet();
            // 即使玩家拿的是没有的 FireMode，这里也能切换到正常情况
            int nextIndex = (fireModeSet.indexOf(fireMode) + 1) % fireModeSet.size();
            FireMode nextFireMode = fireModeSet.get(nextIndex);
            this.setFireMode(gunItem, nextFireMode);
            return nextFireMode;
        });
    }

    @Override
    public int getLevel(int exp) {
        return 0;
    }

    @Override
    public int getExp(int level) {
        return 0;
    }

    @Override
    public int getMaxLevel() {
        return 0;
    }
}
