package cn.sh1rocu.tacz.util.forge;

import com.mojang.logging.LogUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.class_2960;
import net.minecraft.class_3255;
import net.minecraft.class_3264;
import net.minecraft.class_4239;
import net.minecraft.class_7367;

/**
 * Defines a resource pack from an arbitrary Path.
 * <p>
 * This is primarily intended to support including optional resource packs inside a mod,
 * such as to have alternative textures to use along with Programmer Art, or optional
 * alternative recipes for compatibility ot to replace vanilla recipes.
 */
public class PathPackResources extends class_3255 {
    private static final Logger LOGGER = LogUtils.getLogger();
    private final Path source;

    /**
     * Constructs a java.nio.Path-based resource pack.
     *
     * @param packId    the identifier of the pack.
     *                  This identifier should be unique within the pack finder, preferably the name of the file or folder containing the resources.
     * @param isBuiltin whether this pack resources should be considered builtin
     * @param source    the root path of the pack. This needs to point to the folder that contains "assets" and/or "data", not the asset folder itself!
     */
    public PathPackResources(String packId, boolean isBuiltin, final Path source) {
        super(packId, isBuiltin);
        this.source = source;
    }

    /**
     * Returns the source path containing the resource pack.
     * This is used for error display.
     *
     * @return the root path of the resources.
     */
    public Path getSource() {
        return this.source;
    }

    /**
     * Implement to return a file or folder path for the given set of path components.
     *
     * @param paths One or more path strings to resolve. Can include slash-separated paths.
     * @return the resulting path, which may not exist.
     */
    protected Path resolve(String... paths) {
        Path path = getSource();
        for (String name : paths)
            path = path.resolve(name);
        return path;
    }

    @Nullable
    @Override
    public class_7367<InputStream> method_14410(String... paths) {
        final Path path = resolve(paths);
        if (!Files.exists(path))
            return null;

        return class_7367.create(path);
    }

    @Override
    public void method_14408(class_3264 type, String namespace, String path, class_7664 resourceOutput) {
        class_4239.method_46346(path).get()
                .ifLeft(parts -> net.minecraft.class_3259.method_45183(namespace, resolve(type.method_14413(), namespace).toAbsolutePath(), parts, resourceOutput))
                .ifRight(dataResult -> LOGGER.error("Invalid path {}: {}", path, dataResult.message()));
    }

    @Override
    public Set<String> method_14406(class_3264 type) {
        return getNamespacesFromDisk(type);
    }

    @NotNull
    private Set<String> getNamespacesFromDisk(final class_3264 type) {
        try {
            Path root = resolve(type.method_14413());
            try (Stream<Path> walker = Files.walk(root, 1)) {
                return walker
                        .filter(Files::isDirectory)
                        .map(root::relativize)
                        .filter(p -> p.getNameCount() > 0) // Skip the root entry
                        .map(p -> p.toString().replaceAll("/$", "")) // Remove the trailing slash, if present
                        .filter(s -> !s.isEmpty()) // Filter empty strings, otherwise empty strings default to minecraft namespace in ResourceLocations
                        .collect(Collectors.toSet());
            }
        } catch (IOException e) {
            if (type == class_3264.field_14190) // We still have to add the resource namespace if client resources exist, as we load langs (which are in assets) on server
            {
                return this.method_14406(class_3264.field_14188);
            } else {
                return Collections.emptySet();
            }
        }
    }

    @Override
    public class_7367<InputStream> method_14405(class_3264 type, class_2960 location) {
        return this.method_14410(getPathFromLocation(location.method_12832().startsWith("lang/") ? class_3264.field_14188 : type, location));
    }

    private static String[] getPathFromLocation(class_3264 type, class_2960 location) {
        String[] parts = location.method_12832().split("/");
        String[] result = new String[parts.length + 2];
        result[0] = type.method_14413();
        result[1] = location.method_12836();
        System.arraycopy(parts, 0, result, 2, parts.length);
        return result;
    }

    @Override
    public void close() {
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "%s: %s (%s)", getClass().getName(), this.method_14409(), getSource());
    }
}