/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package cn.sh1rocu.tacz.util.forge;

import cn.sh1rocu.tacz.mixin.accessor.AbstractSliderButtonAccessor;
import com.mojang.blaze3d.systems.RenderSystem;
import org.lwjgl.glfw.GLFW;

import java.text.DecimalFormat;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_357;

/**
 * Slider widget implementation which allows inputting values in a certain range with optional step size.
 *
 * @implNote Note that {@link ExtendedSlider#field_22753} is the fractional progress of the slider from 0 to 1,
 * whereas {@link ExtendedSlider#getValue()} is the actual value from {@code minValue} to {@code maxValue}.
 */
public class ExtendedSlider extends class_357 {
    protected class_2561 prefix;
    protected class_2561 suffix;

    protected double minValue;
    protected double maxValue;

    /**
     * Allows input of discontinuous values with a certain step
     */
    protected double stepSize;

    protected boolean drawString;

    private final DecimalFormat format;

    /**
     * @param x            x position of upper left corner
     * @param y            y position of upper left corner
     * @param width        Width of the widget
     * @param height       Height of the widget
     * @param prefix       {@link class_2561} displayed before the value string
     * @param suffix       {@link class_2561} displayed after the value string
     * @param minValue     Minimum (left) value of slider
     * @param maxValue     Maximum (right) value of slider
     * @param currentValue Starting value when widget is first displayed
     * @param stepSize     Size of step used. Precision will automatically be calculated based on this value if this value is not 0.
     * @param precision    Only used when {@code stepSize} is 0. Limited to a maximum of 4 (inclusive).
     * @param drawString   Should text be displayed on the widget
     */
    public ExtendedSlider(int x, int y, int width, int height, class_2561 prefix, class_2561 suffix, double minValue, double maxValue, double currentValue, double stepSize, int precision, boolean drawString) {
        super(x, y, width, height, class_2561.method_43473(), 0D);
        this.prefix = prefix;
        this.suffix = suffix;
        this.minValue = minValue;
        this.maxValue = maxValue;
        this.stepSize = Math.abs(stepSize);
        this.field_22753 = this.snapToNearest((currentValue - minValue) / (maxValue - minValue));
        this.drawString = drawString;

        if (stepSize == 0D) {
            precision = Math.min(precision, 4);

            StringBuilder builder = new StringBuilder("0");

            if (precision > 0)
                builder.append('.');

            while (precision-- > 0)
                builder.append('0');

            this.format = new DecimalFormat(builder.toString());
        } else if (class_3532.method_20390(this.stepSize, Math.floor(this.stepSize))) {
            this.format = new DecimalFormat("0");
        } else {
            this.format = new DecimalFormat(Double.toString(this.stepSize).replaceAll("\\d", "0"));
        }

        this.method_25346();
    }

    /**
     * Overload with {@code stepSize} set to 1, useful for sliders with whole number values.
     */
    public ExtendedSlider(int x, int y, int width, int height, class_2561 prefix, class_2561 suffix, double minValue, double maxValue, double currentValue, boolean drawString) {
        this(x, y, width, height, prefix, suffix, minValue, maxValue, currentValue, 1D, 0, drawString);
    }

    /**
     * @return Current slider value as a double
     */
    public double getValue() {
        return this.field_22753 * (maxValue - minValue) + minValue;
    }

    /**
     * @return Current slider value as an long
     */
    public long getValueLong() {
        return Math.round(this.getValue());
    }

    /**
     * @return Current slider value as an int
     */
    public int getValueInt() {
        return (int) this.getValueLong();
    }

    /**
     * @param value The new slider value
     */
    public void method_25347(double value) {
        setFractionalValue((value - this.minValue) / (this.maxValue - this.minValue));
    }

    public String getValueString() {
        return this.format.format(this.getValue());
    }

    @Override
    public void method_25348(double mouseX, double mouseY) {
        this.method_25345(mouseX);
    }

    @Override
    protected void method_25349(double mouseX, double mouseY, double dragX, double dragY) {
        super.method_25349(mouseX, mouseY, dragX, dragY);
        this.method_25345(mouseX);
    }

    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        boolean flag = keyCode == GLFW.GLFW_KEY_LEFT;
        if (flag || keyCode == GLFW.GLFW_KEY_RIGHT) {
            if (this.minValue > this.maxValue)
                flag = !flag;
            float f = flag ? -1F : 1F;
            if (stepSize <= 0D)
                this.setFractionalValue(this.field_22753 + (f / (this.field_22758 - 8)));
            else
                this.method_25347(this.getValue() + f * this.stepSize);
        }

        return false;
    }

    private void method_25345(double mouseX) {
        this.setFractionalValue((mouseX - (this.method_46426() + 4)) / (this.field_22758 - 8));
    }

    /**
     * @param fractionalValue fractional progress between 0 and 1
     */
    private void setFractionalValue(double fractionalValue) {
        double oldValue = this.field_22753;
        this.field_22753 = this.snapToNearest(fractionalValue);
        if (!class_3532.method_20390(oldValue, this.field_22753))
            this.method_25344();

        this.method_25346();
    }

    /**
     * Snaps the value, so that the displayed value is the nearest multiple of {@code stepSize}.
     * If {@code stepSize} is 0, no snapping occurs.
     *
     * @param value fractional progress between 0 and 1
     * @return fractional progress between 0 and 1, snapped to the nearest allowed value
     */
    private double snapToNearest(double value) {
        if (stepSize <= 0D)
            return class_3532.method_15350(value, 0D, 1D);

        value = class_3532.method_16436(class_3532.method_15350(value, 0D, 1D), this.minValue, this.maxValue);

        value = (stepSize * Math.round(value / stepSize));

        if (this.minValue > this.maxValue) {
            value = class_3532.method_15350(value, this.maxValue, this.minValue);
        } else {
            value = class_3532.method_15350(value, this.minValue, this.maxValue);
        }

        return class_3532.method_33722(value, this.minValue, this.maxValue, 0D, 1D);
    }

    @Override
    protected void method_25346() {
        if (this.drawString) {
            this.method_25355(class_2561.method_43470("").method_10852(prefix).method_27693(this.getValueString()).method_10852(suffix));
        } else {
            this.method_25355(class_2561.method_43473());
        }
    }

    @Override
    protected void method_25344() {
    }

    @Override
    public void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
        class_310 minecraft = class_310.method_1551();
        guiGraphics.method_51422(1.0F, 1.0F, 1.0F, this.field_22765);
        RenderSystem.enableBlend();
        RenderSystem.defaultBlendFunc();
        RenderSystem.enableDepthTest();
        guiGraphics.method_52706(((AbstractSliderButtonAccessor) this).tacz$getSprite(), this.method_46426(), this.method_46427(), this.method_25368(), this.method_25364());
        guiGraphics.method_52706(((AbstractSliderButtonAccessor) this).tacz$getHandleSprite(), this.method_46426() + (int) (this.field_22753 * (double) (this.field_22758 - 8)), this.method_46427(), 8, this.method_25364());
        guiGraphics.method_51422(1.0F, 1.0F, 1.0F, 1.0F);
        int i = this.field_22763 ? 16777215 : 10526880;
        this.method_49604(guiGraphics, minecraft.field_1772, 2, i | class_3532.method_15386(this.field_22765 * 255.0F) << 24);
    }
}
