package com.tacz.guns.api.item.gun;

import cn.sh1rocu.tacz.api.extension.IItem;
import cn.sh1rocu.tacz.util.itemhandler.IItemHandler;
import cn.sh1rocu.tacz.util.itemhandler.ItemHandlerHelper;
import com.tacz.guns.api.DefaultAssets;
import com.tacz.guns.api.TimelessAPI;
import com.tacz.guns.api.entity.ReloadState;
import com.tacz.guns.api.item.*;
import com.tacz.guns.api.item.attachment.AttachmentType;
import com.tacz.guns.api.item.builder.AmmoItemBuilder;
import com.tacz.guns.api.item.builder.GunItemBuilder;
import com.tacz.guns.client.renderer.item.GunItemRendererWrapper;
import com.tacz.guns.client.resource.index.ClientGunIndex;
import com.tacz.guns.entity.shooter.ShooterDataHolder;
import com.tacz.guns.inventory.tooltip.GunTooltip;
import com.tacz.guns.resource.index.CommonGunIndex;
import com.tacz.guns.resource.pojo.data.gun.FeedType;
import com.tacz.guns.resource.pojo.data.gun.GunData;
import com.tacz.guns.util.AllowAttachmentTagMatcher;
import com.tacz.guns.util.AttachmentDataUtils;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1676;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2371;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_5632;
import net.minecraft.class_756;
import javax.annotation.Nonnull;
import java.util.*;
import java.util.function.Supplier;

public abstract class AbstractGunItem extends class_1792 implements IGun, IAnimationItem, IItem {
    protected AbstractGunItem(class_1793 pProperties) {
        super(pProperties);
    }

    private static Comparator<Map.Entry<class_2960, CommonGunIndex>> idNameSort() {
        return Comparator.comparingInt(m -> m.getValue().getSort());
    }

    /**
     * 开始拉栓时调用，返回 bolt 状态
     *
     * @return bolt 状态。ture 代表开始 bolt，false 则代表不开始。
     */
    public abstract boolean startBolt(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter);

    /**
     * 拉栓 tick 时调用，返回是否仍在 bolt 状态
     *
     * @return 是否仍在 bolt 状态
     */
    public abstract boolean tickBolt(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter);

    /**
     * 射击时触发
     */
    public abstract void shoot(ShooterDataHolder dataHolder, class_1799 gunItem, Supplier<Float> pitch, Supplier<Float> yaw, class_1309 shooter);

    /**
     * 开始换弹时调用
     */
    public abstract boolean startReload(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter);

    /**
     * 换弹时每个 tick 调用
     *
     * @return 如果返回的类型是 NOT_RELOADING 则下一个 tick 不再继续调用
     */
    public abstract ReloadState tickReload(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter);

    /**
     * 尝试打断换弹时调用
     */
    public abstract void interruptReload(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter);

    /**
     * 切换开火模式时调用
     */
    public abstract void fireSelect(ShooterDataHolder dataHolder, class_1799 gunItem);

    /**
     * 近战时调用
     */
    public abstract void melee(ShooterDataHolder dataHolder, class_1309 user, class_1799 gunItem);

    /**
     * 过热 tick 处理<br/>
     * 默认不做任何事情
     */
    public void tickHeat(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter) {
    }

    ;

    /**
     * 初始化子弹角度和速度
     *
     * @param dataHolder     状态数据
     * @param gunItem        枪械物品
     * @param shooter        射击者
     * @param projectile     子弹
     * @param bulletCnt      多弹丸的子弹序数
     * @param processedSpeed 修正后的子弹初速
     * @param inaccuracy     修正后的子弹不准确度
     * @param pitch          射击方向
     * @param yaw            射击方向
     */
    public void doBulletSpread(ShooterDataHolder dataHolder, class_1799 gunItem, class_1309 shooter, class_1676 projectile,
                               int bulletCnt, float processedSpeed, float inaccuracy, float pitch, float yaw) {
        projectile.method_24919(shooter, pitch, yaw, 0.0F, processedSpeed, inaccuracy);
    }

    /**
     * 换弹前的检查，完成如下检查：枪内弹药是否已经填满？玩家背包是否有可用弹药？是否为背包直读？
     *
     * @param shooter 准备换弹的实体
     * @param gunItem 枪械物品
     * @return 是否满足换弹条件
     */
    public boolean canReload(class_1309 shooter, class_1799 gunItem) {
        class_2960 gunId = this.getGunId(gunItem);
        CommonGunIndex gunIndex = TimelessAPI.getCommonGunIndex(gunId).orElse(null);
        if (gunIndex == null) {
            return false;
        }

        int currentAmmoCount = getCurrentAmmoCount(gunItem);
        int maxAmmoCount = AttachmentDataUtils.getAmmoCountWithAttachment(gunItem, gunIndex.getGunData());
        if (currentAmmoCount >= maxAmmoCount) {
            return false;
        }
        // 背包直读不进行换弹
        if (useInventoryAmmo(gunItem)) {
            return false;
        }
        // 无限备弹不需要消耗实际子弹
        if (gunIndex.getGunData().getReloadData().isInfinite()) {
            return true;
        }
        // 虚拟备弹处理
        if (useDummyAmmo(gunItem)) {
            return getDummyAmmoAmount(gunItem) > 0;
        }
        // 检查背包内的弹药数量
        return shooter.tacz$getItemHandler(null).map(cap -> {
            // 背包检查
            for (int i = 0; i < cap.getSlots(); i++) {
                class_1799 checkAmmoStack = cap.getStackInSlot(i);
                if (checkAmmoStack.method_7909() instanceof IAmmo iAmmo && iAmmo.isAmmoOfGun(gunItem, checkAmmoStack)) {
                    return true;
                }
                if (checkAmmoStack.method_7909() instanceof IAmmoBox iAmmoBox && iAmmoBox.isAmmoBoxOfGun(gunItem, checkAmmoStack)) {
                    return true;
                }
            }
            return false;
        }).orElse(false);
    }

    /**
     * 将枪内的弹药全部退至背包（如果背包满了会丢到地上）。不会退枪膛内的弹药。
     * 目前，仅更换弹匣配件时调用。
     *
     * @param player  玩家
     * @param gunItem 枪械物品
     */
    @Override
    public void dropAllAmmo(class_1657 player, class_1799 gunItem) {
        // 背包直读时不调用退弹
        if (useInventoryAmmo(gunItem)) {
            return;
        }
        //TODO 这里操作的对象不应该是 Player 而是 LivingEntity。此外枪膛内的子弹也要退
        int ammoCount = getCurrentAmmoCount(gunItem);
        if (ammoCount <= 0) {
            return;
        }
        class_2960 gunId = getGunId(gunItem);
        TimelessAPI.getCommonGunIndex(gunId).ifPresent(index -> {
            // 如果使用的是虚拟备弹，返还至虚拟备弹
            if (useDummyAmmo(gunItem)) {
                setCurrentAmmoCount(gunItem, 0);
                // 燃料罐类型的换弹不返还
                if (index.getGunData().getReloadData().getType().equals(FeedType.FUEL)) {
                    return;
                }
                addDummyAmmoAmount(gunItem, ammoCount);
                return;
            }

            class_2960 ammoId = index.getGunData().getAmmoId();
            // 创造模式类型的换弹，只填满子弹总数，不进行任何卸载弹药逻辑
            if (player.method_7337()) {
                int maxAmmCount = AttachmentDataUtils.getAmmoCountWithAttachment(gunItem, index.getGunData());
                setCurrentAmmoCount(gunItem, maxAmmCount);
                return;
            }
            // 燃料罐类型的只清空不返还
            if (index.getGunData().getReloadData().getType().equals(FeedType.FUEL)) {
                setCurrentAmmoCount(gunItem, 0);
                return;
            }
            TimelessAPI.getCommonAmmoIndex(ammoId).ifPresent(ammoIndex -> {
                int stackSize = ammoIndex.getStackSize();
                int tmpAmmoCount = ammoCount;
                int roundCount = tmpAmmoCount / (stackSize + 1);
                for (int i = 0; i <= roundCount; i++) {
                    int count = Math.min(tmpAmmoCount, stackSize);
                    class_1799 ammoItem = AmmoItemBuilder.create().setId(ammoId).setCount(count).build();
                    ItemHandlerHelper.giveItemToPlayer(player, ammoItem);
                    tmpAmmoCount -= stackSize;
                }
                setCurrentAmmoCount(gunItem, 0);
            });
        });
    }

    /**
     * 枪械寻弹和扣除背包弹药逻辑
     *
     * @param itemHandler   目标实体的背包
     * @param gunItem       枪械物品
     * @param needAmmoCount 需要的弹药 (物品) 数量
     * @return 寻找到的弹药 (物品) 数量
     */
    @Deprecated
    public int findAndExtractInventoryAmmos(IItemHandler itemHandler, class_1799 gunItem, int needAmmoCount) {
        return findAndExtractInventoryAmmo(itemHandler, gunItem, needAmmoCount);
    }

    /**
     * 枪械寻弹和扣除背包弹药逻辑
     *
     * @param itemHandler   目标实体的背包
     * @param gunItem       枪械物品
     * @param needAmmoCount 需要的弹药 (物品) 数量
     * @return 寻找到的弹药 (物品) 数量
     */
    public int findAndExtractInventoryAmmo(IItemHandler itemHandler, class_1799 gunItem, int needAmmoCount) {
        int cnt = needAmmoCount;
        // 背包检查
        for (int i = 0; i < itemHandler.getSlots(); i++) {
            class_1799 checkAmmoStack = itemHandler.getStackInSlot(i);
            if (checkAmmoStack.method_7909() instanceof IAmmo iAmmo && iAmmo.isAmmoOfGun(gunItem, checkAmmoStack)) {
                class_1799 extractItem = itemHandler.extractItem(i, cnt, false);
                cnt = cnt - extractItem.method_7947();
                if (cnt <= 0) {
                    break;
                }
            }
            if (checkAmmoStack.method_7909() instanceof IAmmoBox iAmmoBox && iAmmoBox.isAmmoBoxOfGun(gunItem, checkAmmoStack)) {
                int boxAmmoCount = iAmmoBox.getAmmoCount(checkAmmoStack);
                int extractCount = Math.min(boxAmmoCount, cnt);
                int remainCount = boxAmmoCount - extractCount;
                iAmmoBox.setAmmoCount(checkAmmoStack, remainCount);
                if (remainCount <= 0) {
                    iAmmoBox.setAmmoId(checkAmmoStack, DefaultAssets.EMPTY_AMMO_ID);
                }
                cnt = cnt - extractCount;
                if (cnt <= 0) {
                    break;
                }
            }
        }
        return needAmmoCount - cnt;
    }

    /**
     * 扣除虚拟弹药逻辑，该方法具有通用的实现，放在此处
     *
     * @param gunItem       枪械物品
     * @param needAmmoCount 需要的弹药(物品)数量
     * @return 找到的弹药(物品)数量
     */
    public int findAndExtractDummyAmmo(class_1799 gunItem, int needAmmoCount) {
        int dummyAmmoCount = getDummyAmmoAmount(gunItem);
        int extractCount = Math.min(dummyAmmoCount, needAmmoCount);
        addDummyAmmoAmount(gunItem, -extractCount);
        return extractCount;
    }

    /**
     * 检查枪械是否允许安装指定的物品作为配件
     */
    @Override
    public boolean allowAttachment(class_1799 gun, class_1799 attachmentItem) {
        IAttachment iAttachment = IAttachment.getIAttachmentOrNull(attachmentItem);
        IGun iGun = IGun.getIGunOrNull(gun);
        if (iGun != null && iAttachment != null) {
            class_2960 gunId = iGun.getGunId(gun);
            class_2960 attachmentId = iAttachment.getAttachmentId(attachmentItem);
            return AllowAttachmentTagMatcher.match(gunId, attachmentId);
        }
        return false;
    }

    /**
     * 检查枪械是否允许安装某种类型的配件
     */
    @Override
    public boolean allowAttachmentType(class_1799 gun, AttachmentType type) {
        IGun iGun = IGun.getIGunOrNull(gun);
        if (iGun != null) {
            return TimelessAPI.getCommonGunIndex(iGun.getGunId(gun)).map(gunIndex -> {
                List<AttachmentType> allowAttachments = gunIndex.getGunData().getAllowAttachments();
                if (allowAttachments == null) {
                    return false;
                }
                return allowAttachments.contains(type);
            }).orElse(false);
        } else {
            return false;
        }
    }

    /**
     * 获取枪械的显示名称
     */
    @Override
    @Nonnull
    @Environment(EnvType.CLIENT)
    public class_2561 method_7864(@Nonnull class_1799 stack) {
        class_2960 gunId = this.getGunId(stack);
        Optional<ClientGunIndex> gunIndex = TimelessAPI.getClientGunIndex(gunId);
        if (gunIndex.isPresent()) {
            return class_2561.method_43471(gunIndex.get().getName());
        }
        return super.method_7864(stack);
    }

    /**
     * 获取某一类 TabType 的所有枪械物品的实例。用于填充创造物品栏和枪械制造台。
     */
    public static class_2371<class_1799> fillItemCategory(GunTabType type) {
        class_2371<class_1799> stacks = class_2371.method_10211();
        TimelessAPI.getAllCommonGunIndex().stream().sorted(idNameSort()).forEach(entry -> {
            CommonGunIndex index = entry.getValue();
            GunData gunData = index.getGunData();
            String key = type.name().toLowerCase(Locale.US);
            String indexType = index.getType();
            if (key.equals(indexType)) {
                class_1799 itemStack = GunItemBuilder.create()
                        .setId(entry.getKey())
                        .setFireMode(gunData.getFireModeSet().get(0))
                        .setAmmoCount(gunData.getAmmoAmount())
                        .setHeatData(gunData.hasHeatData())
                        .setAmmoInBarrel(true)
                        .build();
                stacks.add(itemStack);
            }
        });
        return stacks;
    }

    /**
     * 阻止玩家手臂挥动
     */
    @Override
    public boolean onEntitySwing(class_1799 stack, class_1309 entity) {
        return true;
    }

    @Override
    @Environment(EnvType.CLIENT)
    public class_756 getCustomRenderer() {
        return GunItemRendererWrapper.INSTANCE.get();
    }

    /**
     * 获取在 Tooltip 中渲染的图片
     */
    @Override
    @Nonnull
    public Optional<class_5632> method_32346(class_1799 stack) {
        if (stack.method_7909() instanceof IGun iGun) {
            Optional<CommonGunIndex> optional = TimelessAPI.getCommonGunIndex(this.getGunId(stack));
            if (optional.isPresent()) {
                CommonGunIndex gunIndex = optional.get();
                class_2960 ammoId = gunIndex.getGunData().getAmmoId();
                return Optional.of(new GunTooltip(stack, iGun, ammoId, gunIndex));
            }
        }
        return Optional.empty();
    }

    /**
     * 获取是否使用弹药直读
     *
     * @param gun 枪械
     * @return 是否使用弹药直读
     */
    @Override
    public boolean useInventoryAmmo(class_1799 gun) {
        if (gun.method_7909() instanceof IGun) {
            Optional<CommonGunIndex> gunIndexOptional = TimelessAPI.getCommonGunIndex(this.getGunId(gun));
            if (gunIndexOptional.isEmpty()) {
                return false;
            }
            CommonGunIndex gunIndex = gunIndexOptional.get();
            // 是否为弹药直读
            return gunIndex.getGunData().getReloadData().getType().equals(FeedType.INVENTORY);
        }
        return false;
    }

    /**
     * 获取是否有供给弹药直读的弹药
     *
     * @param gun 枪械
     * @return 是否有供给弹药直读的弹药
     */
    @Override
    public boolean hasInventoryAmmo(class_1309 shooter, class_1799 gun, boolean needCheckAmmo) {
        // 如果不是背包直读，则直接返回 false
        if (!useInventoryAmmo(gun)) {
            return false;
        }
        // 如果不需要检查子弹，则直接返回 true
        if (!needCheckAmmo) {
            return true;
        }
        // 虚拟备弹处理
        if (useDummyAmmo(gun)) {
            return getDummyAmmoAmount(gun) > 0;
        }
        // 检查背包内的弹药数量
        return shooter.tacz$getItemHandler(null).map(cap -> {
            // 背包检查
            for (int i = 0; i < cap.getSlots(); i++) {
                class_1799 checkAmmoStack = cap.getStackInSlot(i);
                if (checkAmmoStack.method_7909() instanceof IAmmo iAmmo && iAmmo.isAmmoOfGun(gun, checkAmmoStack)) {
                    return true;
                }
                if (checkAmmoStack.method_7909() instanceof IAmmoBox iAmmoBox && iAmmoBox.isAmmoBoxOfGun(gun, checkAmmoStack)) {
                    return true;
                }
            }
            return false;
        }).orElse(false);
    }

    /**
     * 获取 RPM
     *
     * @param gun 枪械
     * @return RPM 数值
     */
    public int getRPM(class_1799 gun) {
        if (gun.method_7909() instanceof IGun iGun) {
            return TimelessAPI.getCommonGunIndex(this.getGunId(gun))
                    .map(CommonGunIndex::getGunData)
                    .map(gunData -> {
                        FireMode fireMode = getFireMode(gun);
                        int rpm = gunData.getRoundsPerMinute(fireMode);
                        if (iGun.hasHeatData(gun)) {
                            rpm *= (int) iGun.lerpRPM(gun);
                        }
                        return rpm;
                    }).orElse(300);
        }
        return 300;
    }

    /**
     * 获取是否可以趴下射击
     *
     * @param gun 枪械
     * @return 是否可以趴下射击
     */
    public boolean isCanCrawl(class_1799 gun) {
        if (gun.method_7909() instanceof IGun) {
            return TimelessAPI.getCommonGunIndex(this.getGunId(gun))
                    .map(CommonGunIndex::getGunData)
                    .map(GunData::isCanCrawl)
                    .orElse(false);
        }
        return false;
    }

    @Override
    public boolean isSame(class_1799 i, class_1799 j) {
        IGun iGun1 = IGun.getIGunOrNull(i);
        IGun iGun2 = IGun.getIGunOrNull(j);
        if (iGun1 != null && iGun2 != null) {
            return iGun1.getGunId(i).equals(iGun2.getGunId(j));
        }
        if (i.method_7960() || j.method_7960()) {
            return i.method_7960() && j.method_7960();
        }
        return class_1799.method_7973(i, j);
    }
}
