package com.tacz.guns.resource.network;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.reflect.TypeToken;
import com.tacz.guns.GunMod;
import com.tacz.guns.api.modifier.JsonProperty;
import com.tacz.guns.resource.CommonAssetsManager;
import com.tacz.guns.resource.ICommonResourceProvider;
import com.tacz.guns.resource.filter.RecipeFilter;
import com.tacz.guns.resource.index.CommonAmmoIndex;
import com.tacz.guns.resource.index.CommonAttachmentIndex;
import com.tacz.guns.resource.index.CommonBlockIndex;
import com.tacz.guns.resource.index.CommonGunIndex;
import com.tacz.guns.resource.modifier.AttachmentPropertyManager;
import com.tacz.guns.resource.pojo.data.attachment.AttachmentData;
import com.tacz.guns.resource.pojo.data.block.BlockData;
import com.tacz.guns.resource.pojo.data.gun.GunData;
import com.tacz.guns.util.AllowAttachmentTagMatcher;
import org.jetbrains.annotations.Nullable;
import org.luaj.vm2.LuaTable;

import java.util.*;
import net.minecraft.class_2960;

/**
 * 网络位置的缓存<br/>
 * 用于存储从网络获取的数据
 */
public enum CommonNetworkCache implements ICommonResourceProvider {
    INSTANCE;

    public Map<class_2960, GunData> gunData = new HashMap<>();
    public Map<class_2960, AttachmentData> attachmentData = new HashMap<>();
    public Map<class_2960, RecipeFilter> recipeFilter = new HashMap<>();
    public Map<class_2960, BlockData> blockData = new HashMap<>();
    public Map<class_2960, CommonGunIndex> gunIndex = new HashMap<>();
    public Map<class_2960, CommonAmmoIndex> ammoIndex = new HashMap<>();
    public Map<class_2960, CommonAttachmentIndex> attachmentIndex = new HashMap<>();
    public Map<class_2960, CommonBlockIndex> blockIndex = new HashMap<>();
    public Map<class_2960, Set<String>> attachmentTags = new HashMap<>();
    public Map<class_2960, Set<String>> allowAttachmentTags = new HashMap<>();

    @Nullable
    @Override
    public GunData getGunData(class_2960 id) {
        return gunData.get(id);
    }

    @Nullable
    @Override
    public AttachmentData getAttachmentData(class_2960 attachmentId) {
        return attachmentData.get(attachmentId);
    }

    @Nullable
    @Override
    public BlockData getBlockData(class_2960 id) {
        return blockData.get(id);
    }

    @Nullable
    @Override
    public RecipeFilter getRecipeFilter(class_2960 id) {
        return recipeFilter.get(id);
    }

    @Nullable
    @Override
    public CommonGunIndex getGunIndex(class_2960 id) {
        return gunIndex.get(id);
    }

    @Override
    public @Nullable CommonAmmoIndex getAmmoIndex(class_2960 ammoId) {
        return ammoIndex.get(ammoId);
    }

    @Override
    public @Nullable CommonAttachmentIndex getAttachmentIndex(class_2960 attachmentId) {
        return attachmentIndex.get(attachmentId);
    }

    @Override
    public @Nullable CommonBlockIndex getBlockIndex(class_2960 blockId) {
        return blockIndex.get(blockId);
    }

    @Override
    public @Nullable LuaTable getScript(class_2960 scriptId) {
        return null; // 脚本不需要同步
    }

    @Override
    public Set<Map.Entry<class_2960, CommonGunIndex>> getAllGuns() {
        return gunIndex.entrySet();
    }

    @Override
    public Set<Map.Entry<class_2960, CommonAmmoIndex>> getAllAmmos() {
        return ammoIndex.entrySet();
    }

    @Override
    public Set<Map.Entry<class_2960, CommonAttachmentIndex>> getAllAttachments() {
        return attachmentIndex.entrySet();
    }

    @Override
    public Set<Map.Entry<class_2960, CommonBlockIndex>> getAllBlocks() {
        return blockIndex.entrySet();
    }

    @Override
    public Set<String> getAttachmentTags(class_2960 registryName) {
        return attachmentTags.get(registryName);
    }

    @Override
    public Set<String> getAllowAttachmentTags(class_2960 registryName) {
        return allowAttachmentTags.get(registryName);
    }

    public void fromNetwork(Map<DataType, Map<class_2960, String>> cache) {
        gunData.clear();
        attachmentData.clear();
        gunIndex.clear();
        ammoIndex.clear();
        attachmentIndex.clear();
        blockIndex.clear();
        recipeFilter.clear();
        blockData.clear();

        attachmentTags.clear();
        allowAttachmentTags.clear();
        // 延后处理
        Map<DataType, Map<class_2960, String>> delayed = new HashMap<>();
        for (Map.Entry<DataType, Map<class_2960, String>> entry : cache.entrySet()) {
            switch (entry.getKey()) {
                case GUN_INDEX:
                case AMMO_INDEX:
                case ATTACHMENT_INDEX:
                case BLOCK_INDEX:
                    delayed.put(entry.getKey(), entry.getValue());
                    break;
                default:
                    fromNetwork(entry.getKey(), entry.getValue());
            }
        }
        for (Map.Entry<DataType, Map<class_2960, String>> entry : delayed.entrySet()) {
            fromNetwork(entry.getKey(), entry.getValue());
        }
        AllowAttachmentTagMatcher.resetCache();
    }

    private <T> T parse(String json, Class<T> dataClass) {
        return CommonAssetsManager.GSON.fromJson(json, dataClass);
    }

    private AttachmentData parseAttachmentData(String json) {
        AttachmentData data = CommonAssetsManager.GSON.fromJson(json, AttachmentData.class);
        JsonElement element = CommonAssetsManager.GSON.fromJson(json, JsonElement.class);
        if (data != null) {
            // 序列化注册的配件属性修改
            AttachmentPropertyManager.getModifiers().forEach((key, value) -> {
                if (!element.isJsonObject()) {
                    return;
                }
                JsonObject jsonObject = element.getAsJsonObject();
                if (jsonObject.has(key)) {
                    JsonProperty<?> property = value.readJson(json);
                    property.initComponents();
                    data.addModifier(key, property);
                } else if (jsonObject.has(value.getOptionalFields())) {
                    // 为了兼容旧版本，读取可选字段名
                    JsonProperty<?> property = value.readJson(json);
                    property.initComponents();
                    data.addModifier(key, property);
                }
            });
        }
        return data;
    }

    private void resolveAttachmentTags(Map<class_2960, String> data) {
        for (Map.Entry<class_2960, String> entry : data.entrySet()) {
            List<String> tags = CommonAssetsManager.GSON.fromJson(entry.getValue(), new TypeToken<>() {
            });
            if (entry.getKey().method_12832().startsWith("allow_attachments/") && entry.getKey().method_12832().length() > 18) {
                class_2960 gunId = entry.getKey().method_45136(entry.getKey().method_12832().substring(18));
                allowAttachmentTags.computeIfAbsent(gunId, (v) -> new HashSet<>()).addAll(tags);
            } else {
                attachmentTags.computeIfAbsent(entry.getKey(), (v) -> new HashSet<>()).addAll(tags);
            }
        }
    }


    private void fromNetwork(DataType type, Map<class_2960, String> data) {
        for (Map.Entry<class_2960, String> entry : data.entrySet()) {
            try {
                switch (type) {
                    case GUN_DATA -> gunData.put(entry.getKey(), parse(entry.getValue(), GunData.class));
                    case GUN_INDEX -> gunIndex.put(entry.getKey(), parse(entry.getValue(), CommonGunIndex.class));
                    case AMMO_INDEX -> ammoIndex.put(entry.getKey(), parse(entry.getValue(), CommonAmmoIndex.class));
                    case ATTACHMENT_DATA -> attachmentData.put(entry.getKey(), parseAttachmentData(entry.getValue()));
                    case ATTACHMENT_INDEX ->
                            attachmentIndex.put(entry.getKey(), parse(entry.getValue(), CommonAttachmentIndex.class));
                    case ATTACHMENT_TAGS -> resolveAttachmentTags(data);
                    case BLOCK_INDEX -> blockIndex.put(entry.getKey(), parse(entry.getValue(), CommonBlockIndex.class));
                    case RECIPE_FILTER -> recipeFilter.put(entry.getKey(), parse(entry.getValue(), RecipeFilter.class));
                    case BLOCK_DATA -> blockData.put(entry.getKey(), parse(entry.getValue(), BlockData.class));
                }
            } catch (IllegalArgumentException | JsonParseException exception) {
                GunMod.LOGGER.warn("Failed to parse data from network for {} with id {}", type, entry.getKey(), exception);
            }
        }
    }
}
