package cn.sh1rocu.tacz.util.forge;

import com.google.common.collect.ImmutableSet;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import net.fabricmc.fabric.api.recipe.v1.ingredient.CustomIngredient;
import net.fabricmc.fabric.api.recipe.v1.ingredient.CustomIngredientSerializer;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1935;
import net.minecraft.class_2105;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_7923;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class PartialNBTIngredient implements CustomIngredient {
    private final Set<class_1792> items;
    private final class_2487 nbt;
    private final class_2105 predicate;

    protected PartialNBTIngredient(Set<class_1792> items, class_2487 nbt) {
        if (items.isEmpty()) {
            throw new IllegalArgumentException("Cannot create a PartialNBTIngredient with no items");
        }
        this.items = Collections.unmodifiableSet(items);
        this.nbt = nbt;
        this.predicate = new class_2105(nbt);
    }

    /**
     * Creates a new ingredient matching any item from the list, containing the given NBT
     */
    public static PartialNBTIngredient of(class_2487 nbt, class_1935... items) {
        return new PartialNBTIngredient(Arrays.stream(items).map(class_1935::method_8389).collect(Collectors.toSet()), nbt);
    }

    /**
     * Creates a new ingredient matching the given item, containing the given NBT
     */
    public static PartialNBTIngredient of(class_1935 item, class_2487 nbt) {
        return new PartialNBTIngredient(Set.of(item.method_8389()), nbt);
    }

    @Override
    public boolean test(@Nullable class_1799 input) {
        if (input == null)
            return false;
        return items.contains(input.method_7909()) && predicate.method_9077(input.method_7969());
    }

    @Override
    public List<class_1799> getMatchingStacks() {
        return items.stream().map(item -> {
            class_1799 stack = new class_1799(item);
            // copy NBT to prevent the stack from modifying the original, as capabilities or vanilla item durability will modify the tag
            stack.method_7980(nbt.method_10553());
            return stack;
        }).toList();
    }

    @Override
    public boolean requiresTesting() {
        return true;
    }

    @Override
    public CustomIngredientSerializer<PartialNBTIngredient> getSerializer() {
        return Serializer.INSTANCE;
    }

    public static final class_2960 ID = new class_2960("forge", "partial_nbt");

    public static class Serializer implements CustomIngredientSerializer<PartialNBTIngredient> {
        public static final Serializer INSTANCE = new Serializer();

        @Override
        public class_2960 getIdentifier() {
            return ID;
        }

        @Override
        public PartialNBTIngredient read(JsonObject json) {
            // parse items
            Set<class_1792> items;
            if (json.has("item"))
                items = Set.of(CraftingHelper.getItem(class_3518.method_15265(json, "item"), true));
            else if (json.has("items")) {
                ImmutableSet.Builder<class_1792> builder = ImmutableSet.builder();
                JsonArray itemArray = class_3518.method_15261(json, "items");
                for (int i = 0; i < itemArray.size(); i++) {
                    builder.add(CraftingHelper.getItem(class_3518.method_15287(itemArray.get(i), "items[" + i + ']'), true));
                }
                items = builder.build();
            } else
                throw new JsonSyntaxException("Must set either 'item' or 'items'");

            // parse NBT
            if (!json.has("nbt"))
                throw new JsonSyntaxException("Missing nbt, expected to find a String or JsonObject");
            class_2487 nbt = CraftingHelper.getNBT(json.get("nbt"));

            return new PartialNBTIngredient(items, nbt);
        }

        @Override
        public void write(JsonObject json, PartialNBTIngredient ingredient) {
            json.addProperty("type", ID.toString());
            if (ingredient.items.size() == 1) {
                json.addProperty("item", class_7923.field_41178.method_10221(ingredient.items.iterator().next()).toString());
            } else {
                JsonArray items = new JsonArray();
                // ensure the order of items in the set is deterministic when saved to JSON
                ingredient.items.stream().map(class_7923.field_41178::method_10221).sorted().forEach(name -> items.add(name.toString()));
                json.add("items", items);
            }
            json.addProperty("nbt", ingredient.nbt.toString());
        }

        @Override
        public PartialNBTIngredient read(class_2540 buffer) {
            Set<class_1792> items = Stream.generate(() -> class_7923.field_41178.method_10223(buffer.method_10810())).limit(buffer.method_10816()).collect(Collectors.toSet());
            class_2487 nbt = buffer.method_10798();
            return new PartialNBTIngredient(items, Objects.requireNonNull(nbt));
        }

        @Override
        public void write(class_2540 buffer, PartialNBTIngredient ingredient) {
            buffer.method_10804(ingredient.items.size());
            for (class_1792 item : ingredient.items)
                buffer.method_10812(class_7923.field_41178.method_10221(item));
            buffer.method_10794(ingredient.nbt);
        }
    }
}
