/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.wmb.tracking;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.minecraft.core.BlockPos;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.phys.AABB;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.texboobcat.wmb.config.WmbConfig;
import org.texboobcat.wmb.metrics.Metrics;
import org.texboobcat.wmb.tune.AutoTuner;

public final class AsyncEntityTracker {
    private static final Logger LOGGER = LogManager.getLogger((String)"wmb:tracker");
    private static volatile ThreadPoolExecutor globalPool;
    private static final ConcurrentMap<String, ThreadPoolExecutor> dimensionPools;
    private static final ReentrantReadWriteLock poolLock;
    private static final ConcurrentMap<UUID, PlayerTrackingState> playerStates;
    private static final Set<UUID> playersInProgress;
    private static final ConcurrentLinkedQueue<TrackingResult> completedResults;
    private static final ConcurrentMap<UUID, Long> lastResultProcessTime;
    private static final AtomicInteger currentTick;
    private static final AtomicLong totalTasksSubmitted;
    private static final AtomicLong totalTasksCompleted;
    private static final AtomicLong totalTasksTimedOut;
    private static final AtomicLong totalEntitiesProcessed;
    private static final int MAX_ENTITIES_PER_TASK = 2048;
    private static final int MAX_BATCH_SIZE = 32;
    private static final long TASK_TIMEOUT_MS = 50L;
    private static final int MIN_CHUNK_SIZE = 64;

    private AsyncEntityTracker() {
    }

    private static void ensureThreadPools() {
        WmbConfig cfg = WmbConfig.get();
        poolLock.writeLock().lock();
        try {
            if (!cfg.asyncTracker.perDimensionPools) {
                if (globalPool == null || globalPool.getCorePoolSize() != cfg.asyncTracker.threadPoolSize) {
                    if (globalPool != null) {
                        AsyncEntityTracker.shutdownPool(globalPool, "global");
                    }
                    globalPool = AsyncEntityTracker.createThreadPool(cfg.asyncTracker.threadPoolSize, "WMB-AsyncTrack-Global");
                }
                dimensionPools.values().forEach(pool -> AsyncEntityTracker.shutdownPool(pool, "dimension"));
                dimensionPools.clear();
            }
        }
        finally {
            poolLock.writeLock().unlock();
        }
    }

    private static ThreadPoolExecutor createThreadPool(int size, final String namePrefix) {
        return new ThreadPoolExecutor(size, size, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(size * 4), new ThreadFactory(){
            private final AtomicInteger threadIndex = new AtomicInteger(1);

            @Override
            public Thread newThread(Runnable r) {
                Thread t = new Thread(r, namePrefix + "-" + this.threadIndex.getAndIncrement());
                t.setDaemon(true);
                t.setUncaughtExceptionHandler((thread, ex) -> LOGGER.error("Uncaught exception in tracker thread {}", (Object)thread.getName(), (Object)ex));
                return t;
            }
        }, new ThreadPoolExecutor.CallerRunsPolicy());
    }

    private static ThreadPoolExecutor getExecutorFor(String dimensionId) {
        WmbConfig cfg = WmbConfig.get();
        if (!cfg.asyncTracker.perDimensionPools) {
            AsyncEntityTracker.ensureThreadPools();
            return globalPool;
        }
        return dimensionPools.computeIfAbsent(dimensionId, dimId -> {
            LOGGER.debug("Creating thread pool for dimension: {}", dimId);
            return AsyncEntityTracker.createThreadPool(cfg.asyncTracker.threadPoolSizePerDim, "WMB-AsyncTrack-" + dimId.replace(':', '_'));
        });
    }

    private static void shutdownPool(ThreadPoolExecutor pool, String name) {
        if (pool != null && !pool.isShutdown()) {
            LOGGER.debug("Shutting down {} thread pool", (Object)name);
            pool.shutdown();
            try {
                if (!pool.awaitTermination(5L, TimeUnit.SECONDS)) {
                    LOGGER.warn("{} pool didn't terminate gracefully, forcing shutdown", (Object)name);
                    pool.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                pool.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    public static void onServerPreTick(MinecraftServer server) {
        List players;
        if (!WmbConfig.get().asyncTracker.enabled) {
            return;
        }
        AsyncEntityTracker.ensureThreadPools();
        int tick = currentTick.incrementAndGet();
        if (WmbConfig.get().metrics.enabled) {
            Metrics.onTrackerQueueSize(AsyncEntityTracker.getTotalQueueSize());
        }
        if ((players = server.m_6846_().m_11314_()).isEmpty()) {
            return;
        }
        int processed = 0;
        int maxPerTick = Math.max(1, players.size() / 4);
        for (ServerPlayer player : players) {
            if (processed >= maxPerTick) break;
            if (!AsyncEntityTracker.shouldSchedulePlayer(player, tick)) continue;
            AsyncEntityTracker.schedulePlayerTracking(player, tick);
            ++processed;
        }
        AsyncEntityTracker.cleanupDisconnectedPlayers(server);
    }

    private static boolean shouldSchedulePlayer(ServerPlayer player, int currentTick) {
        int threadCount;
        ThreadPoolExecutor executor;
        int queueSize;
        int backpressureAdjustment;
        int tunerAdjustment;
        UUID playerId = player.m_20148_();
        if (playersInProgress.contains(playerId)) {
            return false;
        }
        PlayerTrackingState state = (PlayerTrackingState)playerStates.get(playerId);
        if (state == null) {
            return true;
        }
        WmbConfig cfg = WmbConfig.get();
        int baseInterval = Math.max(1, cfg.asyncTracker.updateInterval);
        int effectiveInterval = Math.max(1, baseInterval + (tunerAdjustment = AutoTuner.getTrackerIntervalAddend()) + (backpressureAdjustment = Math.min(10, (queueSize = (executor = AsyncEntityTracker.getExecutorFor(player.m_284548_().m_46472_().m_135782_().toString())).getQueue().size()) / Math.max(1, threadCount = executor.getCorePoolSize()))));
        if (currentTick - state.lastScheduledTick < effectiveInterval) {
            return false;
        }
        return currentTick - state.lastCompletedTick >= Math.max(1, cfg.asyncTracker.cacheDuration);
    }

    private static void schedulePlayerTracking(ServerPlayer player, int tick) {
        UUID playerId = player.m_20148_();
        if (!playersInProgress.add(playerId)) {
            return;
        }
        try {
            PlayerSnapshot snapshot = PlayerSnapshot.capture(player);
            PlayerTrackingState state = playerStates.computeIfAbsent(playerId, k -> new PlayerTrackingState());
            state.lastScheduledTick = tick;
            List<EntitySnapshot> entitySnapshots = AsyncEntityTracker.collectEntitySnapshots(player);
            if (entitySnapshots.isEmpty()) {
                TrackingResult emptyResult = new TrackingResult(playerId, snapshot.dimensionId, Collections.emptySet(), new HashSet<UUID>(state.currentlyTracked), tick, 0);
                completedResults.offer(emptyResult);
                return;
            }
            ThreadPoolExecutor executor = AsyncEntityTracker.getExecutorFor(snapshot.dimensionId);
            CompletableFuture<Object> future = CompletableFuture.supplyAsync(() -> AsyncEntityTracker.processPlayerTracking(snapshot, entitySnapshots, state, tick), executor).completeOnTimeout(null, 50L, TimeUnit.MILLISECONDS);
            future.whenComplete((result, throwable) -> {
                if (throwable != null) {
                    if (throwable instanceof TimeoutException) {
                        totalTasksTimedOut.incrementAndGet();
                        LOGGER.debug("Tracking task timed out for player {}", (Object)snapshot.playerName);
                    } else {
                        LOGGER.warn("Tracking task failed for player {}", (Object)snapshot.playerName, throwable);
                    }
                    completedResults.offer(new TrackingResult(playerId, snapshot.dimensionId, Collections.emptySet(), Collections.emptySet(), tick, 0));
                } else if (result != null) {
                    completedResults.offer((TrackingResult)result);
                } else {
                    completedResults.offer(new TrackingResult(playerId, snapshot.dimensionId, Collections.emptySet(), Collections.emptySet(), tick, 0));
                }
            });
            totalTasksSubmitted.incrementAndGet();
        }
        catch (Exception e) {
            LOGGER.error("Failed to schedule tracking for player {}", (Object)player.m_6302_(), (Object)e);
            playersInProgress.remove(playerId);
        }
    }

    private static List<EntitySnapshot> collectEntitySnapshots(ServerPlayer player) {
        ServerLevel level = player.m_284548_();
        MinecraftServer server = level.m_7654_();
        int viewDistanceChunks = server.m_6846_().m_11312_();
        int searchRadius = Math.max(64, viewDistanceChunks * 16 + 32);
        double px = player.m_20185_();
        double py = player.m_20186_();
        double pz = player.m_20189_();
        AABB searchBox = new AABB(px - (double)searchRadius, py - 256.0, pz - (double)searchRadius, px + (double)searchRadius, py + 256.0, pz + (double)searchRadius);
        List nearbyEntities = level.m_6443_(Entity.class, searchBox, entity -> !(entity instanceof ServerPlayer) && !entity.m_213877_());
        int entityCount = Math.min(nearbyEntities.size(), 2048);
        ArrayList<EntitySnapshot> snapshots = new ArrayList<EntitySnapshot>(entityCount);
        for (int i = 0; i < entityCount; ++i) {
            Entity entity2 = (Entity)nearbyEntities.get(i);
            UUID entityId = entity2.m_20148_();
            if (entityId == null || entityId.equals(player.m_20148_())) continue;
            snapshots.add(new EntitySnapshot(entityId, entity2.m_20185_(), entity2.m_20186_(), entity2.m_20189_(), entity2.m_6095_().m_20675_()));
        }
        return snapshots;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static TrackingResult processPlayerTracking(PlayerSnapshot playerSnapshot, List<EntitySnapshot> entitySnapshots, PlayerTrackingState state, int tick) {
        long startTime = System.nanoTime();
        try {
            WmbConfig cfg = WmbConfig.get();
            double maxTrackingDistanceSq = Math.pow(cfg.asyncTracker.maxTrackingDistance, 2.0);
            HashSet<UUID> newTrackedEntities = new HashSet<UUID>();
            int chunkSize = Math.max(64, entitySnapshots.size() / 4);
            for (int start = 0; start < entitySnapshots.size(); start += chunkSize) {
                int end = Math.min(start + chunkSize, entitySnapshots.size());
                for (int i = start; i < end; ++i) {
                    EntitySnapshot entity = entitySnapshots.get(i);
                    double dx = entity.x - playerSnapshot.x;
                    double dz = entity.z - playerSnapshot.z;
                    double distanceSq = dx * dx + dz * dz;
                    if (!(distanceSq <= maxTrackingDistanceSq)) continue;
                    newTrackedEntities.add(entity.id);
                }
            }
            HashSet<UUID> currentTracked = new HashSet<UUID>(state.currentlyTracked);
            HashSet<UUID> toAdd = new HashSet<UUID>(newTrackedEntities);
            toAdd.removeAll(currentTracked);
            HashSet<UUID> toRemove = new HashSet<UUID>(currentTracked);
            toRemove.removeAll(newTrackedEntities);
            totalEntitiesProcessed.addAndGet(entitySnapshots.size());
            TrackingResult trackingResult = new TrackingResult(playerSnapshot.playerId, playerSnapshot.dimensionId, toAdd, toRemove, tick, entitySnapshots.size());
            return trackingResult;
        }
        finally {
            if (WmbConfig.get().metrics.enabled) {
                long duration = System.nanoTime() - startTime;
                Metrics.onTrackerTaskDuration(duration);
            }
        }
    }

    public static void onServerPostTick(MinecraftServer server) {
        if (!WmbConfig.get().asyncTracker.enabled) {
            return;
        }
        AsyncEntityTracker.processCompletedResults(server);
        if (currentTick.get() % 200 == 0) {
            AsyncEntityTracker.performMaintenance(server);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void processCompletedResults(MinecraftServer server) {
        TrackingResult result;
        int processed = 0;
        while ((result = completedResults.poll()) != null && processed < 32) {
            try {
                AsyncEntityTracker.applyTrackingResult(server, result);
                ++processed;
            }
            catch (Exception e) {
                LOGGER.error("Failed to apply tracking result for player {}", (Object)result.playerId, (Object)e);
            }
            finally {
                playersInProgress.remove(result.playerId);
                totalTasksCompleted.incrementAndGet();
            }
        }
    }

    private static void applyTrackingResult(MinecraftServer server, TrackingResult result) {
        ServerPlayer player = server.m_6846_().m_11259_(result.playerId);
        if (player == null) {
            playerStates.remove(result.playerId);
            lastResultProcessTime.remove(result.playerId);
            return;
        }
        String currentDimension = player.m_284548_().m_46472_().m_135782_().toString();
        if (!currentDimension.equals(result.playerDimensionId)) {
            LOGGER.debug("Player {} changed dimensions, skipping result", (Object)player.m_6302_());
            return;
        }
        PlayerTrackingState state = (PlayerTrackingState)playerStates.get(result.playerId);
        if (state == null) {
            return;
        }
        ServerLevel level = player.m_284548_();
        AsyncEntityTracker.applyEntityVisibilityChanges(player, level, result.toAdd, result.toRemove);
        state.currentlyTracked.addAll(result.toAdd);
        state.currentlyTracked.removeAll(result.toRemove);
        state.lastCompletedTick = result.tick;
        state.totalEntitiesProcessed += (long)result.entitiesProcessed;
        lastResultProcessTime.put(result.playerId, System.currentTimeMillis());
    }

    private static void applyEntityVisibilityChanges(ServerPlayer player, ServerLevel level, Set<UUID> toAdd, Set<UUID> toRemove) {
        Entity entity;
        for (UUID entityId : toAdd) {
            entity = level.m_8791_(entityId);
            if (entity == null || entity.m_213877_()) continue;
            try {
                entity.m_6457_(player);
            }
            catch (Exception e) {
                LOGGER.debug("Failed to start tracking entity {} for player {}", (Object)entityId, (Object)player.m_6302_(), (Object)e);
            }
        }
        for (UUID entityId : toRemove) {
            entity = level.m_8791_(entityId);
            if (entity == null) continue;
            try {
                entity.m_6452_(player);
            }
            catch (Exception e) {
                LOGGER.debug("Failed to stop tracking entity {} for player {}", (Object)entityId, (Object)player.m_6302_(), (Object)e);
            }
        }
    }

    private static void cleanupDisconnectedPlayers(MinecraftServer server) {
        if (currentTick.get() % 100 != 0) {
            return;
        }
        HashSet<UUID> connectedPlayers = new HashSet<UUID>();
        for (ServerPlayer player : server.m_6846_().m_11314_()) {
            connectedPlayers.add(player.m_20148_());
        }
        playerStates.entrySet().removeIf(entry -> {
            UUID playerId = (UUID)entry.getKey();
            if (!connectedPlayers.contains(playerId)) {
                LOGGER.debug("Cleaning up state for disconnected player {}", (Object)playerId);
                playersInProgress.remove(playerId);
                lastResultProcessTime.remove(playerId);
                return true;
            }
            return false;
        });
    }

    private static void performMaintenance(MinecraftServer server) {
        long cutoff = System.currentTimeMillis() - 300000L;
        lastResultProcessTime.entrySet().removeIf(entry -> (Long)entry.getValue() < cutoff);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Tracker stats: submitted={}, completed={}, timedOut={}, entitiesProcessed={}, queueSize={}", (Object)totalTasksSubmitted.get(), (Object)totalTasksCompleted.get(), (Object)totalTasksTimedOut.get(), (Object)totalEntitiesProcessed.get(), (Object)AsyncEntityTracker.getTotalQueueSize());
        }
    }

    private static int getTotalQueueSize() {
        poolLock.readLock().lock();
        try {
            if (!WmbConfig.get().asyncTracker.perDimensionPools) {
                int n = globalPool != null ? globalPool.getQueue().size() : 0;
                return n;
            }
            int n = dimensionPools.values().stream().mapToInt(pool -> pool.getQueue().size()).sum();
            return n;
        }
        finally {
            poolLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getStatusString() {
        WmbConfig cfg = WmbConfig.get();
        long submitted = totalTasksSubmitted.get();
        long completed = totalTasksCompleted.get();
        long timedOut = totalTasksTimedOut.get();
        int queued = AsyncEntityTracker.getTotalQueueSize();
        int inProgress = playersInProgress.size();
        poolLock.readLock().lock();
        try {
            int poolCount;
            int totalThreads;
            if (!cfg.asyncTracker.perDimensionPools) {
                totalThreads = globalPool != null ? globalPool.getCorePoolSize() : 0;
                poolCount = globalPool != null ? 1 : 0;
            } else {
                totalThreads = dimensionPools.values().stream().mapToInt(ThreadPoolExecutor::getCorePoolSize).sum();
                poolCount = dimensionPools.size();
            }
            String string = String.format("AsyncTracker: enabled=%s, perDim=%s, threads=%d, pools=%d, submitted=%d, completed=%d, timedOut=%d, queued=%d, inProgress=%d, players=%d, entitiesProcessed=%d", cfg.asyncTracker.enabled, cfg.asyncTracker.perDimensionPools, totalThreads, poolCount, submitted, completed, timedOut, queued, inProgress, playerStates.size(), totalEntitiesProcessed.get());
            return string;
        }
        finally {
            poolLock.readLock().unlock();
        }
    }

    public static void shutdown() {
        LOGGER.info("Shutting down AsyncEntityTracker");
        poolLock.writeLock().lock();
        try {
            AsyncEntityTracker.shutdownPool(globalPool, "global");
            globalPool = null;
            dimensionPools.values().forEach(pool -> AsyncEntityTracker.shutdownPool(pool, "dimension"));
            dimensionPools.clear();
        }
        finally {
            poolLock.writeLock().unlock();
        }
        playerStates.clear();
        playersInProgress.clear();
        completedResults.clear();
        lastResultProcessTime.clear();
    }

    static {
        dimensionPools = new ConcurrentHashMap<String, ThreadPoolExecutor>();
        poolLock = new ReentrantReadWriteLock();
        playerStates = new ConcurrentHashMap<UUID, PlayerTrackingState>();
        playersInProgress = ConcurrentHashMap.newKeySet();
        completedResults = new ConcurrentLinkedQueue();
        lastResultProcessTime = new ConcurrentHashMap<UUID, Long>();
        currentTick = new AtomicInteger(0);
        totalTasksSubmitted = new AtomicLong(0L);
        totalTasksCompleted = new AtomicLong(0L);
        totalTasksTimedOut = new AtomicLong(0L);
        totalEntitiesProcessed = new AtomicLong(0L);
    }

    private static class PlayerTrackingState {
        final Set<UUID> currentlyTracked = ConcurrentHashMap.newKeySet();
        volatile int lastScheduledTick = 0;
        volatile int lastCompletedTick = 0;
        volatile long totalEntitiesProcessed = 0L;

        private PlayerTrackingState() {
        }
    }

    private static final class PlayerSnapshot {
        final UUID playerId;
        final String playerName;
        final String dimensionId;
        final double x;
        final double y;
        final double z;
        final int chunkX;
        final int chunkZ;

        private PlayerSnapshot(UUID playerId, String playerName, String dimensionId, double x, double y, double z, int chunkX, int chunkZ) {
            this.playerId = playerId;
            this.playerName = playerName;
            this.dimensionId = dimensionId;
            this.x = x;
            this.y = y;
            this.z = z;
            this.chunkX = chunkX;
            this.chunkZ = chunkZ;
        }

        static PlayerSnapshot capture(ServerPlayer player) {
            BlockPos blockPos = player.m_20183_();
            return new PlayerSnapshot(player.m_20148_(), player.m_6302_(), player.m_284548_().m_46472_().m_135782_().toString(), player.m_20185_(), player.m_20186_(), player.m_20189_(), blockPos.m_123341_() >> 4, blockPos.m_123343_() >> 4);
        }
    }

    private record TrackingResult(UUID playerId, String playerDimensionId, Set<UUID> toAdd, Set<UUID> toRemove, int tick, int entitiesProcessed) {
    }

    private record EntitySnapshot(UUID id, double x, double y, double z, String type) {
    }
}

