/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.wmb.tune;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.texboobcat.wmb.config.WmbConfig;

public final class AutoTuner {
    private static final Logger LOGGER = LogManager.getLogger((String)"wmb:autotuner");
    private static final AtomicInteger ticksInWindow = new AtomicInteger(0);
    private static final AtomicLong nanosInWindow = new AtomicLong(0L);
    private static final AtomicInteger cooldownTicksLeft = new AtomicInteger(0);
    private static final AtomicInteger trackerIntervalAddend = new AtomicInteger(0);
    private static final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private static volatile ConfigSnapshot lastConfig = null;
    private static final PidState pidState = new PidState();
    private static volatile double tickEmaMs = 0.0;
    private static final double EMA_ALPHA = 0.1;

    private AutoTuner() {
    }

    public static void onServerTick(long tickDurationNanos) {
        int windowSize;
        int currentTicks;
        WmbConfig cfg;
        try {
            cfg = WmbConfig.get();
        }
        catch (Throwable t) {
            LOGGER.debug("Config not available, skipping auto-tune tick");
            return;
        }
        if (!cfg.tuning.enabled) {
            AutoTuner.reset();
            return;
        }
        ConfigSnapshot currentConfig = new ConfigSnapshot(cfg);
        if (AutoTuner.hasConfigChanged(currentConfig)) {
            AutoTuner.resetForConfigChange(currentConfig);
        }
        cooldownTicksLeft.updateAndGet(val -> Math.max(0, val - 1));
        if (tickDurationNanos > 0L) {
            AutoTuner.updateTickMetrics(tickDurationNanos);
        }
        if ((currentTicks = ticksInWindow.incrementAndGet()) >= (windowSize = Math.max(1, cfg.tuning.windowTicks))) {
            AutoTuner.processWindow(cfg);
        }
    }

    public static int getTrackerIntervalAddend() {
        return trackerIntervalAddend.get();
    }

    public static PerformanceMetrics getMetrics() {
        lock.readLock().lock();
        try {
            PerformanceMetrics performanceMetrics = new PerformanceMetrics(tickEmaMs, AutoTuner.pidState.integral, AutoTuner.pidState.lastError, trackerIntervalAddend.get(), cooldownTicksLeft.get());
            return performanceMetrics;
        }
        finally {
            lock.readLock().unlock();
        }
    }

    private static void updateTickMetrics(long tickDurationNanos) {
        double newEma;
        double currentEma;
        double tickMs = (double)tickDurationNanos / 1000000.0;
        while (!AutoTuner.compareAndSetTickEma(currentEma, newEma = (currentEma = tickEmaMs) == 0.0 ? tickMs : 0.1 * tickMs + 0.9 * currentEma)) {
        }
        nanosInWindow.addAndGet(tickDurationNanos);
    }

    private static boolean compareAndSetTickEma(double expect, double update) {
        if (Double.compare(tickEmaMs, expect) == 0) {
            tickEmaMs = update;
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void processWindow(WmbConfig cfg) {
        lock.writeLock().lock();
        try {
            int ticks = ticksInWindow.getAndSet(0);
            long nanos = nanosInWindow.getAndSet(0L);
            if (ticks <= 0) {
                return;
            }
            double windowAvgMs = (double)nanos / 1000000.0 / (double)ticks;
            double target = cfg.tuning.targetTickMs;
            double smoothedMs = tickEmaMs > 0.0 ? tickEmaMs : windowAvgMs;
            LOGGER.debug("Window complete: {} ticks, windowAvg={}ms, smoothed={}ms, target={}ms", (Object)ticks, (Object)String.format("%.2f", windowAvgMs), (Object)String.format("%.2f", smoothedMs), (Object)target);
            ControlOutput output = AutoTuner.calculateControl(cfg, smoothedMs, target);
            AutoTuner.applyPathfindingControl(cfg, output.pathfindingStep);
            int clampedTrackerAdj = AutoTuner.clamp(output.trackerAdjustment, -cfg.tuning.maxStepPerWindow, cfg.tuning.maxStepPerWindow);
            trackerIntervalAddend.set(clampedTrackerAdj);
        }
        finally {
            lock.writeLock().unlock();
        }
    }

    private static ControlOutput calculateControl(WmbConfig cfg, double currentMs, double targetMs) {
        if (currentMs <= 0.0) {
            return new ControlOutput(0, 0);
        }
        double kp = cfg.tuning.kp;
        double ki = cfg.tuning.ki;
        double kd = cfg.tuning.kd;
        if (kp == 0.0 && ki == 0.0 && kd == 0.0) {
            return AutoTuner.calculateSimpleControl(currentMs, targetMs);
        }
        return AutoTuner.calculatePidControl(cfg, currentMs, targetMs);
    }

    private static ControlOutput calculateSimpleControl(double currentMs, double targetMs) {
        double deadband = 1.0;
        if (currentMs > targetMs + 1.0) {
            return new ControlOutput(1, 1);
        }
        if (currentMs < targetMs - 1.0) {
            return new ControlOutput(-1, -1);
        }
        return new ControlOutput(0, 0);
    }

    private static ControlOutput calculatePidControl(WmbConfig cfg, double currentMs, double targetMs) {
        double error = targetMs - currentMs;
        AutoTuner.pidState.integral += error;
        double maxIntegral = Math.max(0.0, cfg.tuning.integralMaxAbs);
        AutoTuner.pidState.integral = AutoTuner.clamp(AutoTuner.pidState.integral, -maxIntegral, maxIntegral);
        double derivative = AutoTuner.pidState.hasLastError ? error - AutoTuner.pidState.lastError : 0.0;
        AutoTuner.pidState.lastError = error;
        AutoTuner.pidState.hasLastError = true;
        double pidOutput = cfg.tuning.kp * error + cfg.tuning.ki * AutoTuner.pidState.integral + cfg.tuning.kd * derivative;
        int maxStep = Math.max(1, cfg.tuning.maxStepPerWindow);
        int pathfindingStep = AutoTuner.clamp((int)Math.round(pidOutput), -maxStep, maxStep);
        int trackerAdjustment = -pathfindingStep;
        LOGGER.debug("PID: error={}, integral={}, derivative={}, output={}, pathStep={}, trackerAdj={}", (Object)String.format("%.3f", error), (Object)String.format("%.3f", AutoTuner.pidState.integral), (Object)String.format("%.3f", derivative), (Object)String.format("%.3f", pidOutput), (Object)pathfindingStep, (Object)trackerAdjustment);
        return new ControlOutput(pathfindingStep, trackerAdjustment);
    }

    private static void applyPathfindingControl(WmbConfig cfg, int step) {
        if (step == 0 || cooldownTicksLeft.get() > 0) {
            return;
        }
        int currentInterval = cfg.pathfinding.minRecalcInterval;
        int newInterval = AutoTuner.clamp(currentInterval + step, cfg.tuning.minRecalcMin, cfg.tuning.minRecalcMax);
        if (newInterval != currentInterval) {
            cfg.pathfinding.minRecalcInterval = newInterval;
            cooldownTicksLeft.set(Math.max(0, cfg.tuning.cooldownTicks));
            LOGGER.info("AutoTuner: pathfinding.minRecalcInterval {} -> {} (step={}, smoothedMs={:.2f}, target={:.2f})", (Object)currentInterval, (Object)newInterval, (Object)step, (Object)tickEmaMs, (Object)cfg.tuning.targetTickMs);
        }
    }

    private static boolean hasConfigChanged(ConfigSnapshot current) {
        return !current.equals(lastConfig);
    }

    private static void resetForConfigChange(ConfigSnapshot newConfig) {
        lock.writeLock().lock();
        try {
            LOGGER.debug("Configuration changed, resetting AutoTuner state");
            ticksInWindow.set(0);
            nanosInWindow.set(0L);
            pidState.reset();
            tickEmaMs = 0.0;
            lastConfig = newConfig;
        }
        finally {
            lock.writeLock().unlock();
        }
    }

    private static void reset() {
        lock.writeLock().lock();
        try {
            ticksInWindow.set(0);
            nanosInWindow.set(0L);
            pidState.reset();
            tickEmaMs = 0.0;
            trackerIntervalAddend.set(0);
            lastConfig = null;
        }
        finally {
            lock.writeLock().unlock();
        }
    }

    private static double clamp(double value, double min, double max) {
        return Math.max(min, Math.min(max, value));
    }

    private static int clamp(int value, int min, int max) {
        return Math.max(min, Math.min(max, value));
    }

    private static class ConfigSnapshot {
        final int windowTicks;
        final double targetTickMs;
        final int minRecalcMin;
        final int minRecalcMax;
        final double kp;
        final double ki;
        final double kd;
        final int maxStepPerWindow;
        final double integralMaxAbs;
        final int cooldownTicks;

        ConfigSnapshot(WmbConfig cfg) {
            this.windowTicks = cfg.tuning.windowTicks;
            this.targetTickMs = cfg.tuning.targetTickMs;
            this.minRecalcMin = cfg.tuning.minRecalcMin;
            this.minRecalcMax = cfg.tuning.minRecalcMax;
            this.kp = cfg.tuning.kp;
            this.ki = cfg.tuning.ki;
            this.kd = cfg.tuning.kd;
            this.maxStepPerWindow = cfg.tuning.maxStepPerWindow;
            this.integralMaxAbs = cfg.tuning.integralMaxAbs;
            this.cooldownTicks = cfg.tuning.cooldownTicks;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof ConfigSnapshot)) {
                return false;
            }
            ConfigSnapshot other = (ConfigSnapshot)obj;
            return this.windowTicks == other.windowTicks && Double.compare(this.targetTickMs, other.targetTickMs) == 0 && this.minRecalcMin == other.minRecalcMin && this.minRecalcMax == other.minRecalcMax && Double.compare(this.kp, other.kp) == 0 && Double.compare(this.ki, other.ki) == 0 && Double.compare(this.kd, other.kd) == 0 && this.maxStepPerWindow == other.maxStepPerWindow && Double.compare(this.integralMaxAbs, other.integralMaxAbs) == 0 && this.cooldownTicks == other.cooldownTicks;
        }

        public int hashCode() {
            return Objects.hash(this.windowTicks, this.targetTickMs, this.minRecalcMin, this.minRecalcMax, this.kp, this.ki, this.kd, this.maxStepPerWindow, this.integralMaxAbs, this.cooldownTicks);
        }
    }

    public static class PerformanceMetrics {
        public final double smoothedTickMs;
        public final double pidIntegral;
        public final double pidLastError;
        public final int trackerAddend;
        public final int cooldownRemaining;

        PerformanceMetrics(double smoothedTickMs, double pidIntegral, double pidLastError, int trackerAddend, int cooldownRemaining) {
            this.smoothedTickMs = smoothedTickMs;
            this.pidIntegral = pidIntegral;
            this.pidLastError = pidLastError;
            this.trackerAddend = trackerAddend;
            this.cooldownRemaining = cooldownRemaining;
        }

        public String toString() {
            return String.format("PerformanceMetrics{tickMs=%.2f, integral=%.3f, lastError=%.3f, trackerAddend=%d, cooldown=%d}", this.smoothedTickMs, this.pidIntegral, this.pidLastError, this.trackerAddend, this.cooldownRemaining);
        }
    }

    private static class PidState {
        double integral = 0.0;
        double lastError = 0.0;
        boolean hasLastError = false;

        private PidState() {
        }

        void reset() {
            this.integral = 0.0;
            this.lastError = 0.0;
            this.hasLastError = false;
        }
    }

    private static class ControlOutput {
        final int pathfindingStep;
        final int trackerAdjustment;

        ControlOutput(int pathfindingStep, int trackerAdjustment) {
            this.pathfindingStep = pathfindingStep;
            this.trackerAdjustment = trackerAdjustment;
        }
    }
}

