package com.bawnorton.neruina.command;

import com.bawnorton.configurable.api.ConfigurableApi;
import com.bawnorton.neruina.Neruina;
import com.bawnorton.neruina.config.Config;
import com.bawnorton.neruina.extend.Errorable;
import com.bawnorton.neruina.handler.MessageHandler;
import com.bawnorton.neruina.handler.TickHandler;
import com.bawnorton.neruina.report.GithubAuthManager;
import com.bawnorton.neruina.report.ReportStatus;
import com.bawnorton.neruina.util.ErroredType;
import com.bawnorton.neruina.util.TickingEntry;
import com.bawnorton.neruina.version.Texter;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import java.util.Collection;
import java.util.UUID;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2186;
import net.minecraft.class_2248;
import net.minecraft.class_2262;
import net.minecraft.class_2338;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_2568;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2818;
import net.minecraft.class_5242;

public final class NeruinaCommandHandler {
    private static final MessageHandler messageHandler = Neruina.getInstance().getMessageHandler();
    
    public static void register(CommandDispatcher<class_2168> dispatcher) {
        dispatcher.register(class_2170.method_9247("neruina")
                .requires(source -> source.method_9259(Config.minPermissionLevelForCommands))
                .then(class_2170.method_9247("reload")
                        .executes(context -> {
                            ConfigurableApi.loadFromDisk();
                            sendSuccess(context, messageHandler.formatText("commands.neruina.reload"));
                            return 1;
                        })
                )
                .then(class_2170.method_9247("resume")
                        .then(class_2170.method_9247("entity")
                                .then(class_2170.method_9244("entity", class_2186.method_9309())
                                        .executes(NeruinaCommandHandler::executeResumeEntity)
                                )
                        )
                        .then(class_2170.method_9247("block_entity")
                                .then(class_2170.method_9244("pos", class_2262.method_9698())
                                        .executes(NeruinaCommandHandler::executeResumeBlockEntity)
                                )
                        )
                        .then(class_2170.method_9247("block_state")
                                .then(class_2170.method_9244("pos", class_2262.method_9698())
                                        .executes(NeruinaCommandHandler::executeResumeBlockState)
                                )
                        )
                        .then(class_2170.method_9247("item_stack")
                                .then(class_2170.method_9244("player", class_2186.method_9309())
                                        .executes(NeruinaCommandHandler::executeResumeHeldItem)
                                )
                        )
                )
                .then(class_2170.method_9247("kill")
                        .then(class_2170.method_9244("entity", class_2186.method_9306())
                                .executes(NeruinaCommandHandler::executeKill)
                        )
                )
                .then(class_2170.method_9247("report")
                        .then(class_2170.method_9244("id", class_5242.method_27643())
                                .executes(NeruinaCommandHandler::executeReport)
                        )
//                        .then(Commands.literal("test")
//                                .executes(NeruinaCommandHandler::executeTestReport)
//                        )
                )
                .then(class_2170.method_9247("cancel_login")
                        .executes(NeruinaCommandHandler::executeCancelLogin)
                )
                .then(class_2170.method_9247("id")
                        .then(class_2170.method_9244("entity", class_2186.method_9309())
                                .executes(NeruinaCommandHandler::executeIdEntity)
                        )
                        .then(class_2170.method_9244("pos", class_2262.method_9698())
                                .executes(NeruinaCommandHandler::executeIdBlock)
                        )
                )
                .then(class_2170.method_9247("info")
                        .then(class_2170.method_9244("id", class_5242.method_27643())
                                .executes(NeruinaCommandHandler::executeInfo)
                        )
                )
                .then(class_2170.method_9247("clear_tracked")
                        .executes(NeruinaCommandHandler::executeClear)
                )
                .then(class_2170.method_9247("show_suspended")
                        .executes(NeruinaCommandHandler::executeShowSuspended)
                )
        );
    }

    private static int executeResumeEntity(CommandContext<class_2168> context) {
        try {
            class_1297 entity = class_2186.method_9313(context, "entity");
            if (!((Errorable) entity).neruina$isErrored()) {
                context.getSource().method_9213(messageHandler.formatText(
                        "commands.neruina.resume.entity.not_errored",
                        entity.method_5477().getString()
                ));
                return 0;
            }
            Neruina.getInstance().getTickHandler().removeErrored(entity);
            sendSuccess(context, messageHandler.formatText(
                    "commands.neruina.resume.entity",
                    entity.method_5477().getString()
            ));
        } catch (CommandSyntaxException ignored) {
            context.getSource().method_9213(messageHandler.formatText("commands.neruina.resume.entity.not_found"));
        }
        return 1;
    }

    private static int executeResumeBlockEntity(CommandContext<class_2168> context) throws CommandSyntaxException {
        class_2338 pos = class_2262.method_48299(context, "pos");
        class_2586 blockEntity = context.getSource()
                .method_9225()
                .method_8321(pos);
        if (blockEntity == null) {
            context.getSource().method_9213(messageHandler.formatText(
                    "commands.neruina.resume.block_entity.not_found",
                    messageHandler.posAsNums(pos)
            ));
            return 0;
        }
        class_1937 level = context.getSource().method_9225();
        class_2818 levelChunk = level.method_8500(pos);
        class_2680 state = levelChunk.method_8320(pos);
        class_2248 block = state.method_26204();
        String name = block.method_9518().getString();
        if (!((Errorable) blockEntity).neruina$isErrored()) {
            context.getSource().method_9213(messageHandler.formatText(
                    "commands.neruina.resume.block_entity.not_errored",
                    name,
                    messageHandler.posAsNums(pos)
            ));
            return 0;
        }
        Neruina.getInstance().getTickHandler().removeErrored(blockEntity);
        levelChunk.method_12216(blockEntity);
        sendSuccess(context, messageHandler.formatText(
                "commands.neruina.resume.block_entity",
                name,
                messageHandler.posAsNums(pos)
        ));
        return 1;
    }

    private static int executeResumeBlockState(CommandContext<class_2168> context) throws CommandSyntaxException {
        class_2338 pos = class_2262.method_48299(context, "pos");
        class_2680 blockState = context.getSource().method_9225().method_8320(pos);
        String name = blockState.method_26204().method_9518().getString();
        if (!(Neruina.getInstance().getTickHandler().isErrored(blockState, pos))) {
            context.getSource().method_9213(messageHandler.formatText(
                    "commands.neruina.resume.block_state.not_errored",
                    name,
                    messageHandler.posAsNums(pos)
            ));
            return 0;
        }
        Neruina.getInstance().getTickHandler().removeErrored(blockState, pos);
        sendSuccess(context, messageHandler.formatText(
                "commands.neruina.resume.block_state",
                name,
                messageHandler.posAsNums(pos)
        ));
        return 1;
    }

    private static int executeResumeHeldItem(CommandContext<class_2168> context) {
        try {
            class_1657 player;
            try {
                player = class_2186.method_9315(context, "player");
            } catch (CommandSyntaxException ignored) {
                player = context.getSource().method_9207();
            }

            class_1799 stack = player.method_5998(player.method_6058());
            if(!((Errorable) (Object) stack).neruina$isErrored()) {
                context.getSource().method_9213(messageHandler.formatText(
                        "commands.neruina.resume.item_stack.not_errored",
                        player.method_5477().getString(),
                        stack.method_7964().getString()
                ));
                return 0;
            }
            Neruina.getInstance().getTickHandler().removeErrored(stack);
            sendSuccess(context, messageHandler.formatText(
                    "commands.neruina.resume.item_stack",
                    player.method_5477().getString(),
                    stack.method_7964().getString()
            ));
        } catch (CommandSyntaxException ignored) {
            context.getSource().method_9213(messageHandler.formatText("commands.neruina.resume.entity.not_found"));
        }
        return 1;
    }

    private static int executeKill(CommandContext<class_2168> context) {
        try {
            Collection<? extends class_1297> entities = class_2186.method_9317(context, "entity");
            if (entities.size() == 1) {
                class_1297 entity = entities.iterator().next();
                String name = entity.method_5477().getString();
                if (!((Errorable) entity).neruina$isErrored()) {
                    context.getSource().method_9213(messageHandler.formatText(
                            "commands.neruina.kill.not_errored",
                            name
                    ));
                    return 0;
                }
                Neruina.getInstance().getTickHandler()
                        .killEntity(entity, messageHandler.formatText("commands.neruina.kill", name));
            } else {
                int killed = 0;
                for (class_1297 entity : entities) {
                    if (!((Errorable) entity).neruina$isErrored()) {
                        continue;
                    }
                    Neruina.getInstance().getTickHandler().killEntity(entity, null);
                    killed++;
                }

                sendSuccess(context, getKilledResultMessage(entities, killed));
            }
        } catch (CommandSyntaxException ignored) {
            context.getSource().method_9213(messageHandler.formatText("commands.neruina.kill.not_found"));
        }
        return 1;
    }

    private static class_2561 getKilledResultMessage(Collection<? extends class_1297> entities, int killed) {
        int missed = entities.size() - killed;
        class_2561 message;
        if (killed == 1 && missed == 1) {
            message = messageHandler.formatText("commands.neruina.kill.multiple.singular_singular");
        } else if (killed == 1) {
            message = messageHandler.formatText("commands.neruina.kill.multiple.singular_plural", missed);
        } else if (missed == 1) {
            message = messageHandler.formatText("commands.neruina.kill.multiple.plural_singular", killed);
        } else {
            message = messageHandler.formatText("commands.neruina.kill.multiple", killed, missed);
        }
        return message;
    }

    private static int executeReport(CommandContext<class_2168> context) throws CommandSyntaxException {
        UUID id = class_5242.method_27645(context, "id");
        TickingEntry entry = Neruina.getInstance().getTickHandler().getTickingEntry(id);
        if (entry == null) {
            context.getSource().method_9213(messageHandler.formatText(
                    "commands.neruina.report.not_found",
                    id.toString()
            ));
            return 0;
        }

        try {
            Neruina.getInstance().getAutoReportHandler()
                    .createReports(context.getSource().method_9207(), entry)
                    .thenAccept(result -> {
                ReportStatus.Code reportCode = result.code();
                switch (reportCode) {
                    case SUCCESS -> sendSuccess(
                            context,
                            Texter.concatDelimited(
                                    Texter.LINE_BREAK,
                                    Texter.format(
                                            Texter.translatable("commands.neruina.report.success")
                                    ),
                                    messageHandler.generateOpenReportAction(result.message())
                            )
                    );
                    case ALREADY_EXISTS -> context.getSource().method_9213(
                            messageHandler.formatText("commands.neruina.report.already_exists")
                    );
                    case FAILURE -> context.getSource().method_9213(
                            messageHandler.formatText("commands.neruina.report.failure")
                    );
                    case TIMEOUT -> context.getSource().method_9213(
                            messageHandler.formatText("commands.neruina.report.timeout")
                    );
                    case ABORTED -> context.getSource().method_9213(
                            messageHandler.formatText("commands.neruina.report.aborted")
                    );
                    case IN_PROGRESS -> context.getSource().method_9213(
                            messageHandler.formatText("commands.neruina.report.in_progress")
                    );
                    case TESTING -> {}
                }
            });
        } catch (Throwable e) {
            context.getSource().method_9213(messageHandler.formatText("commands.neruina.report.failure"));
            Neruina.LOGGER.error("Failed to create report", e);
        }
        return 1;
    }

    private static int executeTestReport(CommandContext<class_2168> context) {
        try {
            if(!context.getSource().method_43737()) {
                return 0;
            }
            class_1657 player = context.getSource().method_9207();
						//? if >=1.21.10 {
	          if (!player.method_7334().id().equals(UUID.fromString("17c06cab-bf05-4ade-a8d6-ed14aaf70545"))) {
						//?} else {
            /*if(!player.getGameProfile().getId().equals(UUID.fromString("17c06cab-bf05-4ade-a8d6-ed14aaf70545"))) {
						*///?}
                return 0;
            }
            Neruina.getInstance().getAutoReportHandler().testReporting(context.getSource().method_9207());
            context.getSource().method_45068(messageHandler.formatText("commands.neruina.report.test.pass"));
        } catch (Exception e) {
            context.getSource().method_45068(messageHandler.formatText("commands.neruina.report.test.fail"));
            Neruina.LOGGER.error("Failed", e);
        }
        return 1;
    }

    private static int executeCancelLogin(CommandContext<class_2168> context) throws CommandSyntaxException {
        boolean wasLoggingIn = GithubAuthManager.cancelLogin(context.getSource().method_9207());
        if (!wasLoggingIn) {
            context.getSource().method_9213(messageHandler.formatText("commands.neruina.cancel.not_logging_in"));
            return 0;
        }
        return 1;
    }

    private static int executeIdBlock(CommandContext<class_2168> context) throws CommandSyntaxException {
        class_2338 pos = class_2262.method_48299(context, "pos");
        class_2586 blockEntity = context.getSource().method_9225().method_8321(pos);
        Neruina.getInstance().getTickHandler().getTickingEntryId(blockEntity).ifPresentOrElse(uuid -> sendSuccess(
                context,
                Texter.withStyle(
                        messageHandler.formatText("commands.neruina.id", uuid.toString()),
                        style -> style.method_10958(Texter.clickEvent(class_2558.class_2559.field_21462, uuid.toString()))
                                .method_10949(Texter.hoverEvent(class_2568.class_5247.field_24342, Texter.translatable("commands.neruina.id.tooltip")))
                )
        ), () -> context.getSource().method_9213(
                messageHandler.formatText(
                        "commands.neruina.id.block.not_errored",
                        context.getSource().method_9225().method_8320(pos).method_26204().method_9518().getString(),
                        messageHandler.posAsNums(pos)
                )
        ));
        return 1;
    }

    private static int executeIdEntity(CommandContext<class_2168> context) throws CommandSyntaxException {
        class_1297 entity = class_2186.method_9313(context, "entity");
        TickHandler tickHandler = Neruina.getInstance().getTickHandler();
        if(entity instanceof class_1657 player) {
            class_1799 stack = player.method_5998(player.method_6058());
            tickHandler.getTickingEntryId(stack).ifPresentOrElse(uuid -> sendSuccess(
                    context,
                    Texter.withStyle(
                            messageHandler.formatText("commands.neruina.id", uuid.toString()),
                            style -> style.method_10958(Texter.clickEvent(class_2558.class_2559.field_21462, uuid.toString()))
                                    .method_10949(Texter.hoverEvent(class_2568.class_5247.field_24342, Texter.translatable("commands.neruina.id.tooltip"))
                            )
                    )
            ), () -> context.getSource().method_9213(
                    messageHandler.formatText(
                            "commands.neruina.id.item_stack.not_errored",
                            player.method_5477().getString(),
                            stack.method_7964().getString()
                    )
            ));
        } else {
            tickHandler.getTickingEntryId(entity).ifPresentOrElse(uuid -> sendSuccess(
                    context,
                    Texter.withStyle(
                            messageHandler.formatText("commands.neruina.id", uuid.toString()),
                            style -> style.method_10958(Texter.clickEvent(class_2558.class_2559.field_21462, uuid.toString()))
                                    .method_10949(Texter.hoverEvent(class_2568.class_5247.field_24342, Texter.translatable("commands.neruina.id.tooltip"))
                            )
                    )
            ), () -> context.getSource().method_9213(
                    messageHandler.formatText(
                            "commands.neruina.id.entity.not_errored",
                            entity.method_5477().getString()
                    )
            ));
        }
        return 1;
    }

    private static int executeInfo(CommandContext<class_2168> context) throws CommandSyntaxException {
        UUID id = UuidArgument.getUuid(context, "id");
        TickingEntry entry = Neruina.getInstance().getTickHandler().getTickingEntry(id);
        if (entry == null) {
            context.getSource().sendFailure(messageHandler.formatText(
                    "commands.neruina.info.not_found",
                    id.toString()
            ));
            return 0;
        }
        Object cause = entry.getCause();
        Player player = context.getSource().getPlayerOrException();
        switch (cause) {
            case Entity entity -> sendSuccess(
                    context,
                    Texter.pad(
                            Texter.concatDelimited(
                                    Texter.LINE_BREAK,
                                    Texter.format(Texter.translatable(
                                                    "commands.neruina.info.entity",
                                                    entry.getCauseName(),
                                                    messageHandler.posAsNums(entry.pos())
                                            )
                                    ),
                                    messageHandler.generateEntityActions(player, entity),
                                    messageHandler.generateResourceActions(player, entry)
                            )
                    )
            );
            case BlockEntity ignored -> sendSuccess(
                    context,
                    Texter.pad(
                            Texter.concatDelimited(
                                    Texter.LINE_BREAK,
                                    Texter.format(Texter.translatable(
                                            "commands.neruina.info.block_entity",
                                            entry.getCauseName(),
                                            messageHandler.posAsNums(entry.pos())
                                    )),
                                    messageHandler.generateHandlingActions(player, ErroredType.BLOCK_ENTITY, entry.dimension(), entry.pos()),
                                    messageHandler.generateResourceActions(player, entry)
                            )
                    )
            );
            case ItemStack ignored -> sendSuccess(
                    context,
                    Texter.pad(
                            Texter.concatDelimited(
                                    Texter.LINE_BREAK,
                                    Texter.format(Texter.translatable(
                                            "commands.neruina.info.item_stack",
                                            entry.getCauseName()
                                    )),
                                    messageHandler.generateResumeAction(player, ErroredType.ITEM_STACK, entry.uuid().toString()),
                                    messageHandler.generateResourceActions(player, entry)
                            )
                    )
            );
            case Block ignored -> sendSuccess(
                    context,
                    Texter.pad(
                            Texter.concatDelimited(
                                    Texter.LINE_BREAK,
                                    Texter.format(Texter.translatable(
                                            "commands.neruina.info.block_state",
                                            entry.getCauseName(),
                                            messageHandler.posAsNums(entry.pos())
                                    )),
                                    messageHandler.generateHandlingActions(player, ErroredType.BLOCK_STATE, entry.dimension(), entry.pos()),
                                    messageHandler.generateResourceActions(player, entry)
                            )
                    )
            );
            case null, default -> sendSuccess(
                    context,
                    Texter.pad(
                            Texter.concatDelimited(
                                    Texter.LINE_BREAK,
                                    Texter.format(Texter.translatable(
                                                    "commands.neruina.info.null_cause",
                                                    entry.getCauseName(),
                                                    messageHandler.posAsNums(entry.pos())
                                            )
                                    ),
                                    messageHandler.generateTeleportAction(player, ErroredType.UNKNOWN, entry.dimension(), entry.pos()),
                                    messageHandler.generateResourceActions(player, entry)
                            )
                    )
            );
        }
        return 1;
    }

    private static int executeClear(CommandContext<class_2168> context) {
        int count = Neruina.getInstance().getTickHandler().clearTracked();
        if (count == 0) {
            context.getSource().method_9213(messageHandler.formatText("commands.neruina.clear.none"));
            return 0;
        }
        sendSuccess(context, messageHandler.formatText("commands.neruina.clear", count));
        return 1;
    }

    private static int executeShowSuspended(CommandContext<class_2168> context) throws CommandSyntaxException {
        int count = Neruina.getInstance().getTickHandler().getTickingEntries().size();
        if (count == 0) {
            context.getSource().method_9213(messageHandler.formatText("commands.neruina.show_suspended.none"));
            return 0;
        }
        class_1657 player = context.getSource().method_9207();
        class_2561 message = messageHandler.generateSuspendedInfo(player);
        sendSuccess(context, message);
        return 1;
    }

    private static void sendSuccess(CommandContext<class_2168> context, class_2561 text) {
        context.getSource().method_9226(() -> text, true);
    }
}