package com.bawnorton.neruina.handler;

import com.bawnorton.neruina.Neruina;
import com.bawnorton.neruina.config.Config;
import com.bawnorton.neruina.exception.TickingException;
import com.bawnorton.neruina.extend.Errorable;
import com.bawnorton.neruina.handler.client.ClientTickHandler;
import com.bawnorton.neruina.mixin.accessor.LevelChunkAccessor;
import com.bawnorton.neruina.platform.Platform;
import com.bawnorton.neruina.util.ErroredType;
import com.bawnorton.neruina.util.MultiSetMap;
import com.bawnorton.neruina.util.TickingEntry;
import com.bawnorton.neruina.version.Texter;
import com.llamalad7.mixinextras.injector.wrapoperation.Operation;
import net.minecraft.class_128;
import net.minecraft.class_129;
import net.minecraft.class_1297;
import net.minecraft.class_1304;
import net.minecraft.class_148;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2818;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_5558;
import net.minecraft.class_7923;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;


public final class TickHandler {
	private final List<TickingEntry> recentErrors = new ArrayList<>();
	private final Map<UUID, TickingEntry> tickingEntries = new HashMap<>();
	private final MultiSetMap<class_2680, class_2338> erroredBlockStates = new MultiSetMap<>();
	private int stopwatch = 0;

	public void tick() {
		stopwatch++;
		if (stopwatch >= 600) {
			if (!recentErrors.isEmpty()) {
				recentErrors.removeFirst();
			}
			stopwatch = 0;
		}
	}

	public void init() {
		tickingEntries.clear();
		recentErrors.clear();
	}

	@SuppressWarnings("unused")
	public void safelyTickItemStack(class_1799 instance, class_1937 level, class_1297 entity, class_1304 slot, int slotIndex, Operation<Void> original) {
		try {
			if (isErrored(instance)) {
				return;
			}
			original.call(instance, level, entity, slot);
		} catch (Throwable e) {
			handleTickingItemStack(e, instance, !level.method_8608(), (class_1657) entity, slotIndex);
		}
	}

	@SuppressWarnings("unused")
	public void safelyTickItemStack(class_1799 instance, class_1937 level, class_1297 entity, int slot, boolean selected, Operation<Void> original) {
		try {
			if (isErrored(instance)) {
				return;
			}
			original.call(instance, level, entity, slot, selected);
		} catch (Throwable e) {
			handleTickingItemStack(e, instance, !level.method_8608(), (class_1657) entity, slot);
		}
	}

	@SuppressWarnings("unused")
	public void safelyTickItemStack(class_1799 instance, class_1937 level, class_1657 player, int slot, int selected, Operation<Void> original) {
		try {
			if (isErrored(instance)) {
				return;
			}
			original.call(instance, level, player, slot, selected);
		} catch (Throwable e) {
			handleTickingItemStack(e, instance, !level.method_8608(), player, slot);
		}
	}

	public void safelyTickEntities(Consumer<Object> instance, class_1297 entity, Operation<Void> original) {
		try {
			if (isErrored(entity)) {
				handleErroredEntity(entity);
				return;
			}
			original.call(instance, entity);
		} catch (TickingException e) {
			throw e;
		} catch (Throwable e) {
			preHandleTickingEntity(entity, e);
		}
	}

	public <T extends class_1297> void safelyTickEntities(Consumer<T> consumer, T entity, class_1937 level, Object random, Operation<Void> original) {
		try {
			if (isErrored(entity)) {
				handleErroredEntity(entity);
				return;
			}
			original.call(consumer, entity, level, random);
		} catch (TickingException e) {
			throw e;
		} catch (Throwable e) {
			preHandleTickingEntity(entity, e);
		}
	}

	public void safelyTickPlayer(class_3222 instance, Operation<Void> original) {
		try {
			original.call(instance);
		} catch (Throwable e) {
			if (!Config.handleTickingPlayers) {
				throw TickingException.notHandled("handle_ticking_players", e);
			}
			handleTickingPlayer(instance, e);
		}
	}

	public void safelyTickBlockState(class_2680 instance, class_3218 level, class_2338 pos, Object random, Operation<Void> original) {
		try {
			if (isErrored(instance, pos)) {
				return;
			}
			original.call(instance, level, pos, random);
		} catch (Throwable e) {
			if (!Config.handleTickingBlockStates) {
				throw TickingException.notHandled("handle_ticking_block_states", e);
			}
			class_2960 blockId = class_7923.field_41175.method_10221(instance.method_26204());
			class_2960 owningBlacklist = getBlacklistFor(ErroredType.BLOCK_STATE, blockId);
			if (owningBlacklist != null) {
				throw TickingException.blacklisted(owningBlacklist, blockId, e);
			}
			MessageHandler messageHandler = Neruina.getInstance().getMessageHandler();
			class_2561 message = messageHandler.formatText(
					"neruina.ticking.block_state",
					instance.method_26204().method_9518().getString(),
					messageHandler.posAsNums(pos)
			);
			Neruina.LOGGER.warn("Neruina Caught An Exception, see below for cause", e);
			addErrored(instance, pos);
			TickingEntry tickingEntry = new TickingEntry(instance, true, level.method_27983(), pos, e);
			trackError(tickingEntry);
			messageHandler.broadcastToPlayers(
					level.method_8503(),
					message,
					forPlayer -> Texter.concatDelimited(
							Texter.LINE_BREAK,
							messageHandler.generateHandlingActions(forPlayer, ErroredType.BLOCK_STATE, level.method_27983(), pos),
							messageHandler.generateResourceActions(forPlayer, tickingEntry)
					)
			);
		}
	}

	public void safelyTickBlockEntity(class_5558<? extends class_2586> instance, class_1937 level, class_2338 pos, class_2680 state, class_2586 blockEntity, Operation<Void> original) {
		try {
			if (isErrored(blockEntity)) {
				if (level.method_8608()) {
					return;
				}

				class_2818 chunk = level.method_8500(pos);
				((LevelChunkAccessor) chunk).neruina$removeBlockEntityTicker(pos);
				return;
			}
			original.call(instance, level, pos, state, blockEntity);
		} catch (Throwable e) {
			if (!Config.handleTickingBlockEntities) {
				throw TickingException.notHandled("handle_ticking_block_entities", e);
			}
			class_2960 blockEntityId = class_7923.field_41181.method_10221(blockEntity.method_11017());
			class_2960 owningBlacklist = getBlacklistFor(ErroredType.BLOCK_ENTITY, blockEntityId);
			if (owningBlacklist != null) {
				throw TickingException.blacklisted(owningBlacklist, blockEntityId, e);
			}
			MessageHandler messageHandler = Neruina.getInstance().getMessageHandler();
			class_2561 message = messageHandler.formatText(
					"neruina.ticking.block_entity",
					state.method_26204().method_9518().getString(),
					messageHandler.posAsNums(pos)
			);
			Neruina.LOGGER.warn("Neruina caught an exception, see below for cause", e);
			addErrored(blockEntity);
			if (!level.method_8608()) {
				TickingEntry tickingEntry = new TickingEntry(blockEntity, true, level.method_27983(), pos, e);
				trackError(blockEntity, tickingEntry);
				messageHandler.broadcastToPlayers(
						level.method_8503(),
						message,
						forPlayer -> Texter.concatDelimited(
								Texter.LINE_BREAK,
								messageHandler.generateHandlingActions(forPlayer, ErroredType.BLOCK_ENTITY, level.method_27983(), pos),
								messageHandler.generateResourceActions(forPlayer, tickingEntry)
						)
				);
			}
		}
	}

	private void preHandleTickingEntity(class_1297 entity, Throwable e) {
		if (!Config.handleTickingEntities) {
			throw TickingException.notHandled("handle_ticking_entities", e);
		}
		class_2960 entityId = class_7923.field_41177.method_10221(entity.method_5864());
		class_2960 owningBlacklist = getBlacklistFor(ErroredType.ENTITY, entityId);
		if (owningBlacklist != null) {
			throw TickingException.blacklisted(owningBlacklist, entityId, e);
		}
		handleTickingEntity(entity, e);
	}

	private void handleTickingItemStack(Throwable e, class_1799 instance, boolean isServer, class_1657 player, int slot) {
		if (!Config.handleTickingItemStacks) {
			throw TickingException.notHandled("handle_ticking_item_stacks", e);
		}
		class_2960 itemId = class_7923.field_41178.method_10221(instance.method_7909());
		class_2960 owningBlacklist = getBlacklistFor(ErroredType.ITEM_STACK, itemId);
		if (owningBlacklist != null) {
			throw TickingException.blacklisted(owningBlacklist, itemId, e);
		}
		Neruina.LOGGER.warn("Neruina caught an exception, see below for cause", e);
		addErrored(instance);
		if (isServer) {
			TickingEntry tickingEntry = new TickingEntry(instance, false, player.method_37908().method_27983(), player.method_23312(), e);
			trackError(instance, tickingEntry);
			MessageHandler messageHandler = Neruina.getInstance().getMessageHandler();
			messageHandler.sendToPlayer(
					player,
					Texter.translatable("neruina.ticking.item_stack", instance.method_7964().getString(), slot),
					messageHandler.generateResumeAction(player, ErroredType.ITEM_STACK, player.method_5845()),
					messageHandler.generateResourceActions(player, tickingEntry)
			);
		}
	}

	private void handleErroredEntity(class_1297 entity) {
		try {
			if (entity instanceof class_1657) {
				return;
			}
			if (entity.method_37908().method_8608()) {
				return;
			}

			entity.method_5670();
			if (Config.autoKillTickingEntities || !entity.method_5805()) {
				killEntity(entity, null);
			}
		} catch (Throwable e) {
			try {
				killEntity(entity, Neruina.getInstance().getMessageHandler().formatText("neruina.ticking.entity.suspend_failed", entity.method_5477().getString()));
			} catch (Throwable ex) {
				throw new TickingException("Exception occurred while handling errored entity", ex);
			}
		}
	}

	public void killEntity(class_1297 entity, @Nullable class_2561 withMessage) {
		//? if >1.21.1 {
		if (entity.method_37908() instanceof class_3218 serverWorld) {
			entity.method_5768(serverWorld);
		}
		//?} else {
		/*entity.kill();
		 *///?}
		entity.method_5650(class_1297.class_5529.field_26998); // Necessary for any living entity
		removeErrored(entity);
		if (withMessage != null) {
			Neruina.getInstance().getMessageHandler().broadcastToPlayers(entity.method_37908().method_8503(), withMessage);
		}
	}

	private void handleTickingEntity(class_1297 entity, Throwable e) {
		if (entity instanceof class_1657 player) {
			if (player instanceof class_3222 serverPlayer) {
				handleTickingPlayer(serverPlayer, e);
			} else {
				handleTickingClient(player, e);
			}
			return;
		}

		Neruina.LOGGER.warn("Neruina caught an exception, see below for cause", e);
		addErrored(entity);
		class_1937 level = entity.method_37908();
		if (!level.method_8608()) {
			class_2338 pos = entity.method_23312();
			TickingEntry tickingEntry = new TickingEntry(entity, true, level.method_27983(), pos, e);
			trackError(entity, tickingEntry);
			MessageHandler messageHandler = Neruina.getInstance().getMessageHandler();
			class_2561 message = messageHandler.formatText(
					"neruina.ticking.entity.%s".formatted(
							Config.autoKillTickingEntities
									? "killed" : "suspended"
					),
					entity.method_5477().getString(),
					messageHandler.posAsNums(pos)
			);
			messageHandler.broadcastToPlayers(
					entity.method_37908().method_8503(), message, forPlayer -> {
						class_2561 actions = messageHandler.generateResourceActions(forPlayer, tickingEntry);
						if (!Config.autoKillTickingEntities) {
							actions = Texter.concatDelimited(
									Texter.LINE_BREAK,
									messageHandler.generateEntityActions(forPlayer, entity),
									actions
							);
						}
						return actions;
					}
			);
		}
	}

	private void handleTickingPlayer(class_3222 player, Throwable e) {
		Neruina.LOGGER.warn("Neruina caught an exception, see below for cause", e);
		MinecraftServer server = player.method_37908().method_8503();
		String name = player.method_5476().getString();
		MessageHandler messageHandler = Neruina.getInstance().getMessageHandler();
		class_2561 message = messageHandler.formatText("neruina.ticking.player", name);
		TickingEntry tickingEntry = new TickingEntry(player, false, player.method_37908().method_27983(), player.method_23312(), e);
		trackError(tickingEntry);
		messageHandler.broadcastToPlayers(server, message, forPlayer -> messageHandler.generateResourceActions(forPlayer, tickingEntry));
		try {
			player.field_13987.method_52396(
					Texter.concat(
							Texter.translatable("neruina.kick.message"),
							Texter.translatable("neruina.kick.reason")
					)
			);
		} catch (NullPointerException ex) {
			Neruina.LOGGER.error("Neruina caught an exception on a player, but the player is not connected, this should not happen. Behaviour is undefined.", ex);
		}
	}

	private void handleTickingClient(class_1657 player, Throwable e) {
		if (player.method_37908().method_8608() || Platform.isClient()) {
			ClientTickHandler.handleTickingClient(player, e);
		} else {
			Neruina.LOGGER.error("Neruina caught an exception, but the player is not a server player, this should not happen. Behaviour is undefined.", e);
		}
	}

	private void trackError(TickingEntry entry) {
		trackError(null, entry);
	}

	private void trackError(Object object, TickingEntry entry) {
		if (object instanceof Errorable errorable) {
			trackError(errorable, entry);
		} else if (object == null) {
			trackError(null, entry);
		}
	}

	private void trackError(@Nullable Errorable errorable, TickingEntry entry) {
		recentErrors.add(entry);
		addTickingEntry(entry);
		if (errorable != null) {
			errorable.neruina$setTickingEntryId(entry.uuid());
		}
		if (Config.tickingExceptionThreshold != -1 && recentErrors.size() >= Config.tickingExceptionThreshold) {
			class_128 report = class_128.method_560(
					new RuntimeException("Too Many Ticking Exceptions"),
					"Neruina has caught too many ticking exceptions in a short period of time, something is very wrong, see below for more info"
			);
			class_129 header = report.method_562("Information");
			header.method_578(
					"Threshold",
					"%d, set \"ticking_exception_threshold\" to -1 to disable.".formatted(
							Config.tickingExceptionThreshold
					)
			);
			header.method_578("Caught", recentErrors.size());
			String wiki = "https://github.com/Bawnorton/Neruina/wiki/Too-Many-Ticking-Exceptions";
			String lines = "=".repeat(wiki.length() + "Wiki".length() + 2);
			header.method_578("", lines);
			header.method_578("Wiki", wiki);
			header.method_578("", lines);
			for (int i = 0; i < recentErrors.size(); i++) {
				TickingEntry error = recentErrors.get(i);
				class_129 category = report.method_562("Ticking Exception #%s - (%s: %s)".formatted(i + 1, error.getCauseType(), error.getCauseName()));
				error.populate(category);
			}
			throw new class_148(report);
		}
	}

	public boolean isErrored(Object obj) {
		if (obj instanceof Errorable errorable) {
			return errorable.neruina$isErrored();
		}
		return false;
	}

	public boolean isErrored(class_2680 state, class_2338 pos) {
		return erroredBlockStates.contains(state, pos);
	}

	private void addErrored(Object obj) {
		if (obj instanceof Errorable errorable) {
			errorable.neruina$setErrored();
		}
	}

	private void addErrored(class_2680 state, class_2338 pos) {
		erroredBlockStates.put(state, pos);
	}

	public void removeErrored(Object obj) {
		if (obj instanceof Errorable errorable) {
			errorable.neruina$clearErrored();
			tickingEntries.remove(errorable.neruina$getTickingEntryId());
		}
	}

	public void removeErrored(class_2680 state, class_2338 pos) {
		erroredBlockStates.remove(state, pos);
	}

	public @Nullable TickingEntry getTickingEntry(UUID uuid) {
		return tickingEntries.get(uuid);
	}

	public Collection<TickingEntry> getTickingEntries() {
		return tickingEntries.values();
	}

	public void addTickingEntry(TickingEntry entry) {
		Object cause = entry.getCause();
		boolean shouldAdd = false;
		if (isErrored(cause)) {
			shouldAdd = true;
		} else if (cause instanceof class_2680 state) {
			shouldAdd = isErrored(state, entry.pos());
		}
		if (shouldAdd) {
			tickingEntries.put(entry.uuid(), entry);
		}
	}

	public void addTickingEntryUnsafe(TickingEntry entry) {
		tickingEntries.put(entry.uuid(), entry);
	}

	public Optional<UUID> getTickingEntryId(Object obj) {
		if (obj instanceof Errorable errorable && errorable.neruina$isErrored()) {
			return Optional.ofNullable(errorable.neruina$getTickingEntryId());
		}
		return Optional.empty();
	}

	public int clearTracked() {
		int size = tickingEntries.size();
		tickingEntries.clear();
		recentErrors.clear();
		return size;
	}

	private class_2960 getBlacklistFor(ErroredType type, class_2960 id) {
		return Neruina.getInstance().getBlacklistHandler().getBlacklistFor(type, id);
	}
}