package net.atif.buildnotes.gui.helper;

import com.mojang.blaze3d.systems.RenderSystem;
import net.atif.buildnotes.Buildnotes;
import net.minecraft.class_1159;
import net.minecraft.class_1162;
import net.minecraft.class_310;
import net.minecraft.class_4587;
import java.util.Stack;

public class ScissorStack {

    private static final Stack<Rect> stack = new Stack<>();

    private record Rect(int x, int y, int width, int height) {}

    /**
     * Pushes a new scissor rectangle onto the stack.
     * The new rectangle is intersected with the current one to create a nested effect.
     * Coordinates are transformed by the provided MatrixStack to get absolute screen positions.
     *
     * @param x The local X coordinate of the scissor box.
     * @param y The local Y coordinate of the scissor box.
     * @param width The width of the scissor box.
     * @param height The height of the scissor box.
     * @param matrices The current MatrixStack used for rendering the element.
     */

    public static void push(int x, int y, int width, int height, class_4587 matrices) {
        // Get the correct Matrix4f type from the MatrixStack
        class_1159 matrix = matrices.method_23760().method_23761();

        // Use Minecraft's Vector4f for transformations
        class_1162 corner1 = new class_1162(x, y, 0, 1);
        corner1.method_22674(matrix);

        class_1162 corner2 = new class_1162(x + width, y + height, 0, 1);
        corner2.method_22674(matrix);

        int transformedX = (int) Math.min(corner1.method_4953(), corner2.method_4953());
        int transformedY = (int) Math.min(corner1.method_4956(), corner2.method_4956());
        int transformedWidth = (int) (Math.max(corner1.method_4953(), corner2.method_4953()) - transformedX);
        int transformedHeight = (int) (Math.max(corner1.method_4956(), corner2.method_4956()) - transformedY);

        Rect newRect = new Rect(transformedX, transformedY, transformedWidth, transformedHeight);

        // If a scissor is already active, intersect the new one with it
        if (!stack.isEmpty()) {
            Rect parent = stack.peek();
            newRect = intersect(parent, newRect);
        }

        stack.push(newRect);
        applyScissor(newRect);
    }

    /**
     * Pops the current scissor rectangle from the stack and restores the previous one.
     */
    public static void pop() {
        if (stack.isEmpty()) {
            Buildnotes.LOGGER.warn("ScissorStack pop attempted on an empty stack.");
            return;
        }

        stack.pop();

        if (stack.isEmpty()) {
            RenderSystem.disableScissor();
        } else {
            applyScissor(stack.peek());
        }
    }

    private static Rect intersect(Rect r1, Rect r2) {
        int x = Math.max(r1.x, r2.x);
        int y = Math.max(r1.y, r2.y);
        int width = Math.min(r1.x + r1.width, r2.x + r2.width) - x;
        int height = Math.min(r1.y + r1.height, r2.y + r2.height) - y;

        return new Rect(x, y, Math.max(0, width), Math.max(0, height));
    }

    private static void applyScissor(Rect rect) {
        double scale = class_310.method_1551().method_22683().method_4495();
        // OpenGL's Y is from the bottom-left, so we must convert our top-left Y coordinate
        int windowHeight = class_310.method_1551().method_22683().method_4506();
        int scissorY = (int)(windowHeight - (rect.y + rect.height) * scale);

        RenderSystem.enableScissor(
                (int)(rect.x * scale),
                scissorY,
                (int)(rect.width * scale),
                (int)(rect.height * scale)
        );
    }
}