package net.atif.buildnotes.gui.screen;

import com.google.common.collect.Lists;
import net.atif.buildnotes.gui.helper.Colors;
import net.atif.buildnotes.gui.helper.ScissorStack;
import net.minecraft.class_2561;
import net.minecraft.class_364;
import net.minecraft.class_4068;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import net.minecraft.class_6379;
import java.util.List;

public abstract class ScrollableScreen extends BaseScreen {

    protected final List<class_364> scrollableWidgets = Lists.newArrayList();
    protected double scrollY = 0.0;
    protected int totalContentHeight = 0;

    // Scrolling mechanics
    private boolean isDraggingScrollbar = false;
    private static final int SCROLLBAR_WIDTH = 6;
    private static final int SCROLLBAR_PADDING = 2;

    protected ScrollableScreen(class_2561 title, class_437 parent) {
        super(title, parent);
    }

    protected abstract void initContent();
    protected abstract void renderContent(class_4587 matrices, int mouseX, int mouseY, float delta);
    protected abstract int getTopMargin();
    protected abstract int getBottomMargin();

    @Override
    protected void method_25426() {
        super.method_25426();
        this.scrollableWidgets.clear();
        this.scrollY = 0;
        initContent();
    }

    /**
     * Type-safe addSelectableChild helper for scrollable widgets.
     */
    protected <T extends class_364 & class_6379> void addScrollableWidget(T widget) {
        this.scrollableWidgets.add(widget);
        this.method_25429(widget);
    }

    @Override
    public void method_25394(class_4587 matrices, int mouseX, int mouseY, float delta) {
        this.method_25420(matrices);
        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();
        if (bottom <= top) return;

        ScissorStack.push(0, top, this.field_22789, bottom - top, new class_4587());

        matrices.method_22903();
        // 1. Move origin to the top of the scrollable area.
        matrices.method_22904(0.0, top, 0.0);
        // 2. Move origin up by the scroll amount.
        matrices.method_22904(0.0, -this.scrollY, 0.0);

        int adjustedMouseY = (int)(mouseY - top + this.scrollY);
        this.renderContent(matrices, mouseX, adjustedMouseY, delta);

        for (class_364 widget : this.scrollableWidgets) {
            if (widget instanceof class_4068 drawable) {
                drawable.method_25394(matrices, mouseX, adjustedMouseY, delta);
            }
        }

        matrices.method_22909();
        ScissorStack.pop();

        super.method_25394(matrices, mouseX, mouseY, delta);
        renderScrollbar(matrices);
    }

    private void renderScrollbar(class_4587 matrices) {
        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();
        int trackHeight = bottom - top;
        int maxScroll = getMaxScroll();

        if (maxScroll > 0) {
            int scrollbarX = this.field_22789 - SCROLLBAR_WIDTH - SCROLLBAR_PADDING;
            float thumbHeight = Math.max(20, (trackHeight / (float)this.totalContentHeight) * trackHeight);
            float thumbY = (float) ((this.scrollY / maxScroll) * (trackHeight - thumbHeight));

            int color = isDraggingScrollbar ? Colors.SCROLLBAR_THUMB_ACTIVE : Colors.SCROLLBAR_THUMB_INACTIVE;
            method_25294(matrices, scrollbarX, top + (int)thumbY, scrollbarX + SCROLLBAR_WIDTH, top + (int)(thumbY + thumbHeight), color);
        }
    }

    private int getMaxScroll() {
        int visibleHeight = this.field_22790 - getTopMargin() - getBottomMargin();
        return Math.max(0, this.totalContentHeight - visibleHeight);
    }

    @Override
    public boolean method_25401(double mouseX, double mouseY, double amount) {
        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();

        // First, offer the scroll event to child widgets under the cursor
        if (mouseY >= top && mouseY < bottom) {
            double adjustedMouseY = mouseY - top + this.scrollY;
            for (class_364 widget : this.scrollableWidgets) {
                if (widget.method_25405(mouseX, adjustedMouseY)) {
                    if (widget.method_25401(mouseX, adjustedMouseY, amount)) {
                        return true;
                    }
                }
            }
        }

        // If no child consumed it, scroll the main panel
        if (mouseY >= top && mouseY < bottom) {
            this.scrollY -= amount * 10;
            clampScroll();
            return true;
        }
        return false;
    }

    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        for (class_364 child : method_25396()) {
            if (!scrollableWidgets.contains(child)) {
                if (child.method_25402(mouseX, mouseY, button)) {
                    this.method_25395(child);
                    if (button == 0) {
                        this.method_25398(true);
                    }
                    return true;
                }
            }
        }

        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();
        int scrollbarX = this.field_22789 - SCROLLBAR_WIDTH - SCROLLBAR_PADDING;

        if (mouseX >= scrollbarX && mouseX < this.field_22789 && mouseY >= top && mouseY < bottom) {
            isDraggingScrollbar = true;
            return true;
        }

        if (mouseY >= top && mouseY < bottom) {
            double adjustedMouseY = mouseY - top + scrollY;
            for (class_364 widget : this.scrollableWidgets) {
                if (widget.method_25402(mouseX, adjustedMouseY, button)) {
                    this.method_25395(widget);
                    if (button == 0) this.method_25398(true);
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public boolean method_25403(double mouseX, double mouseY, int button, double deltaX, double deltaY) {
        // Ensure consistent coordinate space for the dragged element
        if (this.method_25399() != null && this.method_25397() && button == 0) {
            // If the dragged element is scrollable, pass it the adjusted coordinates
            if (this.scrollableWidgets.contains(this.method_25399())) {
                double adjustedMouseY = mouseY - getTopMargin() + this.scrollY;
                return this.method_25399().method_25403(mouseX, adjustedMouseY, button, deltaX, deltaY);
            }
            // Otherwise, pass normal coordinates (for fixed buttons, etc.)
            else {
                return this.method_25399().method_25403(mouseX, mouseY, button, deltaX, deltaY);
            }
        }

        if (isDraggingScrollbar) {
            int trackHeight = (this.field_22790 - getTopMargin() - getBottomMargin());
            if (trackHeight <= 0) return true;

            float thumbHeight = Math.max(20, (trackHeight / (float)this.totalContentHeight) * trackHeight);
            if (trackHeight - thumbHeight <= 0) return true;

            double scrollRatio = (double)getMaxScroll() / (trackHeight - thumbHeight);
            this.scrollY += deltaY * scrollRatio;
            clampScroll();
            return true;
        }
        return false;
    }

    @Override
    public boolean method_25406(double mouseX, double mouseY, int button) {
        // First, handle the screen's own scrollbar state.
        if (isDraggingScrollbar) {
            isDraggingScrollbar = false;
            return true;
        }

        if (this.method_25399() != null) {
            boolean handled;
            // Adjust coordinates for scrollable widgets before sending the event.
            if (this.scrollableWidgets.contains(this.method_25399())) {
                double adjustedMouseY = mouseY - getTopMargin() + this.scrollY;
                handled = this.method_25399().method_25406(mouseX, adjustedMouseY, button);
            } else {
                handled = this.method_25399().method_25406(mouseX, mouseY, button);
            }

            // After dispatching, reset the screen's global dragging state.
            this.method_25398(false);

            // If the focused element handled it, we are done.
            if (handled) {
                return true;
            }
        }

        // Fallback to default behavior if no specific element was focused.
        return super.method_25406(mouseX, mouseY, button);
    }

    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        if (this.method_25399() != null && this.method_25399().method_25404(keyCode, scanCode, modifiers)) {
            return true;
        }
        if (keyCode == 265) { // Up Arrow
            this.scrollY -= 10; clampScroll(); return true;
        } else if (keyCode == 264) { // Down Arrow
            this.scrollY += 10; clampScroll(); return true;
        }
        return super.method_25404(keyCode, scanCode, modifiers);
    }

    private void clampScroll() {
        if (this.scrollY < 0) this.scrollY = 0;
        int maxScroll = getMaxScroll();
        if (this.scrollY > maxScroll) this.scrollY = maxScroll;
    }
}