package net.atif.buildnotes.server.command;

import com.mojang.authlib.GameProfile;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.BoolArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.atif.buildnotes.Buildnotes;
import net.atif.buildnotes.server.PermissionEntry;
import net.fabricmc.fabric.api.command.v2.CommandRegistrationCallback;
import net.minecraft.class_124;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2191;
import net.minecraft.class_2561;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class BuildNotesCommands {

    public static void register() {
        CommandRegistrationCallback.EVENT.register(
                (dispatcher, registryAccess, environment) -> register(dispatcher)
        );
    }

    public static void register(CommandDispatcher<class_2168> dispatcher) {
        dispatcher.register(class_2170.method_9247("buildnotes")
                // Only server operators (or permission level 2+) can use these commands
                .requires(source -> source.method_9259(2))
                .then(class_2170.method_9247("allow")
                        .then(class_2170.method_9244("players", class_2191.method_9329())
                                .executes(BuildNotesCommands::allowPlayer)))
                .then(class_2170.method_9247("disallow")
                        .then(class_2170.method_9244("players", class_2191.method_9329())
                                .executes(BuildNotesCommands::disallowPlayer)))
                .then(class_2170.method_9247("list")
                        .executes(BuildNotesCommands::listPlayers))
                .then(class_2170.method_9247("allow_all")
                        .then(class_2170.method_9244("enabled", BoolArgumentType.bool())
                                .executes(BuildNotesCommands::allowAll)))
        );
    }

    private static int allowPlayer(CommandContext<class_2168> context) throws CommandSyntaxException {
        Collection<GameProfile> profiles = class_2191.method_9330(context, "players");
        class_2168 source = context.getSource();

        List<String> addedPlayers = new ArrayList<>();
        List<String> alreadyAllowedPlayers = new ArrayList<>();

        for (GameProfile profile : profiles) {
            if (Buildnotes.PERMISSION_MANAGER.addPlayer(profile)) {
                addedPlayers.add(profile.getName());
            } else {
                alreadyAllowedPlayers.add(profile.getName());
            }
        }

        // Report successfully added players
        if (!addedPlayers.isEmpty()) {
            source.method_9226(() -> class_2561.method_43470("Added ").method_10852(class_2561.method_43470(String.join(", ", addedPlayers)).method_27692(class_124.field_1060)).method_27693(" to the BuildNotes editor list."), true);
        }

        // Report players who were already on the list
        if (!alreadyAllowedPlayers.isEmpty()) {
            source.method_9213(class_2561.method_43470(String.join(", ", alreadyAllowedPlayers) + " were already on the list."));
        }

        return addedPlayers.size();
    }

    private static int disallowPlayer(CommandContext<class_2168> context) throws CommandSyntaxException {
        Collection<GameProfile> profiles = class_2191.method_9330(context, "players");
        class_2168 source = context.getSource();

        List<String> removedPlayers = new ArrayList<>();
        List<String> notOnListPlayers = new ArrayList<>();

        for (GameProfile profile : profiles) {
            if (Buildnotes.PERMISSION_MANAGER.removePlayer(profile)) {
                removedPlayers.add(profile.getName());
            } else {
                notOnListPlayers.add(profile.getName());
            }
        }

        // Report successfully removed players
        if (!removedPlayers.isEmpty()) {
            source.method_9226(() -> class_2561.method_43470("Removed ").method_10852(class_2561.method_43470(String.join(", ", removedPlayers)).method_27692(class_124.field_1061)).method_27693(" from the BuildNotes editor list."), true);
        }

        // Report players who were not on the list to begin with
        if (!notOnListPlayers.isEmpty()) {
            source.method_9213(class_2561.method_43470(String.join(", ", notOnListPlayers) + " were not on the list."));
        }

        return removedPlayers.size();
    }

    private static int listPlayers(CommandContext<class_2168> context) {
        class_2168 source = context.getSource();

        boolean allowAll = Buildnotes.PERMISSION_MANAGER.getAllowAll();
        if (allowAll) {
            source.method_9226(() -> class_2561.method_43470("Note: 'allow_all' is currently TRUE. All players can edit.").method_27692(class_124.field_1065), false);
        }

        Set<PermissionEntry> allowedPlayers = Buildnotes.PERMISSION_MANAGER.getAllowedPlayers();

        if (allowedPlayers.isEmpty()) {
            source.method_9226(() -> class_2561.method_43470("There are no players on the BuildNotes editor list."), false);
            return 1;
        }

        String playerNames = allowedPlayers.stream()
                .map(PermissionEntry::getName)
                .collect(Collectors.joining(", "));

        source.method_9226(() -> class_2561.method_43470("BuildNotes Editors: ").method_27692(class_124.field_1054).method_10852(class_2561.method_43470(playerNames)), false);
        return allowedPlayers.size();
    }

    private static int allowAll(CommandContext<class_2168> context) {
        boolean enabled = BoolArgumentType.getBool(context, "enabled");
        class_2168 source = context.getSource();

        Buildnotes.PERMISSION_MANAGER.setAllowAll(enabled);

        if (enabled) {
            source.method_9226(() -> class_2561.method_43470("All players can now edit BuildNotes.").method_27692(class_124.field_1060), true);
        } else {
            source.method_9226(() -> class_2561.method_43470("Only players on the list (and OPs) can edit BuildNotes.").method_27692(class_124.field_1061), true);
        }
        return 1;
    }
}