package net.atif.buildnotes.gui.screen;

import com.google.common.collect.Lists;
import net.atif.buildnotes.gui.helper.Colors;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_364;
import net.minecraft.class_4068;
import net.minecraft.class_437;
import net.minecraft.class_6379;
import net.minecraft.client.gui.*;
import org.joml.Matrix3x2fStack;

import java.util.List;

public abstract class ScrollableScreen extends BaseScreen {

    protected final List<class_364> scrollableWidgets = Lists.newArrayList();
    protected double scrollY = 0.0;
    protected int totalContentHeight = 0;

    // Scrolling mechanics
    private boolean isDraggingScrollbar = false;
    private static final int SCROLLBAR_WIDTH = 6;
    private static final int SCROLLBAR_PADDING = 2;

    protected ScrollableScreen(class_2561 title, class_437 parent) {
        super(title, parent);
    }

    protected abstract void initContent();
    protected abstract void renderContent(class_332 context, int mouseX, int mouseY, float delta);
    protected abstract int getTopMargin();
    protected abstract int getBottomMargin();

    @Override
    protected void method_25426() {
        super.method_25426();
        this.scrollableWidgets.clear();
        this.scrollY = 0;
        initContent();
    }

    /**
     * Type-safe addSelectableChild helper for scrollable widgets.
     */
    protected <T extends class_364 & class_6379> void addScrollableWidget(T widget) {
        this.scrollableWidgets.add(widget);
        this.method_25429(widget);
    }

    @Override
    public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
        super.method_25394(context, mouseX, mouseY, delta);
        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();
        if (bottom <= top) return;


        context.method_44379( 0, top, this.field_22789, bottom);

        Matrix3x2fStack matrices = context.method_51448();

        matrices.pushMatrix();
        // 1. Move origin to the top of the scrollable area.
        matrices.translate(0.0f, (float) top);
        // 2. Move origin up by the scroll amount.
        matrices.translate(0.0f, (float) -this.scrollY);

        int adjustedMouseY = (int)(mouseY - top + this.scrollY);
        this.renderContent(context, mouseX, adjustedMouseY, delta);

        for (class_364 widget : this.scrollableWidgets) {
            if (widget instanceof class_4068 drawable) {
                drawable.method_25394(context, mouseX, adjustedMouseY, delta);
            }
        }

        matrices.popMatrix();
        context.method_44380();

        renderScrollbar(context);
    }

    private void renderScrollbar(class_332 context) {
        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();
        int trackHeight = bottom - top;
        int maxScroll = getMaxScroll();

        if (maxScroll > 0) {
            int scrollbarX = this.field_22789 - SCROLLBAR_WIDTH - SCROLLBAR_PADDING;
            float thumbHeight = Math.max(20, (trackHeight / (float)this.totalContentHeight) * trackHeight);
            float thumbY = (float) ((this.scrollY / maxScroll) * (trackHeight - thumbHeight));

            int color = isDraggingScrollbar ? Colors.SCROLLBAR_THUMB_ACTIVE : Colors.SCROLLBAR_THUMB_INACTIVE;
            context.method_25294(scrollbarX, top + (int)thumbY, scrollbarX + SCROLLBAR_WIDTH, top + (int)(thumbY + thumbHeight), color);
        }
    }

    private int getMaxScroll() {
        int visibleHeight = this.field_22790 - getTopMargin() - getBottomMargin();
        return Math.max(0, this.totalContentHeight - visibleHeight);
    }

    @Override
    public boolean method_25401(double mouseX, double mouseY, double horizontalAmount, double verticalAmount) {
        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();

        // First, offer the scroll event to child widgets under the cursor
        if (mouseY >= top && mouseY < bottom) {
            double adjustedMouseY = mouseY - top + this.scrollY;
            for (class_364 widget : this.scrollableWidgets) {
                if (widget.method_25405(mouseX, adjustedMouseY)) {
                    if (widget.method_25401(mouseX, adjustedMouseY, horizontalAmount, verticalAmount)) {
                        return true;
                    }
                }
            }
        }

        // If no child consumed it, scroll the main panel
        if (mouseY >= top && mouseY < bottom) {
            this.scrollY -= verticalAmount * 10;
            clampScroll();
            return true;
        }
        return false;
    }

    @Override
    public boolean method_25402(class_11909 click, boolean doubled) {
        double mouseX = click.comp_4798();
        double mouseY = click.comp_4799();
        int button = click.method_74245();

        for (class_364 child : method_25396()) {
            if (!scrollableWidgets.contains(child)) {
                if (child.method_25402(click, doubled)) {
                    this.method_25395(child);
                    if (button == 0) {
                        this.method_25398(true);
                    }
                    return true;
                }
            }
        }

        int top = getTopMargin();
        int bottom = this.field_22790 - getBottomMargin();
        int scrollbarX = this.field_22789 - SCROLLBAR_WIDTH - SCROLLBAR_PADDING;

        if (mouseX >= scrollbarX && mouseX < this.field_22789 && mouseY >= top && mouseY < bottom) {
            isDraggingScrollbar = true;
            return true;
        }

        if (mouseY >= top && mouseY < bottom) {
            double adjustedMouseY = mouseY - top + scrollY;
            class_11909 adjustedClick = new class_11909(mouseX, adjustedMouseY, click.comp_4800());

            for (class_364 widget : this.scrollableWidgets) {
                if (widget.method_25402(adjustedClick, doubled)) {
                    this.method_25395(widget);
                    if (button == 0) this.method_25398(true);
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public boolean method_25403(class_11909 click, double offsetX, double offsetY) {
        double mouseX = click.comp_4798();
        double mouseY = click.comp_4799();
        int button = click.method_74245();

        if (this.method_25399() != null && this.method_25397() && button == 0) {
            // Create a new Click object with adjusted coordinates for scrollable widgets
            if (this.scrollableWidgets.contains(this.method_25399())) {
                double adjustedMouseY = mouseY - getTopMargin() + this.scrollY;
                class_11909 adjustedClick = new class_11909(mouseX, adjustedMouseY, click.comp_4800());
                return this.method_25399().method_25403(adjustedClick, offsetX, offsetY);
            } else {
                // Pass original click to fixed widgets
                return this.method_25399().method_25403(click, offsetX, offsetY);
            }
        }

        if (isDraggingScrollbar) {
            int trackHeight = (this.field_22790 - getTopMargin() - getBottomMargin());
            if (trackHeight <= 0) return true;

            float thumbHeight = Math.max(20, (trackHeight / (float)this.totalContentHeight) * trackHeight);
            if (trackHeight - thumbHeight <= 0) return true;

            double scrollRatio = (double)getMaxScroll() / (trackHeight - thumbHeight);
            this.scrollY += offsetY * scrollRatio;
            clampScroll();
            return true;
        }
        return false;
    }

    @Override
    public boolean method_25406(class_11909 click) {
        // First, handle the screen's own scrollbar state.
        if (isDraggingScrollbar) {
            isDraggingScrollbar = false;
            return true;
        }


        if (this.method_25399() != null) {
            boolean handled;
            // Adjust coordinates for scrollable widgets
            if (this.scrollableWidgets.contains(this.method_25399())) {
                double adjustedMouseY = click.comp_4799() - getTopMargin() + this.scrollY;
                class_11909 adjustedClick = new class_11909(click.comp_4798(), adjustedMouseY, click.comp_4800());
                handled = this.method_25399().method_25406(adjustedClick);
            } else {
                handled = this.method_25399().method_25406(click);
            }

            this.method_25398(false);
            if (handled) {
                return true;
            }
        }

        // Use the new method on the superclass
        return super.method_25406(click);
    }

    @Override
    public boolean method_25404(class_11908 input) {
        if (this.method_25399() != null && this.method_25399().method_25404(input)) {
            return true;
        }
        // Use GLFW constants for key codes
        if (input.method_74234()) {
            this.scrollY -= 10; clampScroll(); return true;
        } else if (input.method_74235()) {
            this.scrollY += 10; clampScroll(); return true;
        }
        return super.method_25404(input);
    }

    private void clampScroll() {
        if (this.scrollY < 0) this.scrollY = 0;
        int maxScroll = getMaxScroll();
        if (this.scrollY > maxScroll) this.scrollY = maxScroll;
    }
}