package net.atif.buildnotes.gui.screen;

import net.atif.buildnotes.Buildnotes;
import net.atif.buildnotes.client.ClientImageTransferManager;
import net.atif.buildnotes.client.ClientSession;
import net.atif.buildnotes.data.Build;
import net.atif.buildnotes.data.CustomField;
import net.atif.buildnotes.data.DataManager;
import net.atif.buildnotes.data.Scope;
import net.atif.buildnotes.gui.helper.BuildScreenLayouts;
import net.atif.buildnotes.gui.helper.Colors;
import net.atif.buildnotes.gui.helper.UIHelper;
import net.atif.buildnotes.gui.widget.DarkButtonWidget;
import net.atif.buildnotes.gui.widget.MultiLineTextFieldWidget;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_10799;
import net.minecraft.class_124;
import net.minecraft.class_1959;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_332;
import net.minecraft.class_342;
import net.minecraft.class_364;
import net.minecraft.class_437;
import net.minecraft.class_5321;
import net.minecraft.class_6880;
import org.lwjgl.glfw.GLFW;
import org.lwjgl.util.tinyfd.TinyFileDialogs;

import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.*;
import java.util.function.Consumer;

public class EditBuildScreen extends ScrollableScreen {

    private final Build build;

    // --- Image Gallery Fields ---
    private record ImageData(class_2960 textureId, int width, int height) {}
    private int currentImageIndex = 0;
    private final Map<String, ImageData> textureCache = new HashMap<>();
    private final Set<String> downloadingImages = new HashSet<>();

    private DarkButtonWidget prevImageButton;
    private DarkButtonWidget nextImageButton;

    // --- Standard Fields ---
    private MultiLineTextFieldWidget nameField, coordsField, dimensionField, descriptionField, designerField;
    private final Map<CustomField, MultiLineTextFieldWidget> customFieldWidgets = new LinkedHashMap<>();
    private MultiLineTextFieldWidget lastFocusedTextField;

    public EditBuildScreen(class_437 parent, Build build) {
        super(class_2561.method_43471("gui.buildnotes.edit_build_title"), parent);
        this.build = build;
    }

    @Override
    protected int getTopMargin() { return 20; }

    @Override
    protected int getBottomMargin() {
        return (UIHelper.BUTTON_HEIGHT * 2) + (UIHelper.OUTER_PADDING * 2) + UIHelper.BUTTON_ROW_SPACING;
    }

    @Override
    protected void method_25426() {
        super.method_25426(); // calls initContent()

        // --- BOTTOM BUTTON ROW ---
        int bottomRowY = UIHelper.getBottomButtonY(this, 2);
        List<class_2561> bottomTexts = List.of(
                class_2561.method_43471("gui.buildnotes.save_button"),
                class_2561.method_43471("gui.buildnotes.close_button")
        );
        UIHelper.createButtonRow(this, bottomRowY, bottomTexts, (index, x, width) -> {
            if (index == 0) {
                this.method_37063(new DarkButtonWidget(x, bottomRowY, width, UIHelper.BUTTON_HEIGHT,
                    bottomTexts.get(0), button -> {
                        saveBuild();
                        open(new ViewBuildScreen(this.parent, this.build));
                    })
                );
            } else {
                this.method_37063(new DarkButtonWidget(x, bottomRowY, width, UIHelper.BUTTON_HEIGHT, bottomTexts.get(1), button -> this.method_25419()));
            }
        });

        // --- TOP BUTTON ROW ---
        int topRowY = UIHelper.getBottomButtonY(this, 1);
        List<class_2561> topTexts = List.of(
                class_2561.method_43471("gui.buildnotes.edit.coords"),
                class_2561.method_43471("gui.buildnotes.edit.dimension"),
                class_2561.method_43471("gui.buildnotes.edit.biome"),
                class_2561.method_43471("gui.buildnotes.edit.add_images"),
                class_2561.method_43471("gui.buildnotes.edit.add_field"),
                getScopeButtonText()
        );
        UIHelper.createButtonRow(this, topRowY, topTexts, (index, x, width) -> {
            switch (index) {
                case 0 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(0), b -> insertCoords()));
                case 1 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(1), b -> insertDimension()));
                case 2 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(2), b -> insertBiome()));
                case 3 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(3), b -> openImageSelectionDialog()));
                case 4 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(4), b -> {
                    saveBuild();
                    this.open(new RequestFieldTitleScreen(this, this::addCustomField));
                }));
                case 5 ->
                        this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(5), button -> {
                            saveBuild();
                            cycleScope();
                            this.method_25423(this.field_22787, this.field_22789, this.field_22790);
                        }));
            }
        });

        // --- IMAGE GALLERY WIDGETS ---
        if (!build.getImageFileNames().isEmpty()) {
            int panelSpacing = BuildScreenLayouts.PANEL_SPACING;

            int contentWidth = (int) (this.field_22789 * BuildScreenLayouts.CONTENT_WIDTH_RATIO);
            int contentX = (this.field_22789 - contentWidth) / 2;
            int galleryHeight = (int) (contentWidth * (BuildScreenLayouts.GALLERY_ASPECT_RATIO_H / BuildScreenLayouts.GALLERY_ASPECT_RATIO_W));
            int galleryY = getTopMargin() + BuildScreenLayouts.NAME_FIELD_HEIGHT + panelSpacing + BuildScreenLayouts.SMALL_FIELD_HEIGHT + panelSpacing;
            int navButtonY = galleryY + (galleryHeight - 20) / 2;

            prevImageButton = new DarkButtonWidget(contentX - 25, navButtonY, 20, 20, class_2561.method_43471("gui.buildnotes.gallery.previous"), b -> switchImage(-1));
            nextImageButton = new DarkButtonWidget(contentX + contentWidth + 5, navButtonY, 20, 20, class_2561.method_43471("gui.buildnotes.gallery.next"), b -> switchImage(1));
            DarkButtonWidget deleteImageButton = new DarkButtonWidget(contentX + contentWidth - 22, galleryY + 2, 20, 20, class_2561.method_43471("gui.buildnotes.gallery.delete"), b -> removeCurrentImage());

            addScrollableWidget(prevImageButton);
            addScrollableWidget(nextImageButton);
            addScrollableWidget(deleteImageButton);
            updateNavButtons();
        }

        this.method_48265(this.nameField);
    }

    @Override
    protected void initContent() {
        this.customFieldWidgets.clear();
        int contentWidth = (int) (this.field_22789 * BuildScreenLayouts.CONTENT_WIDTH_RATIO);
        int contentX = (this.field_22789 - contentWidth) / 2;
        int yPos = getTopMargin();
        int panelSpacing = BuildScreenLayouts.PANEL_SPACING;

        // --- Name Widget ---
        this.nameField = new MultiLineTextFieldWidget(
                this.field_22793, contentX, yPos + 5, contentWidth, BuildScreenLayouts.NAME_FIELD_HEIGHT, build.getName(),
                class_2561.method_43471("gui.buildnotes.placeholder.build_name").getString(),
                1, false
        );

        addScrollableWidget(this.nameField);
        yPos += BuildScreenLayouts.NAME_FIELD_HEIGHT + panelSpacing;

        // --- Coords & Dimension Widgets ---
        int fieldWidth = (contentWidth - panelSpacing) / 2;
        this.coordsField = new MultiLineTextFieldWidget(
                this.field_22793, contentX + 50, yPos, fieldWidth - 50, BuildScreenLayouts.SMALL_FIELD_HEIGHT, build.getCoordinates(),
                class_2561.method_43471("gui.buildnotes.placeholder.coords").getString(), 1, false
        );
        addScrollableWidget(this.coordsField);
        int dimensionX = contentX + fieldWidth + panelSpacing;
        this.dimensionField = new MultiLineTextFieldWidget(
                this.field_22793, dimensionX + 65, yPos, fieldWidth - 65, BuildScreenLayouts.SMALL_FIELD_HEIGHT, build.getDimension(),
                class_2561.method_43471("gui.buildnotes.placeholder.dimension").getString(), 1, false
        );
        addScrollableWidget(this.dimensionField);
        yPos += BuildScreenLayouts.SMALL_FIELD_HEIGHT + panelSpacing;

        // --- Image Gallery Placeholder ---
        if (!build.getImageFileNames().isEmpty()) {
            int galleryHeight = (int) (contentWidth * (BuildScreenLayouts.GALLERY_ASPECT_RATIO_H / BuildScreenLayouts.GALLERY_ASPECT_RATIO_W));
            yPos += galleryHeight + panelSpacing;
        }

        // --- Description, Credits, and Custom Fields ---
        yPos += BuildScreenLayouts.LABEL_HEIGHT;
        this.descriptionField = new MultiLineTextFieldWidget(
                this.field_22793, contentX, yPos, contentWidth, BuildScreenLayouts.DESCRIPTION_FIELD_HEIGHT, build.getDescription(),
                class_2561.method_43471("gui.buildnotes.placeholder.description").getString(), Integer.MAX_VALUE, true
        );
        addScrollableWidget(this.descriptionField);
        yPos += BuildScreenLayouts.DESCRIPTION_FIELD_HEIGHT + panelSpacing;

        yPos += BuildScreenLayouts.LABEL_HEIGHT;
        this.designerField = new MultiLineTextFieldWidget(
                this.field_22793, contentX, yPos, contentWidth, BuildScreenLayouts.CREDITS_FIELD_HEIGHT, build.getCredits(),
                class_2561.method_43471("gui.buildnotes.placeholder.credits").getString(), Integer.MAX_VALUE, true
        );
        addScrollableWidget(this.designerField);
        yPos += BuildScreenLayouts.CREDITS_FIELD_HEIGHT + panelSpacing;

        for (CustomField field : this.build.getCustomFields()) {
            yPos += BuildScreenLayouts.LABEL_HEIGHT;
            int fieldWidgetWidth = contentWidth - BuildScreenLayouts.CUSTOM_FIELD_REMOVE_BTN_WIDTH - panelSpacing;

            MultiLineTextFieldWidget fieldArea = new MultiLineTextFieldWidget(this.field_22793, contentX, yPos, fieldWidgetWidth, BuildScreenLayouts.CUSTOM_FIELD_HEIGHT,
                    field.getContent(), "", Integer.MAX_VALUE, true);

            addScrollableWidget(fieldArea);

            this.customFieldWidgets.put(field, fieldArea);
            DarkButtonWidget removeButton = new DarkButtonWidget(contentX + fieldWidgetWidth + panelSpacing, yPos, BuildScreenLayouts.CUSTOM_FIELD_REMOVE_BTN_WIDTH, 20,
                    class_2561.method_43471("gui.buildnotes.edit.remove_field"),  button -> removeCustomField(field));

            addScrollableWidget(removeButton);
            yPos += BuildScreenLayouts.CUSTOM_FIELD_HEIGHT + panelSpacing;
        }

        this.totalContentHeight = yPos;
    }

    @Override
    protected void renderContent(class_332 context, int mouseX, int mouseY, float delta) {
        int contentWidth = (int) (this.field_22789 * BuildScreenLayouts.CONTENT_WIDTH_RATIO);
        int contentX = (this.field_22789 - contentWidth) / 2;
        int yPos = getTopMargin();
        int panelSpacing = BuildScreenLayouts.PANEL_SPACING;

        UIHelper.drawPanel(context, contentX, yPos, contentWidth, BuildScreenLayouts.NAME_FIELD_HEIGHT);
        yPos += BuildScreenLayouts.NAME_FIELD_HEIGHT + panelSpacing;


        int fieldWidth = (contentWidth - panelSpacing) / 2;
        UIHelper.drawPanel(context, contentX, yPos, fieldWidth, BuildScreenLayouts.SMALL_FIELD_HEIGHT);
        context.method_51439(this.field_22793, class_2561.method_43471("gui.buildnotes.label.coords").method_27692(class_124.field_1080), contentX + 4, (int)(yPos + (BuildScreenLayouts.SMALL_FIELD_HEIGHT - 8) / 2f + 1), Colors.TEXT_MUTED, false);
        int dimensionX = contentX + fieldWidth + panelSpacing;
        UIHelper.drawPanel(context, dimensionX, yPos, fieldWidth, BuildScreenLayouts.SMALL_FIELD_HEIGHT);
        context.method_51439(this.field_22793, class_2561.method_43471("gui.buildnotes.label.dimension").method_27692(class_124.field_1080), dimensionX + 4, (int)(yPos + (BuildScreenLayouts.SMALL_FIELD_HEIGHT - 8) / 2f + 1), Colors.TEXT_MUTED, false);
        yPos += BuildScreenLayouts.SMALL_FIELD_HEIGHT + panelSpacing;

        if (!build.getImageFileNames().isEmpty()) {
            int galleryBoxHeight = (int) (contentWidth * (BuildScreenLayouts.GALLERY_ASPECT_RATIO_H / BuildScreenLayouts.GALLERY_ASPECT_RATIO_W));
            UIHelper.drawPanel(context, contentX, yPos, contentWidth, galleryBoxHeight);
            String currentImageName = build.getImageFileNames().get(currentImageIndex);
            if (downloadingImages.contains(currentImageName)) {
                context.method_27534(field_22793, class_2561.method_43471("gui.buildnotes.gallery.loading").method_27692(class_124.field_1054), this.field_22789 / 2, yPos + galleryBoxHeight / 2 - 4, Colors.TEXT_PRIMARY);
            } else {
                ImageData data = getImageDataForCurrentImage();
                if (data != null && data.textureId != null) {
                    int boxWidth = contentWidth - 4;
                    int boxHeight = galleryBoxHeight - 4;
                    float imageAspect = (float) data.width / (float) data.height;
                    float boxAspect = (float) boxWidth / (float) boxHeight;
                    int renderWidth = boxWidth;
                    int renderHeight = boxHeight;
                    if (imageAspect > boxAspect) {
                        renderHeight = (int) (boxWidth / imageAspect);
                    } else {
                        renderWidth = (int) (boxHeight * imageAspect);
                    }
                    int renderX = contentX + 2 + (boxWidth - renderWidth) / 2;
                    int renderY = yPos + 2 + (boxHeight - renderHeight) / 2;
                    context.method_25290(class_10799.field_56883, data.textureId, renderX, renderY, 0, 0, renderWidth, renderHeight, renderWidth, renderHeight);
                } else {
                    context.method_27534(field_22793, class_2561.method_43471("gui.buildnotes.gallery.error").method_27692(class_124.field_1061), this.field_22789 / 2, yPos + galleryBoxHeight / 2 - 4, Colors.TEXT_PRIMARY);
                }
            }
            String counter = (currentImageIndex + 1) + " / " + build.getImageFileNames().size();
            int counterWidth = field_22793.method_1727(counter);
            context.method_51433(this.field_22793, counter, contentX + contentWidth - counterWidth - 5, yPos + galleryBoxHeight - 12, Colors.TEXT_PRIMARY, true);
            yPos += galleryBoxHeight + panelSpacing;
        }

        context.method_51439(this.field_22793, class_2561.method_43471("gui.buildnotes.label.description").method_27692(class_124.field_1080), contentX, yPos, Colors.TEXT_PRIMARY, false);
        yPos += BuildScreenLayouts.LABEL_HEIGHT;
        UIHelper.drawPanel(context, contentX, yPos, contentWidth, BuildScreenLayouts.DESCRIPTION_FIELD_HEIGHT);
        yPos += BuildScreenLayouts.DESCRIPTION_FIELD_HEIGHT + panelSpacing;
        context.method_51439(this.field_22793, class_2561.method_43471("gui.buildnotes.label.credits").method_27692(class_124.field_1080), contentX, yPos, Colors.TEXT_PRIMARY, false);
        yPos += BuildScreenLayouts.LABEL_HEIGHT;
        UIHelper.drawPanel(context, contentX, yPos, contentWidth, BuildScreenLayouts.CREDITS_FIELD_HEIGHT);
        yPos += BuildScreenLayouts.CREDITS_FIELD_HEIGHT + panelSpacing;
        for (CustomField field : this.build.getCustomFields()) {
            context.method_51439(this.field_22793, class_2561.method_43471(field.getTitle() + ":").method_27692(class_124.field_1080), contentX, yPos, Colors.TEXT_PRIMARY, false);
            yPos += BuildScreenLayouts.LABEL_HEIGHT;
            UIHelper.drawPanel(context, contentX, yPos, contentWidth, BuildScreenLayouts.CUSTOM_FIELD_HEIGHT);
            yPos += BuildScreenLayouts.CUSTOM_FIELD_HEIGHT + panelSpacing;
        }
        context.method_27534(this.field_22793, this.field_22785, this.field_22789 / 2, 8, Colors.TEXT_PRIMARY);
    }

    // --- Image Management Logic ---
    private void switchImage(int direction) {
        int newIndex = this.currentImageIndex + direction;
        if (newIndex >= 0 && newIndex < build.getImageFileNames().size()) {
            this.currentImageIndex = newIndex;
            updateNavButtons();
        }
    }

    private void removeCurrentImage() {
        if (build.getImageFileNames().isEmpty()) return;
        String fileNameToRemove = build.getImageFileNames().get(currentImageIndex);
        build.getImageFileNames().remove(fileNameToRemove);
        if (currentImageIndex >= build.getImageFileNames().size()) {
            currentImageIndex = Math.max(0, build.getImageFileNames().size() - 1);
        }
        rebuild();
    }

    private void updateNavButtons() {
        if (prevImageButton != null) prevImageButton.field_22763 = currentImageIndex > 0;
        if (nextImageButton != null) nextImageButton.field_22763 = currentImageIndex < build.getImageFileNames().size() - 1;
    }

    private ImageData getImageDataForCurrentImage() {
        if (build.getImageFileNames().isEmpty()) return null;
        String fileName = build.getImageFileNames().get(currentImageIndex);
        if (textureCache.containsKey(fileName)) {
            return textureCache.get(fileName);
        }
        try {
            Path imagePath = FabricLoader.getInstance().getConfigDir().resolve("buildnotes").resolve("images").resolve(build.getId().toString()).resolve(fileName);
            if (Files.exists(imagePath)) {
                try (InputStream stream = Files.newInputStream(imagePath)) {
                    class_1011 image = class_1011.method_4309(stream);

                    class_2960 textureId = class_2960.method_60655(Buildnotes.MOD_ID, "buildnotes_image_" + build.getId() + "_" + fileName.hashCode());
                    class_1043 texture = new class_1043(() -> textureId.toString(), image);

                    this.field_22787.method_1531().method_4616(textureId, texture);

                    ImageData data = new ImageData(textureId, image.method_4307(), image.method_4323());
                    textureCache.put(fileName, data);
                    return data;
                }
            }
            else {
                // --- Only request images for SERVER-scoped builds when on a dedicated server ---
                boolean isDedicatedServer = this.field_22787 != null && !this.field_22787.method_1496();
                if (build.getScope() == Scope.SERVER && isDedicatedServer) {
                    // Image does NOT exist, request it from the server
                    if (!downloadingImages.contains(fileName)) {
                        downloadingImages.add(fileName);
                        ClientImageTransferManager.requestImage(build.getId(), fileName, () -> {
                            // This is the CALLBACK! It runs when the download is finished (success or fail).
                            this.field_22787.execute(() -> downloadingImages.remove(fileName));
                        });
                    }
                }
                // Return null for now, the render method will see this and show "Loading..."
                return null;
            }
        } catch (Exception e) {
            textureCache.put(fileName, null);
        }
        return null;
    }

    // --- Overridden & Helper Methods ---
    @Override
    public void method_25419() {
        // Clean up textures to prevent memory leaks
        textureCache.values().forEach(data -> {
            if (data != null && data.textureId != null) {
                field_22787.method_1531().method_4615(data.textureId);
            }
        });
        saveBuild();
        super.method_25419();
    }

    private void rebuild() {
        saveBuild();
        this.open(new EditBuildScreen(this.parent, this.build));
    }

    private void saveBuild() {
        build.setName(nameField.getText());
        build.setCoordinates(coordsField.getText());
        build.setDimension(dimensionField.getText());
        build.setDescription(descriptionField.getText());
        build.setCredits(designerField.getText());
        for (Map.Entry<CustomField, MultiLineTextFieldWidget> entry : customFieldWidgets.entrySet()) {
            entry.getKey().setContent(entry.getValue().getText());
        }
        build.updateTimestamp();
        DataManager.getInstance().saveBuild(this.build);
    }


    private void insertTextAtLastFocus(String text) {
        if (this.lastFocusedTextField != null) {
            this.lastFocusedTextField.insertText(text);
            this.method_25395(this.lastFocusedTextField);
        } else if (this.descriptionField != null) { // Fallback to a default field
            this.descriptionField.insertText(text);
            this.method_25395(this.descriptionField);
        }
    }

    private void insertCoords() {
        if (this.field_22787 == null || this.field_22787.field_1724 == null) return;
        String coords = String.format("%.0f, %.0f, %.0f", this.field_22787.field_1724.method_23317(), this.field_22787.field_1724.method_23318(), this.field_22787.field_1724.method_23321());
        insertTextAtLastFocus(coords);
    }

    private void insertDimension() {
        if (this.field_22787 == null || this.field_22787.field_1724 == null) return;
        String dim = this.field_22787.field_1724.method_73183().method_27983().method_29177().toString();
        insertTextAtLastFocus(dim);
    }

    private void insertBiome() {
        if (this.field_22787 == null || this.field_22787.field_1724 == null || this.field_22787.field_1687 == null) return;
        class_2338 playerPos = this.field_22787.field_1724.method_24515();
        class_6880<class_1959> biomeEntry = this.field_22787.field_1687.method_23753(playerPos);
        String biomeId = biomeEntry.method_40230().map(class_5321::method_29177).map(class_2960::toString).orElse("minecraft:unknown");
        insertTextAtLastFocus(biomeId);
    }

    private void openImageSelectionDialog() {
        new Thread(() -> {
            String selectedFiles = TinyFileDialogs.tinyfd_openFileDialog("Select Image(s)", null, null, "Image Files (*.png, *.jpg, *.jpeg)", true);
            field_22787.execute(() -> {
                if (selectedFiles != null) {
                    processSelectedFiles(selectedFiles.split("\\|"));
                }
                if (field_22787 != null) {
                    long handle = field_22787.method_22683().method_4490();
                    GLFW.glfwRestoreWindow(handle);
                    GLFW.glfwFocusWindow(handle);
                }
            });
        }).start();
    }

    private void processSelectedFiles(String[] paths) {
        new Thread(() -> {
            for (String pathStr : paths) {
                try {
                    Path sourcePath = Path.of(pathStr);
                    String fileName = sourcePath.getFileName().toString().toLowerCase();
                    if (!(fileName.endsWith(".png") || fileName.endsWith(".jpg") || fileName.endsWith(".jpeg"))) continue;
                    Path destDir = FabricLoader.getInstance().getConfigDir().resolve("buildnotes").resolve("images").resolve(build.getId().toString());
                    Files.createDirectories(destDir);
                    Path destPath = destDir.resolve(fileName);
                    Files.copy(sourcePath, destPath, StandardCopyOption.REPLACE_EXISTING);
                    if (!build.getImageFileNames().contains(fileName)) {
                        build.getImageFileNames().add(fileName);
                    }
                } catch (Exception e) {
                    // Log error
                }
            }
            field_22787.execute(this::rebuild);
        }).start();
    }

    private void addCustomField(String title) {
        if (title == null || title.isBlank()) return;
        this.build.getCustomFields().add(new CustomField(title, ""));
        rebuild();
    }

    private void removeCustomField(CustomField field) {
        this.build.getCustomFields().remove(field);
        rebuild();
    }

    private void cycleScope() {
        Scope currentScope = build.getScope();
        if (ClientSession.isOnServer() && ClientSession.hasEditPermission()) {
            if (currentScope == Scope.WORLD) build.setScope(Scope.GLOBAL);
            else if (currentScope == Scope.GLOBAL) build.setScope(Scope.SERVER);
            else build.setScope(Scope.WORLD);
        } else {
            build.setScope(currentScope == Scope.WORLD ? Scope.GLOBAL : Scope.WORLD);
        }
    }

    private class_2561 getScopeButtonText() {
        class_2561 scopeName;
        Scope currentScope = build.getScope();
        if (currentScope == Scope.GLOBAL) {
            scopeName = class_2561.method_43471("gui.buildnotes.edit.scope.global");
        } else if (currentScope == Scope.SERVER) {
            scopeName = class_2561.method_43471("gui.buildnotes.edit.scope.server");
        } else {
            scopeName = this.field_22787 != null && this.field_22787.method_1496()
                    ? class_2561.method_43471("gui.buildnotes.edit.scope.world")
                    : class_2561.method_43471("gui.buildnotes.edit.scope.per_server");
        }
        return class_2561.method_43469("gui.buildnotes.edit.scope_button", scopeName);
    }

    @Override
    public void method_25395(class_364 focused) {
        super.method_25395(focused);
        if (focused instanceof MultiLineTextFieldWidget widget) {
            this.lastFocusedTextField = widget;
        }
    }

    // --- Inner Class for RequestFieldTitleScreen (Unchanged) ---
    private static class RequestFieldTitleScreen extends BaseScreen {
        private final Consumer<String> onConfirm;
        private class_342 titleField;

        protected RequestFieldTitleScreen(class_437 parent, Consumer<String> onConfirm) {
            super(class_2561.method_43471("gui.buildnotes.prompt.field_title"), parent);
            this.onConfirm = onConfirm;
        }

        @Override
        protected void method_25426() {
            super.method_25426();
            int panelW = 200;
            int panelX = (this.field_22789 - panelW) / 2;
            int panelY = (this.field_22790 - 100) / 2;

            this.titleField = new class_342(this.field_22793, panelX + 10, panelY + 20, panelW - 20, 20, class_2561.method_30163(""));
            this.method_25429(this.titleField);

            int buttonsY = panelY + 60;
            int btnStartX = (this.field_22789 - ((85 * 2) + UIHelper.BUTTON_SPACING)) / 2;
            this.method_37063(new DarkButtonWidget(btnStartX, buttonsY, 85, 20, class_2561.method_43471("gui.buildnotes.confirm_button"), button -> {
                this.onConfirm.accept(this.titleField.method_1882());
                this.open(this.parent);
            }));
            this.method_37063(new DarkButtonWidget(btnStartX + 95, buttonsY, 85, 20, class_2561.method_43471("gui.buildnotes.cancel_button"), button -> this.open(parent)));
            this.method_48265(this.titleField);
        }

        @Override
        public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
            parent.method_25394(context, -1, -1, delta);
            int panelW = 200;
            int panelH = 100;
            int panelX = (this.field_22789 - panelW) / 2;
            int panelY = (this.field_22790 - panelH) / 2;
            UIHelper.drawPanel(context, panelX, panelY, panelW, panelH);

            context.method_27534(this.field_22793, this.field_22785, this.field_22789 / 2, panelY + 8, Colors.TEXT_PRIMARY);

            this.titleField.method_25394(context, mouseX, mouseY, delta);

            super.method_25394(context, mouseX, mouseY, delta);
        }
    }
}