package net.atif.buildnotes.gui.screen;

import net.atif.buildnotes.client.ClientImageTransferManager;
import net.atif.buildnotes.client.ClientSession;
import net.atif.buildnotes.data.Build;
import net.atif.buildnotes.data.CustomField;
import net.atif.buildnotes.data.DataManager;
import net.atif.buildnotes.data.Scope;
import net.atif.buildnotes.gui.helper.UIHelper;
import net.atif.buildnotes.gui.widget.DarkButtonWidget;
import net.atif.buildnotes.gui.widget.MultiLineTextFieldWidget;

import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_124;
import net.minecraft.class_2338;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2588;
import net.minecraft.class_2960;
import net.minecraft.class_332;
import net.minecraft.class_342;
import net.minecraft.class_364;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import com.mojang.blaze3d.systems.RenderSystem;
import org.lwjgl.glfw.GLFW;
import org.lwjgl.util.tinyfd.TinyFileDialogs;

import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.*;
import java.util.function.Consumer;

public class EditBuildScreen extends ScrollableScreen {

    private final Build build;

    // --- Image Gallery Fields ---
    private record ImageData(class_2960 textureId, int width, int height) {}
    private int currentImageIndex = 0;
    private final Map<String, ImageData> textureCache = new HashMap<>();
    private final Set<String> downloadingImages = new HashSet<>();

    private DarkButtonWidget prevImageButton, nextImageButton, deleteImageButton;

    // --- Standard Fields ---
    private MultiLineTextFieldWidget nameField, coordsField, dimensionField, descriptionField, designerField;
    private final Map<CustomField, MultiLineTextFieldWidget> customFieldWidgets = new LinkedHashMap<>();
    private MultiLineTextFieldWidget lastFocusedTextField;
    private DarkButtonWidget scopeToggleButton;

    public EditBuildScreen(class_437 parent, Build build) {
        super(new class_2588("gui.buildnotes.edit_build_title"), parent);
        this.build = build;
    }

    @Override
    protected void method_25426() {
        super.method_25426(); // calls initContent()

        // --- BOTTOM BUTTON ROW ---
        int buttonsY = this.field_22790 - UIHelper.BUTTON_HEIGHT - UIHelper.BOTTOM_PADDING;
        List<class_2561> bottomTexts = List.of(
                new class_2588("gui.buildnotes.save_button"),
                new class_2588("gui.buildnotes.close_button")
        );
        UIHelper.createBottomButtonRow(this, buttonsY, bottomTexts, (index, x, width) -> {
            if (index == 0) {
                this.method_37063(new DarkButtonWidget(x, buttonsY, width, UIHelper.BUTTON_HEIGHT, bottomTexts.get(0), button -> saveBuild()));
            } else {
                this.method_37063(new DarkButtonWidget(x, buttonsY, width, UIHelper.BUTTON_HEIGHT, bottomTexts.get(1), button -> this.method_25419()));
            }
        });

        // --- TOP BUTTON ROW ---
        int topRowY = buttonsY - UIHelper.BUTTON_HEIGHT - 5;
        List<class_2561> topTexts = List.of(
                new class_2588("gui.buildnotes.edit.coords"),
                new class_2588("gui.buildnotes.edit.dimension"),
                new class_2588("gui.buildnotes.edit.biome"),
                new class_2588("gui.buildnotes.edit.add_images"),
                new class_2588("gui.buildnotes.edit.add_field"),
                getScopeButtonText()
        );
        UIHelper.createBottomButtonRow(this, topRowY, topTexts, (index, x, width) -> {
            switch (index) {
                case 0 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(0), b -> insertCoords()));
                case 1 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(1), b -> insertDimension()));
                case 2 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(2), b -> insertBiome()));
                case 3 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(3), b -> openImageSelectionDialog()));
                case 4 -> this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(4), b -> {
                    saveBuild();
                    this.open(new RequestFieldTitleScreen(this, this::addCustomField));
                }));
                case 5 -> {
                    this.method_37063(new DarkButtonWidget(x, topRowY, width, UIHelper.BUTTON_HEIGHT, topTexts.get(5), button -> {
                        saveBuild();
                        cycleScope();
                        this.method_25423(this.field_22787, this.field_22789, this.field_22790);
                    }));
                }
            }
        });

        // --- IMAGE GALLERY WIDGETS ---
        if (!build.getImageFileNames().isEmpty()) {
            int contentWidth = (int) (this.field_22789 * 0.6);
            int contentX = (this.field_22789 - contentWidth) / 2;
            int galleryHeight = (int) (contentWidth * (9.0 / 16.0));
            int galleryY = getTopMargin() + 25 + 5 + 20 + 5;
            int navButtonY = galleryY + (galleryHeight - 20) / 2;

            prevImageButton = new DarkButtonWidget(contentX - 25, navButtonY, 20, 20, new class_2588("gui.buildnotes.gallery.previous"), b -> switchImage(-1));
            nextImageButton = new DarkButtonWidget(contentX + contentWidth + 5, navButtonY, 20, 20, new class_2588("gui.buildnotes.gallery.next"), b -> switchImage(1));
            deleteImageButton = new DarkButtonWidget(contentX + contentWidth - 22, galleryY + 2, 20, 20, new class_2588("gui.buildnotes.gallery.delete"), b -> removeCurrentImage());

            addScrollableWidget(prevImageButton);
            addScrollableWidget(nextImageButton);
            addScrollableWidget(deleteImageButton);
            updateNavButtons();
        }

        this.method_20085(this.nameField);
    }

    @Override
    protected void initContent() {
        this.customFieldWidgets.clear();
        int contentWidth = (int) (this.field_22789 * 0.6);
        int contentX = (this.field_22789 - contentWidth) / 2;
        int yPos = getTopMargin();
        final int panelSpacing = 5;
        final int labelHeight = 12;

        // --- Name Widget ---
        this.nameField = new MultiLineTextFieldWidget(
                this.field_22793, contentX, yPos + 5, contentWidth, 25, build.getName(),
                new class_2588("gui.buildnotes.placeholder.build_name").getString(),
                1, false
        );

        this.nameField.setInternalScissoring(false);
        addScrollableWidget(this.nameField);
        yPos += 25 + panelSpacing;

        // --- Coords & Dimension Widgets ---
        int smallFieldHeight = 20;
        int fieldWidth = (contentWidth - panelSpacing) / 2;
        this.coordsField = new MultiLineTextFieldWidget(
                this.field_22793, contentX + 50, yPos, fieldWidth - 50, smallFieldHeight, build.getCoordinates(),
                new class_2588("gui.buildnotes.placeholder.coords").getString(), 1, false
        );
        this.coordsField.setInternalScissoring(false);
        addScrollableWidget(this.coordsField);
        int dimensionX = contentX + fieldWidth + panelSpacing;
        this.dimensionField = new MultiLineTextFieldWidget(
                this.field_22793, dimensionX + 65, yPos, fieldWidth - 65, smallFieldHeight, build.getDimension(),
                new class_2588("gui.buildnotes.placeholder.dimension").getString(), 1, false
        );
        this.dimensionField.setInternalScissoring(false);
        addScrollableWidget(this.dimensionField);
        yPos += smallFieldHeight + panelSpacing;

        // --- Image Gallery Placeholder ---
        if (!build.getImageFileNames().isEmpty()) {
            int galleryHeight = (int) (contentWidth * (9.0 / 16.0)); // 16:9 aspect ratio
            yPos += galleryHeight + panelSpacing;
        }

        // --- Description, Credits, and Custom Fields ---
        yPos += labelHeight;
        this.descriptionField = new MultiLineTextFieldWidget(
                this.field_22793, contentX, yPos, contentWidth, 80, build.getDescription(),
                new class_2588("gui.buildnotes.placeholder.description").getString(), Integer.MAX_VALUE, true
        );
        this.descriptionField.setInternalScissoring(false);
        addScrollableWidget(this.descriptionField);
        yPos += 80 + panelSpacing;

        yPos += labelHeight;
        this.designerField = new MultiLineTextFieldWidget(
                this.field_22793, contentX, yPos, contentWidth, 40, build.getCredits(),
                new class_2588("gui.buildnotes.placeholder.credits").getString(), Integer.MAX_VALUE, true
        );
        this.designerField.setInternalScissoring(false);
        addScrollableWidget(this.designerField);
        yPos += 40 + panelSpacing;

        for (CustomField field : this.build.getCustomFields()) {
            yPos += labelHeight;
            int customRemoveBtnWidth = 20;
            int fieldWidgetWidth = contentWidth - customRemoveBtnWidth - panelSpacing;

            MultiLineTextFieldWidget fieldArea = new MultiLineTextFieldWidget(this.field_22793, contentX, yPos, fieldWidgetWidth, 40,
                    field.getContent(), "", Integer.MAX_VALUE, true);

            fieldArea.setInternalScissoring(false);
            addScrollableWidget(fieldArea);

            this.customFieldWidgets.put(field, fieldArea);
            DarkButtonWidget removeButton = new DarkButtonWidget(contentX + fieldWidgetWidth + panelSpacing, yPos, customRemoveBtnWidth, 20,
                    new class_2588("gui.buildnotes.edit.remove_field"),  button -> removeCustomField(field));

            addScrollableWidget(removeButton);
            yPos += 40 + panelSpacing;
        }

        this.totalContentHeight = yPos;
    }

    @Override
    protected void renderContent(class_4587 matrices, int mouseX, int mouseY, float delta) {
        int contentWidth = (int) (this.field_22789 * 0.6);
        int contentX = (this.field_22789 - contentWidth) / 2;
        int yPos = getTopMargin();
        final int panelSpacing = 5;
        final int labelHeight = 12;

        UIHelper.drawPanel(matrices, contentX, yPos, contentWidth, 25);
        yPos += 25 + panelSpacing;
        int smallFieldHeight = 20;
        int fieldWidth = (contentWidth - panelSpacing) / 2;
        UIHelper.drawPanel(matrices, contentX, yPos, fieldWidth, smallFieldHeight);
        this.field_22793.method_30883(matrices, new class_2588("gui.buildnotes.label.coords").method_27692(class_124.field_1080), contentX + 4, (float)(yPos + (smallFieldHeight - 8) / 2f + 1), 0xCCCCCC);
        int dimensionX = contentX + fieldWidth + panelSpacing;
        UIHelper.drawPanel(matrices, dimensionX, yPos, fieldWidth, smallFieldHeight);
        this.field_22793.method_30883(matrices, new class_2588("gui.buildnotes.label.dimension").method_27692(class_124.field_1080), dimensionX + 4, (float)(yPos + (smallFieldHeight - 8) / 2f + 1), 0xCCCCCC);
        yPos += smallFieldHeight + panelSpacing;

        if (!build.getImageFileNames().isEmpty()) {
            int galleryBoxHeight = (int) (contentWidth * (9.0 / 16.0));
            method_25294(matrices, contentX, yPos, contentX + contentWidth, yPos + galleryBoxHeight, 0x77000000);
            String currentImageName = build.getImageFileNames().get(currentImageIndex);
            if (downloadingImages.contains(currentImageName)) {
                method_27534(matrices, field_22793, new class_2588("gui.buildnotes.gallery.loading").method_27692(class_124.field_1054), this.field_22789 / 2, yPos + galleryBoxHeight / 2 - 4, 0xFFFFFF);
            } else {
                ImageData data = getImageDataForCurrentImage();
                if (data != null && data.textureId != null) {
                    RenderSystem.setShaderTexture(0, data.textureId);
                    RenderSystem.enableBlend();
                    int boxWidth = contentWidth - 4;
                    int boxHeight = galleryBoxHeight - 4;
                    float imageAspect = (float) data.width / (float) data.height;
                    float boxAspect = (float) boxWidth / (float) boxHeight;
                    int renderWidth = boxWidth;
                    int renderHeight = boxHeight;
                    if (imageAspect > boxAspect) {
                        renderHeight = (int) (boxWidth / imageAspect);
                    } else {
                        renderWidth = (int) (boxHeight * imageAspect);
                    }
                    int renderX = contentX + 2 + (boxWidth - renderWidth) / 2;
                    int renderY = yPos + 2 + (boxHeight - renderHeight) / 2;
                    class_332.method_25290(matrices, renderX, renderY, 0, 0, renderWidth, renderHeight, renderWidth, renderHeight);
                    RenderSystem.disableBlend();
                } else {
                    method_27534(matrices, field_22793, new class_2588("gui.buildnotes.gallery.error").method_27692(class_124.field_1061), this.field_22789 / 2, yPos + galleryBoxHeight / 2 - 4, 0xFFFFFF);
                }
            }
            String counter = (currentImageIndex + 1) + " / " + build.getImageFileNames().size();
            int counterWidth = field_22793.method_1727(counter);
            field_22793.method_1729(matrices, counter, contentX + contentWidth - counterWidth - 5, yPos + galleryBoxHeight - 12, 0xFFFFFF);
            yPos += galleryBoxHeight + panelSpacing;
        }

        this.field_22793.method_30883(matrices, new class_2588("gui.buildnotes.label.description").method_27692(class_124.field_1080), contentX, yPos, 0xFFFFFF);
        yPos += labelHeight;
        UIHelper.drawPanel(matrices, contentX, yPos, contentWidth, 80);
        yPos += 80 + panelSpacing;
        this.field_22793.method_30883(matrices, new class_2588("gui.buildnotes.label.credits").method_27692(class_124.field_1080), contentX, yPos, 0xFFFFFF);
        yPos += labelHeight;
        UIHelper.drawPanel(matrices, contentX, yPos, contentWidth, 40);
        yPos += 40 + panelSpacing;
        for (CustomField field : this.build.getCustomFields()) {
            this.field_22793.method_30883(matrices, new class_2588(field.getTitle() + ":").method_27692(class_124.field_1080), contentX, yPos, 0xFFFFFF);
            yPos += labelHeight;
            UIHelper.drawPanel(matrices, contentX, yPos, contentWidth, 40);
            yPos += 40 + panelSpacing;
        }
        method_27534(matrices, this.field_22793, this.field_22785, this.field_22789 / 2, 8, 0xFFFFFF);
    }

    // --- Image Management Logic ---
    private void switchImage(int direction) {
        int newIndex = this.currentImageIndex + direction;
        if (newIndex >= 0 && newIndex < build.getImageFileNames().size()) {
            this.currentImageIndex = newIndex;
            updateNavButtons();
        }
    }

    private void removeCurrentImage() {
        if (build.getImageFileNames().isEmpty()) return;
        String fileNameToRemove = build.getImageFileNames().get(currentImageIndex);
        build.getImageFileNames().remove(fileNameToRemove);
        if (currentImageIndex >= build.getImageFileNames().size()) {
            currentImageIndex = Math.max(0, build.getImageFileNames().size() - 1);
        }
        rebuild();
    }

    private void updateNavButtons() {
        if (prevImageButton != null) prevImageButton.field_22763 = currentImageIndex > 0;
        if (nextImageButton != null) nextImageButton.field_22763 = currentImageIndex < build.getImageFileNames().size() - 1;
    }

    private ImageData getImageDataForCurrentImage() {
        if (build.getImageFileNames().isEmpty()) return null;
        String fileName = build.getImageFileNames().get(currentImageIndex);
        if (textureCache.containsKey(fileName)) {
            return textureCache.get(fileName);
        }
        try {
            Path imagePath = FabricLoader.getInstance().getConfigDir().resolve("buildnotes").resolve("images").resolve(build.getId().toString()).resolve(fileName);
            if (Files.exists(imagePath)) {
                try (InputStream stream = Files.newInputStream(imagePath)) {
                    class_1011 image = class_1011.method_4309(stream);
                    class_1043 texture = new class_1043(image);
                    class_2960 textureId = this.field_22787.method_1531().method_4617("buildnotes_image_" + build.getId() + "_" + fileName.hashCode(), texture);
                    ImageData data = new ImageData(textureId, image.method_4307(), image.method_4323());
                    textureCache.put(fileName, data);
                    return data;
                }
            }
            else {
                // Image does NOT exist, we need to download it.
                if (!downloadingImages.contains(fileName)) {
                    // Only start the download if we haven't already
                    downloadingImages.add(fileName);
                    ClientImageTransferManager.requestImage(build.getId(), fileName, () -> {
                        // This is the CALLBACK! It runs when the download is finished.
                        // We must execute on the client thread.
                        this.field_22787.execute(() -> {
                            downloadingImages.remove(fileName);
                        });
                    });
                }
                // Return null for now, the render method will see this and show "Loading..."
                return null;
            }
        } catch (Exception e) {
            textureCache.put(fileName, null);
        }
        return null;
    }

    // --- Overridden & Helper Methods ---
    @Override
    public void method_25419() {
        // Clean up textures to prevent memory leaks
        textureCache.values().forEach(data -> {
            if (data != null && data.textureId != null) {
                field_22787.method_1531().method_4615(data.textureId);
            }
        });
        saveBuild();
        super.method_25419();
    }

    private void rebuild() {
        saveBuild();
        this.open(new EditBuildScreen(this.parent, this.build));
    }

    @Override
    protected int getTopMargin() { return 20; }

    @Override
    protected int getBottomMargin() { return 70; }

    private void saveBuild() {
        build.setName(nameField.getText());
        build.setCoordinates(coordsField.getText());
        build.setDimension(dimensionField.getText());
        build.setDescription(descriptionField.getText());
        build.setCredits(designerField.getText());
        for (Map.Entry<CustomField, MultiLineTextFieldWidget> entry : customFieldWidgets.entrySet()) {
            entry.getKey().setContent(entry.getValue().getText());
        }
        build.updateTimestamp();
        DataManager.getInstance().saveBuild(this.build);
    }


    private void insertTextAtLastFocus(String text) {
        if (this.lastFocusedTextField != null) {
            this.lastFocusedTextField.insertText(text);
            this.method_25395(this.lastFocusedTextField);
        } else if (this.descriptionField != null) { // Fallback to a default field
            this.descriptionField.insertText(text);
            this.method_25395(this.descriptionField);
        }
    }

    private void insertCoords() {
        if (this.field_22787 == null || this.field_22787.field_1724 == null) return;
        String coords = String.format("%.0f, %.0f, %.0f", this.field_22787.field_1724.method_23317(), this.field_22787.field_1724.method_23318(), this.field_22787.field_1724.method_23321());
        insertTextAtLastFocus(coords);
    }

    private void insertDimension() {
        if (this.field_22787 == null || this.field_22787.field_1724 == null) return;
        String dim = this.field_22787.field_1724.field_6002.method_27983().method_29177().toString();
        insertTextAtLastFocus(dim);
    }

    private void insertBiome() {
        if (this.field_22787 == null || this.field_22787.field_1724 == null || this.field_22787.field_1687 == null) return;
        class_2338 playerPos = this.field_22787.field_1724.method_24515();
        String biomeId = this.field_22787.field_1687.method_30349().method_30530(class_2378.field_25114).method_10221(this.field_22787.field_1687.method_23753(playerPos).comp_349()).toString();
        insertTextAtLastFocus(biomeId);
    }

    private void openImageSelectionDialog() {
        new Thread(() -> {
            String selectedFiles = TinyFileDialogs.tinyfd_openFileDialog("Select Image(s)", null, null, "Image Files (*.png, *.jpg, *.jpeg)", true);
            field_22787.execute(() -> {
                if (selectedFiles != null) {
                    processSelectedFiles(selectedFiles.split("\\|"));
                }
                if (field_22787 != null) {
                    long handle = field_22787.method_22683().method_4490();
                    GLFW.glfwRestoreWindow(handle);
                    GLFW.glfwFocusWindow(handle);
                }
            });
        }).start();
    }

    private void processSelectedFiles(String[] paths) {
        new Thread(() -> {
            for (String pathStr : paths) {
                try {
                    Path sourcePath = Path.of(pathStr);
                    String fileName = sourcePath.getFileName().toString().toLowerCase();
                    if (!(fileName.endsWith(".png") || fileName.endsWith(".jpg") || fileName.endsWith(".jpeg"))) continue;
                    Path destDir = FabricLoader.getInstance().getConfigDir().resolve("buildnotes").resolve("images").resolve(build.getId().toString());
                    Files.createDirectories(destDir);
                    Path destPath = destDir.resolve(fileName);
                    Files.copy(sourcePath, destPath, StandardCopyOption.REPLACE_EXISTING);
                    if (!build.getImageFileNames().contains(fileName)) {
                        build.getImageFileNames().add(fileName);
                    }
                } catch (Exception e) {
                    // Log error
                }
            }
            field_22787.execute(this::rebuild);
        }).start();
    }

    private void addCustomField(String title) {
        if (title == null || title.isBlank()) return;
        this.build.getCustomFields().add(new CustomField(title, ""));
        rebuild();
    }

    private void removeCustomField(CustomField field) {
        this.build.getCustomFields().remove(field);
        rebuild();
    }

    private void cycleScope() {
        Scope currentScope = build.getScope();
        if (ClientSession.isOnServer() && ClientSession.hasEditPermission()) {
            if (currentScope == Scope.WORLD) build.setScope(Scope.GLOBAL);
            else if (currentScope == Scope.GLOBAL) build.setScope(Scope.SERVER);
            else build.setScope(Scope.WORLD);
        } else {
            build.setScope(currentScope == Scope.WORLD ? Scope.GLOBAL : Scope.WORLD);
        }
    }

    private class_2561 getScopeButtonText() {
        class_2561 scopeName;
        Scope currentScope = build.getScope();
        if (currentScope == Scope.GLOBAL) {
            scopeName = new class_2588("gui.buildnotes.edit.scope.global");
        } else if (currentScope == Scope.SERVER) {
            scopeName = new class_2588("gui.buildnotes.edit.scope.server");
        } else {
            scopeName = this.field_22787 != null && this.field_22787.method_1496()
                    ? new class_2588("gui.buildnotes.edit.scope.world")
                    : new class_2588("gui.buildnotes.edit.scope.per_server");
        }
        return new class_2588("gui.buildnotes.edit.scope_button", scopeName);
    }

    @Override
    public void method_25395(class_364 focused) {
        super.method_25395(focused);
        if (focused instanceof MultiLineTextFieldWidget widget) {
            this.lastFocusedTextField = widget;
        }
    }

    // --- Inner Class for RequestFieldTitleScreen (Unchanged) ---
    private static class RequestFieldTitleScreen extends BaseScreen {
        private final Consumer<String> onConfirm;
        private class_342 titleField;

        protected RequestFieldTitleScreen(class_437 parent, Consumer<String> onConfirm) {
            super(new class_2588("gui.buildnotes.prompt.field_title"), parent);
            this.onConfirm = onConfirm;
        }

        @Override
        protected void method_25426() {
            super.method_25426();
            int panelW = 200;
            int panelX = (this.field_22789 - panelW) / 2;
            int panelY = (this.field_22790 - 100) / 2;

            this.titleField = new class_342(this.field_22793, panelX + 10, panelY + 20, panelW - 20, 20, class_2561.method_30163(""));
            this.method_25429(this.titleField);

            int buttonsY = panelY + 60;
            int btnStartX = (this.field_22789 - ((85 * 2) + UIHelper.BUTTON_SPACING)) / 2;
            this.method_37063(new DarkButtonWidget(btnStartX, buttonsY, 85, 20, new class_2588("gui.buildnotes.confirm_button"), button -> {
                this.onConfirm.accept(this.titleField.method_1882());
                this.open(this.parent);
            }));
            this.method_37063(new DarkButtonWidget(btnStartX + 95, buttonsY, 85, 20, new class_2588("gui.buildnotes.cancel_button"), button -> this.open(parent)));
            this.method_20085(this.titleField);
        }

        @Override
        public void method_25394(class_4587 matrices, int mouseX, int mouseY, float delta) {
            parent.method_25394(matrices, -1, -1, delta);
            int panelW = 200;
            int panelH = 100;
            int panelX = (this.field_22789 - panelW) / 2;
            int panelY = (this.field_22790 - panelH) / 2;
            UIHelper.drawPanel(matrices, panelX, panelY, panelW, panelH);

            method_27534(matrices, this.field_22793, this.field_22785, this.field_22789 / 2, panelY + 8, 0xFFFFFF);

            this.titleField.method_25394(matrices, mouseX, mouseY, delta);

            super.method_25394(matrices, mouseX, mouseY, delta);
        }
    }
}