package net.atif.buildnotes.gui.screen;

import com.mojang.blaze3d.systems.RenderSystem;
import net.atif.buildnotes.data.Build;
import net.atif.buildnotes.data.CustomField;
import net.atif.buildnotes.data.DataManager;
import net.atif.buildnotes.data.TabType;
import net.atif.buildnotes.gui.helper.UIHelper;
import net.atif.buildnotes.gui.widget.DarkButtonWidget;
import net.atif.buildnotes.gui.widget.ReadOnlyMultiLineTextFieldWidget;

import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_124;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_2960;
import net.minecraft.class_332;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

public class ViewBuildScreen extends ScrollableScreen {

    private final Build build;

    private record ImageData(class_2960 textureId, int width, int height) {
    }

    private int currentImageIndex = 0;
    private final Map<String, ImageData> textureCache = new HashMap<>();
    private DarkButtonWidget prevImageButton;
    private DarkButtonWidget nextImageButton;

    public ViewBuildScreen(class_437 parent, Build build) {
        super(new class_2585(build.getName()), parent);
        this.build = build;
    }

    // --- Define scrollable area boundaries ---
    @Override
    protected int getTopMargin() { return 20; }
    @Override
    protected int getBottomMargin() { return 45; }

    @Override
    protected void initContent() {
        int contentWidth = (int) (this.field_22789 * 0.6);
        int contentX = (this.field_22789 - contentWidth) / 2;
        int yPos = getTopMargin();

        final int panelSpacing = 5;
        final int labelHeight = 12;

        // --- TITLE WIDGET ---
        int titlePanelHeight = 25;
        ReadOnlyMultiLineTextFieldWidget titleArea = new ReadOnlyMultiLineTextFieldWidget(
                this.field_22793, contentX, yPos + 5, contentWidth, titlePanelHeight,
                this.field_22785.getString(), 1, false
        );
        titleArea.setInternalScissoring(false);
        addScrollableWidget(titleArea);
        yPos += titlePanelHeight + panelSpacing;

        // --- COORDS & DIMENSION WIDGETS ---
        int smallFieldHeight = 20;
        int fieldWidth = (contentWidth - panelSpacing) / 2;

        // Coords Widget (positioned after the label)
        int coordsTextX = contentX + 50;
        ReadOnlyMultiLineTextFieldWidget coordsArea = new ReadOnlyMultiLineTextFieldWidget(
                this.field_22793, coordsTextX, yPos, fieldWidth - 50, smallFieldHeight,
                build.getCoordinates(), 1, false
        );
        coordsArea.setInternalScissoring(false);
        addScrollableWidget(coordsArea);

        // Dimension Widget (positioned after the label)
        int dimensionX = contentX + fieldWidth + panelSpacing;
        int dimensionTextX = dimensionX + 65;
        ReadOnlyMultiLineTextFieldWidget dimensionArea = new ReadOnlyMultiLineTextFieldWidget(
                this.field_22793, dimensionTextX, yPos, fieldWidth - 65, smallFieldHeight,
                build.getDimension(), 1, false
        );
        dimensionArea.setInternalScissoring(false);
        addScrollableWidget(dimensionArea);
        yPos += smallFieldHeight + panelSpacing;

        if (!build.getImageFileNames().isEmpty()) {
            int galleryHeight = (int) (contentWidth * (9.0 / 16.0)); // 16:9 aspect ratio
            yPos += galleryHeight + panelSpacing;
        }

        // --- DESCRIPTION WIDGET ---
        int descriptionHeight = 80;
        ReadOnlyMultiLineTextFieldWidget descriptionArea = new ReadOnlyMultiLineTextFieldWidget(
                this.field_22793, contentX, yPos + labelHeight, contentWidth, descriptionHeight,
                build.getDescription(), Integer.MAX_VALUE, true
        );
        descriptionArea.setInternalScissoring(false);
        addScrollableWidget(descriptionArea);
        yPos += descriptionHeight + labelHeight + panelSpacing;

        // --- CREDITS WIDGET ---
        int creditsHeight = 40;
        ReadOnlyMultiLineTextFieldWidget creditsArea = new ReadOnlyMultiLineTextFieldWidget(
                this.field_22793, contentX, yPos + labelHeight, contentWidth, creditsHeight,
                build.getCredits(), Integer.MAX_VALUE, true
        );
        creditsArea.setInternalScissoring(false);
        addScrollableWidget(creditsArea);
        yPos += creditsHeight + labelHeight + panelSpacing;

        // --- CUSTOM FIELD WIDGETS ---
        for (CustomField field : build.getCustomFields()) {
            int fieldHeight = 40;
            ReadOnlyMultiLineTextFieldWidget fieldArea = new ReadOnlyMultiLineTextFieldWidget(
                    this.field_22793, contentX, yPos + labelHeight, contentWidth, fieldHeight,
                    field.getContent(), Integer.MAX_VALUE, true
            );
            fieldArea.setInternalScissoring(false);
            addScrollableWidget(fieldArea);
            yPos += fieldHeight + labelHeight + panelSpacing;
        }

        this.totalContentHeight = yPos;
    }

    @Override
    protected void method_25426() {
        super.method_25426();

        // Use UIHelper to create the bottom 3 action buttons
        int buttonsY = this.field_22790 - UIHelper.BUTTON_HEIGHT - UIHelper.BOTTOM_PADDING;
        UIHelper.createBottomButtonRow(this, buttonsY, 3, x -> {
            int idx = (x - UIHelper.getCenteredButtonStartX(this.field_22789, 3)) / (UIHelper.BUTTON_WIDTH + UIHelper.BUTTON_SPACING);
            switch (idx) {
                case 0 -> this.method_37063(new DarkButtonWidget(x, buttonsY, UIHelper.BUTTON_WIDTH, UIHelper.BUTTON_HEIGHT,
                        new class_2588("gui.buildnotes.delete_button"), button -> confirmDelete()));
                case 1 -> this.method_37063(new DarkButtonWidget(x, buttonsY, UIHelper.BUTTON_WIDTH, UIHelper.BUTTON_HEIGHT,
                        new class_2588("gui.buildnotes.edit_button"), button -> open(new EditBuildScreen(this.parent, this.build))));
                case 2 -> this.method_37063(new DarkButtonWidget(x, buttonsY, UIHelper.BUTTON_WIDTH, UIHelper.BUTTON_HEIGHT,
                        new class_2588("gui.buildnotes.close_button"), button -> this.open(this.parent)));
            }
        });

        if (!build.getImageFileNames().isEmpty()) {
            int contentWidth = (int) (this.field_22789 * 0.6);
            int contentX = (this.field_22789 - contentWidth) / 2;
            int galleryHeight = (int) (contentWidth * (9.0 / 16.0));
            int galleryY = getTopMargin() + 25 + 5 + 20 + 5; // Y pos after title and coords
            int navButtonY = galleryY + (galleryHeight - 20) / 2;

            prevImageButton = new DarkButtonWidget(contentX - 25, navButtonY, 20, 20, new class_2585("<"), b -> switchImage(-1));
            nextImageButton = new DarkButtonWidget(contentX + contentWidth + 5, navButtonY, 20, 20, new class_2585(">"), b -> switchImage(1));
            addScrollableWidget(prevImageButton);
            addScrollableWidget(nextImageButton);
            updateNavButtons();
        }
    }

    @Override
    protected void renderContent(class_4587 matrices, int mouseX, int mouseY, float delta) {
        int contentWidth = (int) (this.field_22789 * 0.6);
        int contentX = (this.field_22789 - contentWidth) / 2;
        int yPos = getTopMargin();

        final int panelSpacing = 5;
        final int labelHeight = 12;

        // --- TITLE ---
        int titlePanelHeight = 25;
        method_25294(matrices, contentX, yPos, contentX + contentWidth, yPos + titlePanelHeight, 0x77000000);
        yPos += titlePanelHeight + panelSpacing;

        // --- COORDS & DIMENSION ---
        int smallFieldHeight = 20;
        int fieldWidth = (contentWidth - panelSpacing) / 2;

        // Backgrounds and Labels only
        UIHelper.drawPanel(matrices, contentX, yPos, fieldWidth, smallFieldHeight);
        this.field_22793.method_30883(matrices, new class_2585("Coords: ").method_27692(class_124.field_1080), contentX + 4, (float)(yPos + (smallFieldHeight - 8) / 2f + 1), 0xCCCCCC);

        int dimensionX = contentX + fieldWidth + panelSpacing;
        UIHelper.drawPanel(matrices, dimensionX, yPos, fieldWidth, smallFieldHeight);
        this.field_22793.method_30883(matrices, new class_2585("Dimension: ").method_27692(class_124.field_1080), dimensionX + 4, (float)(yPos + (smallFieldHeight - 8) / 2f + 1), 0xCCCCCC);
        yPos += smallFieldHeight + panelSpacing;

        if (!build.getImageFileNames().isEmpty()) {
            int galleryBoxHeight = (int) (contentWidth * (9.0 / 16.0));
            method_25294(matrices, contentX, yPos, contentX + contentWidth, yPos + galleryBoxHeight, 0x77000000);

            ImageData data = getImageDataForCurrentImage();
            if (data != null && data.textureId != null) {
                RenderSystem.setShaderTexture(0, data.textureId);
                RenderSystem.enableBlend();

                // --- ASPECT RATIO LOGIC ---
                int boxWidth = contentWidth - 4;
                int boxHeight = galleryBoxHeight - 4;
                float imageAspect = (float) data.width / (float) data.height;
                float boxAspect = (float) boxWidth / (float) boxHeight;

                int renderWidth = boxWidth;
                int renderHeight = boxHeight;

                if (imageAspect > boxAspect) {
                    renderHeight = (int) (boxWidth / imageAspect);
                } else {
                    renderWidth = (int) (boxHeight * imageAspect);
                }

                int renderX = contentX + 2 + (boxWidth - renderWidth) / 2;
                int renderY = yPos + 2 + (boxHeight - renderHeight) / 2;

                class_332.method_25290(matrices, renderX, renderY, 0, 0, renderWidth, renderHeight, renderWidth, renderHeight);
                RenderSystem.disableBlend();
            } else {
                method_27534(matrices, field_22793, new class_2585("Error loading image").method_27692(class_124.field_1061), this.field_22789 / 2, yPos + galleryBoxHeight / 2 - 4, 0xFFFFFF);
            }

            String counter = (currentImageIndex + 1) + " / " + build.getImageFileNames().size();
            int counterWidth = field_22793.method_1727(counter);
            field_22793.method_1729(matrices, counter, contentX + contentWidth - counterWidth - 5, yPos + galleryBoxHeight - 12, 0xFFFFFF);

            yPos += galleryBoxHeight + panelSpacing;
        }

        // --- DYNAMIC CONTENT ---
        int descriptionHeight = 80;
        this.field_22793.method_30883(matrices, new class_2585("Description:").method_27692(class_124.field_1080), contentX, yPos, 0xFFFFFF);
        method_25294(matrices, contentX, yPos + labelHeight, contentX + contentWidth, yPos + labelHeight + descriptionHeight, 0x77000000);
        yPos += descriptionHeight + labelHeight + panelSpacing;

        int creditsHeight = 40;
        this.field_22793.method_30883(matrices, new class_2585("Credits:").method_27692(class_124.field_1080), contentX, yPos, 0xFFFFFF);
        method_25294(matrices, contentX, yPos + labelHeight, contentX + contentWidth, yPos + labelHeight + creditsHeight, 0x77000000);
        yPos += creditsHeight + labelHeight + panelSpacing;

        for (CustomField field : build.getCustomFields()) {
            int fieldHeight = 40;
            this.field_22793.method_30883(matrices, new class_2585(field.getTitle() + ":").method_27692(class_124.field_1080), contentX, yPos, 0xFFFFFF);
            method_25294(matrices, contentX, yPos + labelHeight, contentX + contentWidth, yPos + labelHeight + fieldHeight, 0x77000000);
            yPos += fieldHeight + labelHeight + panelSpacing;
        }
    }


    private void switchImage(int direction) {
        int newIndex = this.currentImageIndex + direction;
        if (newIndex >= 0 && newIndex < build.getImageFileNames().size()) {
            this.currentImageIndex = newIndex;
            updateNavButtons();
        }
    }

    private void updateNavButtons() {
        if (prevImageButton != null) {
            prevImageButton.field_22763 = currentImageIndex > 0;
        }
        if (nextImageButton != null) {
            nextImageButton.field_22763 = currentImageIndex < build.getImageFileNames().size() - 1;
        }
    }

    private ImageData getImageDataForCurrentImage() {
        if (build.getImageFileNames().isEmpty()) return null;

        String fileName = build.getImageFileNames().get(currentImageIndex);
        if (textureCache.containsKey(fileName)) {
            return textureCache.get(fileName);
        }

        try {
            Path imagePath = FabricLoader.getInstance().getConfigDir()
                    .resolve("buildnotes")
                    .resolve("images")
                    .resolve(build.getId().toString())
                    .resolve(fileName);

            if (Files.exists(imagePath)) {
                try (InputStream stream = Files.newInputStream(imagePath)) {
                    class_1011 image = class_1011.method_4309(stream);
                    class_1043 texture = new class_1043(image);
                    class_2960 textureId = this.field_22787.method_1531().method_4617("buildnotes_image_" + build.getId() + "_" + fileName.hashCode(), texture);

                    ImageData data = new ImageData(textureId, image.method_4307(), image.method_4323());
                    textureCache.put(fileName, data);
                    return data;
                }
            }
        } catch (Exception e) {
            textureCache.put(fileName, null); // Cache failure
        }
        return null;
    }

    private void confirmDelete() {
        Runnable onConfirm = () -> {
            DataManager.getInstance().deleteBuild(this.build);
            this.method_25419();
        };
        this.showConfirm(new class_2585("Delete build \"" + build.getName() + "\"?"), onConfirm);
    }

    @Override
    public void method_25419() {
        textureCache.values().forEach(data -> {
            if (data != null && data.textureId != null) {
                field_22787.method_1531().method_4615(data.textureId);
            }
        });
        super.method_25419();
    }

}