package net.atif.buildnotes.data;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import net.atif.buildnotes.Buildnotes;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_310;
import net.minecraft.class_5218;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class DataManager {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Path GLOBAL_PATH = FabricLoader.getInstance().getConfigDir();
    private static final String NOTES_FILE_NAME = "notes.json";
    private static final String BUILDS_FILE_NAME = "builds.json";
    private static final String MOD_DATA_SUBFOLDER = "buildnotes";

    private static DataManager instance;

    private DataManager() {}

    public static DataManager getInstance() {
        if (instance == null) {
            instance = new DataManager();
        }
        return instance;
    }

    private Path getWorldSpecificPath() {
        class_310 client = class_310.method_1551();
        // Check if we are in a single-player world
        if (client.method_1496() && client.method_1576() != null) {
            // This is the correct, robust way to get the world's save directory
            return client.method_1576().method_27050(class_5218.field_24188).resolve(MOD_DATA_SUBFOLDER);
        }
        // If not in a world (e.g., on the main menu), return null
        return null;
    }

    public List<Note> getNotes() {
        List<Note> globalNotes = loadNotes(GLOBAL_PATH.resolve(MOD_DATA_SUBFOLDER));
        Path worldPath = getWorldSpecificPath();
        List<Note> worldNotes = new ArrayList<>();
        if (worldPath != null) {
            worldNotes = loadNotes(worldPath);
        }
        return Stream.concat(worldNotes.stream(), globalNotes.stream()).sorted(Comparator.comparingLong(Note::getLastModified).reversed()).collect(Collectors.toList());
    }

    public void saveNote(Note noteToSave) {
        List<Note> allNotes = getNotes();
        // Remove old version if it exists
        allNotes.removeIf(n -> n.getId().equals(noteToSave.getId()));
        allNotes.add(noteToSave); // Add the new/updated version

        // Split notes into global and world-specific lists
        List<Note> globalNotes = allNotes.stream().filter(Note::isGlobal).collect(Collectors.toList());
        List<Note> worldNotes = allNotes.stream().filter(n -> !n.isGlobal()).collect(Collectors.toList());

        // Save them to their respective files
        writeNotesToFile(globalNotes, GLOBAL_PATH.resolve(MOD_DATA_SUBFOLDER));
        Path worldPath = getWorldSpecificPath();
        if (worldPath != null) {
            writeNotesToFile(worldNotes, worldPath);
        }
    }

    public void deleteNote(Note noteToDelete) {
        List<Note> allNotes = getNotes();
        allNotes.removeIf(n -> n.getId().equals(noteToDelete.getId()));

        List<Note> globalNotes = allNotes.stream().filter(Note::isGlobal).collect(Collectors.toList());
        List<Note> worldNotes = allNotes.stream().filter(n -> !n.isGlobal()).collect(Collectors.toList());

        writeNotesToFile(globalNotes, GLOBAL_PATH.resolve(MOD_DATA_SUBFOLDER));
        Path worldPath = getWorldSpecificPath();
        if (worldPath != null) {
            writeNotesToFile(worldNotes, worldPath);
        }
    }


    private void writeNotesToFile(List<Note> notes, Path path) {
        try {
            Files.createDirectories(path);
            try (FileWriter writer = new FileWriter(path.resolve(NOTES_FILE_NAME).toFile())) {
                GSON.toJson(notes, writer);
            }
        } catch (IOException e) {
            Buildnotes.LOGGER.error("Could not save notes to " + path.toString(), e);
        }
    }

    private List<Note> loadNotes(Path path) {
        File notesFile = path.resolve(NOTES_FILE_NAME).toFile();
        if (!notesFile.exists()) {
            return new ArrayList<>();
        }
        try (FileReader reader = new FileReader(notesFile)) {
            Type type = new TypeToken<ArrayList<Note>>() {}.getType();
            List<Note> loadedNotes = GSON.fromJson(reader, type);
            return loadedNotes != null ? loadedNotes : new ArrayList<>();
        } catch (IOException e) {
            Buildnotes.LOGGER.warn("Could not load notes from " + path + ", creating new list...");
            return new ArrayList<>();
        }
    }

    public List<Build> getBuilds() {
        List<Build> globalBuilds = loadBuilds(GLOBAL_PATH.resolve(MOD_DATA_SUBFOLDER));
        Path worldPath = getWorldSpecificPath();
        List<Build> worldBuilds = new ArrayList<>();
        if (worldPath != null) {
            worldBuilds = loadBuilds(worldPath);
        }
        // Sort by last modified timestamp, descending (newest first)
        return Stream.concat(worldBuilds.stream(), globalBuilds.stream()).sorted(Comparator.comparingLong(Build::getLastModified).reversed()).collect(Collectors.toList());
    }

    public void saveBuild(Build buildToSave) {
        List<Build> allBuilds = getBuilds();
        allBuilds.removeIf(b -> b.getId().equals(buildToSave.getId()));
        allBuilds.add(buildToSave);

        List<Build> globalBuilds = allBuilds.stream().filter(Build::isGlobal).collect(Collectors.toList());
        List<Build> worldBuilds = allBuilds.stream().filter(b -> !b.isGlobal()).collect(Collectors.toList());

        writeBuildsToFile(globalBuilds, GLOBAL_PATH.resolve(MOD_DATA_SUBFOLDER));
        Path worldPath = getWorldSpecificPath();
        if (worldPath != null) {
            writeBuildsToFile(worldBuilds, worldPath);
        }
    }

    // NEW METHOD for deleting a single build
    public void deleteBuild(Build buildToDelete) {
        List<Build> allBuilds = getBuilds();
        allBuilds.removeIf(b -> b.getId().equals(buildToDelete.getId()));

        List<Build> globalBuilds = allBuilds.stream().filter(Build::isGlobal).collect(Collectors.toList());
        List<Build> worldBuilds = allBuilds.stream().filter(b -> !b.isGlobal()).collect(Collectors.toList());

        writeBuildsToFile(globalBuilds, GLOBAL_PATH.resolve(MOD_DATA_SUBFOLDER));
        Path worldPath = getWorldSpecificPath();
        if (worldPath != null) {
            writeBuildsToFile(worldBuilds, worldPath);
        }

        try {
            Path imageDir = FabricLoader.getInstance().getConfigDir()
                    .resolve(MOD_DATA_SUBFOLDER)
                    .resolve("images")
                    .resolve(buildToDelete.getId().toString());

            if (Files.exists(imageDir)) {
                // Recursively delete the directory and its contents
                try (Stream<Path> walk = Files.walk(imageDir)) {
                    walk.sorted(Comparator.reverseOrder())
                            .map(Path::toFile)
                            .forEach(File::delete);
                }
            }
        } catch (IOException e) {
            Buildnotes.LOGGER.error("Failed to delete image directory for build: {}", buildToDelete.getId(), e);
        }
    }

    private void writeBuildsToFile(List<Build> builds, Path path) {
        try {
            Files.createDirectories(path);
            try (FileWriter writer = new FileWriter(path.resolve(BUILDS_FILE_NAME).toFile())) {
                GSON.toJson(builds, writer);
            }
        } catch (IOException e) {
            Buildnotes.LOGGER.error("Could not save builds to " + path.toString(), e);
        }
    }

    private List<Build> loadBuilds(Path path) {
        File buildsFile = path.resolve(BUILDS_FILE_NAME).toFile();
        if (!buildsFile.exists()) {
            return new ArrayList<>();
        }
        try (FileReader reader = new FileReader(buildsFile)) {
            Type type = new TypeToken<ArrayList<Build>>() {}.getType();
            List<Build> loadedBuilds = GSON.fromJson(reader, type);
            return loadedBuilds != null ? loadedBuilds : new ArrayList<>();
        } catch (IOException e) {
            Buildnotes.LOGGER.warn("Could not load builds from " + path.toString() + ", creating new list...");
            return new ArrayList<>();
        }
    }
}