/*
 * Decompiled with CFR 0.152.
 */
package fr.ax_dev.universejobs.storage.pool;

import fr.ax_dev.universejobs.UniverseJobs;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import org.bukkit.configuration.file.FileConfiguration;

public class ConnectionPool {
    private final UniverseJobs plugin;
    private final boolean enabled;
    private final String jdbcUrl;
    private final String username;
    private final String password;
    private final int minConnections;
    private final int maxConnections;
    private final long validationIntervalMs;
    private final ConcurrentLinkedQueue<PooledConnection> availableConnections;
    private final AtomicInteger activeConnections;
    private final AtomicInteger totalConnections;
    private final AtomicBoolean initialized;
    private final AtomicLong totalConnectionsCreated;
    private final AtomicLong totalConnectionsDestroyed;
    private final AtomicLong totalGetConnectionCalls;
    private final AtomicLong totalConnectionWaitTime;
    private final ScheduledExecutorService healthChecker;

    public ConnectionPool(UniverseJobs plugin, FileConfiguration config) {
        this.plugin = plugin;
        String databaseType = config.getString("database.type", "");
        this.enabled = !databaseType.isEmpty() && (databaseType.equals("sqlite") || databaseType.equals("mysql"));
        String host = config.getString("database.host", "localhost");
        String port = config.getString("database.port", "3306");
        String prefix = config.getString("database.prefix", "UniverseJobs_");
        this.jdbcUrl = "jdbc:mysql://" + host + ":" + port + "/" + prefix;
        this.username = config.getString("database.username", "UniverseJobs");
        this.password = config.getString("database.password", "your_password_here");
        this.minConnections = config.getInt("database.pool.min-connections", 2);
        this.maxConnections = config.getInt("database.pool.max-connections", 10);
        config.getLong("database.pool.connection-timeout-ms", 30000L);
        this.validationIntervalMs = config.getLong("database.pool.validation-interval-ms", 300000L);
        this.availableConnections = new ConcurrentLinkedQueue();
        this.activeConnections = new AtomicInteger(0);
        this.totalConnections = new AtomicInteger(0);
        this.initialized = new AtomicBoolean(false);
        this.totalConnectionsCreated = new AtomicLong(0L);
        this.totalConnectionsDestroyed = new AtomicLong(0L);
        this.totalGetConnectionCalls = new AtomicLong(0L);
        this.totalConnectionWaitTime = new AtomicLong(0L);
        this.healthChecker = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "UniverseJobs-ConnectionPool-HealthChecker"));
    }

    public void initialize() {
        if (!this.enabled) {
            this.plugin.getLogger().info("Database connection pool disabled");
            return;
        }
        try {
            this.loadDatabaseDriver();
            for (int i = 0; i < this.minConnections; ++i) {
                this.createConnection();
            }
            this.startHealthChecker();
            this.initialized.set(true);
            this.plugin.getLogger().info("Database connection pool initialized with " + this.totalConnections.get() + " connections");
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to initialize database connection pool", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Connection getConnection() throws SQLException {
        if (!this.enabled || !this.initialized.get()) {
            throw new SQLException("Connection pool not available");
        }
        long startTime = System.nanoTime();
        this.totalGetConnectionCalls.incrementAndGet();
        try {
            PooledConnection pooledConnection = this.availableConnections.poll();
            if (pooledConnection == null) {
                if (this.totalConnections.get() < this.maxConnections) {
                    pooledConnection = this.createConnection();
                } else {
                    throw new SQLException("Maximum number of connections reached");
                }
            }
            if (!this.isConnectionValid(pooledConnection)) {
                this.closeConnection(pooledConnection);
                pooledConnection = this.createConnection();
            }
            this.activeConnections.incrementAndGet();
            Connection connection = pooledConnection.getConnection();
            return connection;
        }
        finally {
            long waitTime = System.nanoTime() - startTime;
            this.totalConnectionWaitTime.addAndGet(waitTime);
        }
    }

    public void returnConnection(Connection connection) {
        if (connection == null) {
            return;
        }
        try {
            if (!connection.isClosed()) {
                connection.setAutoCommit(true);
                connection.clearWarnings();
                PooledConnection pooledConnection = new PooledConnection(connection);
                this.availableConnections.offer(pooledConnection);
                this.activeConnections.decrementAndGet();
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.WARNING, "Error returning connection to pool", e);
            this.closeConnectionSafely(connection);
        }
    }

    public void shutdown() {
        PooledConnection connection;
        if (!this.enabled) {
            return;
        }
        this.initialized.set(false);
        if (this.healthChecker != null && !this.healthChecker.isShutdown()) {
            this.healthChecker.shutdown();
            try {
                if (!this.healthChecker.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.healthChecker.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.healthChecker.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        while ((connection = this.availableConnections.poll()) != null) {
            this.closeConnection(connection);
        }
        this.plugin.getLogger().info("Database connection pool shutdown complete. Total connections created: " + this.totalConnectionsCreated.get() + ", destroyed: " + this.totalConnectionsDestroyed.get());
    }

    public boolean isEnabled() {
        return this.enabled && this.initialized.get();
    }

    public boolean isHealthy() {
        if (!this.enabled || !this.initialized.get()) {
            return false;
        }
        return this.totalConnections.get() > 0;
    }

    public int getActiveConnections() {
        return this.activeConnections.get();
    }

    public int getTotalConnections() {
        return this.totalConnections.get();
    }

    public int getAvailableConnections() {
        return this.availableConnections.size();
    }

    public Map<String, Object> getStats() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("enabled", this.enabled);
        stats.put("initialized", this.initialized.get());
        stats.put("active_connections", this.getActiveConnections());
        stats.put("available_connections", this.getAvailableConnections());
        stats.put("total_connections", this.getTotalConnections());
        stats.put("max_connections", this.maxConnections);
        stats.put("min_connections", this.minConnections);
        stats.put("total_connections_created", this.totalConnectionsCreated.get());
        stats.put("total_connections_destroyed", this.totalConnectionsDestroyed.get());
        stats.put("total_get_connection_calls", this.totalGetConnectionCalls.get());
        long calls = this.totalGetConnectionCalls.get();
        double avgWaitTimeMs = calls > 0L ? (double)this.totalConnectionWaitTime.get() / (double)calls / 1000000.0 : 0.0;
        stats.put("avg_connection_wait_time_ms", avgWaitTimeMs);
        double poolEfficiency = calls > 0L ? (double)(calls - this.totalConnectionsCreated.get()) / (double)calls * 100.0 : 0.0;
        stats.put("pool_efficiency_percent", poolEfficiency);
        return stats;
    }

    private PooledConnection createConnection() throws SQLException {
        try {
            Connection connection = DriverManager.getConnection(this.jdbcUrl, this.username, this.password);
            connection.setAutoCommit(true);
            PooledConnection pooledConnection = new PooledConnection(connection);
            this.totalConnections.incrementAndGet();
            this.totalConnectionsCreated.incrementAndGet();
            return pooledConnection;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to create database connection", e);
            throw e;
        }
    }

    private boolean isConnectionValid(PooledConnection pooledConnection) {
        try {
            Connection connection = pooledConnection.getConnection();
            if (connection.isClosed()) {
                return false;
            }
            if (System.currentTimeMillis() - pooledConnection.getCreatedTime() > this.validationIntervalMs) {
                return false;
            }
            return connection.isValid(5);
        }
        catch (SQLException e) {
            return false;
        }
    }

    private void closeConnection(PooledConnection pooledConnection) {
        if (pooledConnection != null) {
            this.closeConnectionSafely(pooledConnection.getConnection());
            this.totalConnections.decrementAndGet();
            this.totalConnectionsDestroyed.incrementAndGet();
        }
    }

    private void closeConnectionSafely(Connection connection) {
        if (connection != null) {
            try {
                if (!connection.isClosed()) {
                    connection.close();
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.WARNING, "Error closing connection", e);
            }
        }
    }

    private void loadDatabaseDriver() throws ClassNotFoundException {
        String driverClass;
        if (this.jdbcUrl.startsWith("jdbc:mysql:")) {
            driverClass = "com.mysql.cj.jdbc.Driver";
        } else if (this.jdbcUrl.startsWith("jdbc:postgresql:")) {
            driverClass = "org.postgresql.Driver";
        } else if (this.jdbcUrl.startsWith("jdbc:sqlite:")) {
            driverClass = "org.sqlite.JDBC";
        } else {
            throw new ClassNotFoundException("Unsupported database URL: " + this.jdbcUrl);
        }
        Class.forName(driverClass);
        this.plugin.getLogger().info("Loaded database driver: " + driverClass);
    }

    private void startHealthChecker() {
        this.healthChecker.scheduleAtFixedRate(() -> {
            try {
                this.cleanupStaleConnections();
                this.ensureMinimumConnections();
                if (this.plugin.getConfigManager().isDebugEnabled()) {
                    this.logHealthStatus();
                }
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.WARNING, "Error in connection pool health check", e);
            }
        }, this.validationIntervalMs / 1000L, this.validationIntervalMs / 1000L, TimeUnit.SECONDS);
    }

    private void cleanupStaleConnections() {
        int cleaned = 0;
        Iterator<PooledConnection> iterator = this.availableConnections.iterator();
        while (iterator.hasNext()) {
            PooledConnection pooledConnection = iterator.next();
            if (this.isConnectionValid(pooledConnection)) continue;
            iterator.remove();
            this.closeConnection(pooledConnection);
            ++cleaned;
        }
        if (cleaned > 0 && this.plugin.getConfigManager().isDebugEnabled()) {
            this.plugin.getLogger().info("Cleaned up " + cleaned + " stale database connections");
        }
    }

    private void ensureMinimumConnections() {
        int currentTotal = this.totalConnections.get();
        int needed = this.minConnections - currentTotal;
        if (needed > 0) {
            try {
                for (int i = 0; i < needed; ++i) {
                    PooledConnection connection = this.createConnection();
                    this.availableConnections.offer(connection);
                }
                if (this.plugin.getConfigManager().isDebugEnabled()) {
                    this.plugin.getLogger().info("Created " + needed + " new database connections to maintain minimum pool size");
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.WARNING, "Failed to create minimum connections", e);
            }
        }
    }

    private void logHealthStatus() {
        Map<String, Object> stats = this.getStats();
        this.plugin.getLogger().info("Connection Pool Health: Active=" + String.valueOf(stats.get("active_connections")) + ", Available=" + String.valueOf(stats.get("available_connections")) + ", Total=" + String.valueOf(stats.get("total_connections")) + ", Created=" + String.valueOf(stats.get("total_connections_created")) + ", Destroyed=" + String.valueOf(stats.get("total_connections_destroyed")) + ", Efficiency=" + String.format("%.1f%%", stats.get("pool_efficiency_percent")));
    }

    private static class PooledConnection {
        private final Connection connection;
        private final long createdTime;

        public PooledConnection(Connection connection) {
            this.connection = connection;
            this.createdTime = System.currentTimeMillis();
        }

        public Connection getConnection() {
            return this.connection;
        }

        public long getCreatedTime() {
            return this.createdTime;
        }
    }
}

