/*
 * Decompiled with CFR 0.152.
 */
package fr.ax_dev.universejobs.rewards;

import fr.ax_dev.universejobs.UniverseJobs;
import fr.ax_dev.universejobs.job.JobManager;
import fr.ax_dev.universejobs.job.PlayerJobData;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.milkbowl.vault.economy.Economy;
import org.bukkit.Bukkit;
import org.bukkit.OfflinePlayer;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;

public class BatchedRewardManager {
    private final UniverseJobs plugin;
    private final JobManager jobManager;
    private final Economy economy;
    private final Map<String, BatchedReward> xpBatch = new ConcurrentHashMap<String, BatchedReward>();
    private final Map<UUID, Double> moneyBatch = new ConcurrentHashMap<UUID, Double>();
    private final Map<String, BatchedCommand> commandBatch = new ConcurrentHashMap<String, BatchedCommand>();
    private ScheduledExecutorService batchExecutor = Executors.newScheduledThreadPool(1, r -> {
        Thread thread = new Thread(r, "UniverseJobs-BatchProcessor");
        thread.setDaemon(true);
        return thread;
    });
    private int xpBatchTicks;
    private int moneyBatchTicks;
    private int othersBatchTicks;
    private long lastXpFlush = System.currentTimeMillis();
    private long lastMoneyFlush = System.currentTimeMillis();
    private long lastOthersFlush = System.currentTimeMillis();

    public BatchedRewardManager(UniverseJobs plugin) {
        this.plugin = plugin;
        this.jobManager = plugin.getJobManager();
        this.economy = this.getVaultEconomy();
        this.xpBatchTicks = plugin.getConfig().getInt("performance.batching-xp", 60);
        this.moneyBatchTicks = plugin.getConfig().getInt("performance.batching-money", 60);
        this.othersBatchTicks = plugin.getConfig().getInt("performance.batching-others", 40);
        this.startBatchProcessors();
    }

    public void batchXp(Player player, String jobId, double xp) {
        if (this.xpBatchTicks <= 0) {
            this.processXpImmediate(player, jobId, xp);
            return;
        }
        String key = String.valueOf(player.getUniqueId()) + ":" + jobId;
        this.xpBatch.compute(key, (k, existing) -> {
            if (existing == null) {
                return new BatchedReward(player.getUniqueId(), player.getName(), jobId, xp);
            }
            existing.addAmount(xp);
            return existing;
        });
        this.checkXpFlush();
    }

    public void batchMoney(Player player, double money) {
        if (this.moneyBatchTicks <= 0) {
            this.processMoneyImmediate(player, money);
            return;
        }
        this.moneyBatch.merge(player.getUniqueId(), money, Double::sum);
        this.checkMoneyFlush();
    }

    public void batchCommand(String command, int delay) {
        if (this.othersBatchTicks <= 0) {
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> Bukkit.dispatchCommand((CommandSender)Bukkit.getConsoleSender(), (String)command), (long)delay);
            return;
        }
        String key = command + ":" + delay;
        this.commandBatch.compute(key, (k, existing) -> {
            if (existing == null) {
                return new BatchedCommand(command, delay);
            }
            existing.incrementCount();
            return existing;
        });
        this.checkOthersFlush();
    }

    private void processXpImmediate(Player player, String jobId, double xp) {
        PlayerJobData data = this.jobManager.getPlayerData(player.getUniqueId());
        if (data != null) {
            data.addXp(jobId, xp);
            this.jobManager.savePlayerData(player.getUniqueId());
        }
    }

    private void processMoneyImmediate(Player player, double money) {
        if (this.economy != null && money > 0.0) {
            this.economy.depositPlayer((OfflinePlayer)player, money);
        }
    }

    private void checkXpFlush() {
        long flushInterval;
        long currentTime = System.currentTimeMillis();
        long timeSinceFlush = currentTime - this.lastXpFlush;
        if (timeSinceFlush >= (flushInterval = (long)this.xpBatchTicks * 50L)) {
            this.flushXpBatch();
            this.lastXpFlush = currentTime;
        }
    }

    private void checkMoneyFlush() {
        long flushInterval;
        long currentTime = System.currentTimeMillis();
        long timeSinceFlush = currentTime - this.lastMoneyFlush;
        if (timeSinceFlush >= (flushInterval = (long)this.moneyBatchTicks * 50L)) {
            this.flushMoneyBatch();
            this.lastMoneyFlush = currentTime;
        }
    }

    private void checkOthersFlush() {
        long flushInterval;
        long currentTime = System.currentTimeMillis();
        long timeSinceFlush = currentTime - this.lastOthersFlush;
        if (timeSinceFlush >= (flushInterval = (long)this.othersBatchTicks * 50L)) {
            this.flushCommandBatch();
            this.lastOthersFlush = currentTime;
        }
    }

    public void flushXpBatch() {
        if (this.xpBatch.isEmpty()) {
            return;
        }
        ConcurrentHashMap<String, BatchedReward> toProcess = new ConcurrentHashMap<String, BatchedReward>(this.xpBatch);
        this.xpBatch.clear();
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
            for (BatchedReward reward : toProcess.values()) {
                PlayerJobData data = this.jobManager.getPlayerData(reward.playerUuid);
                if (data == null) continue;
                data.addXp(reward.jobId, reward.amount);
                if (!this.plugin.getConfigManager().isDebugEnabled()) continue;
                this.plugin.getLogger().info("[BATCH] Processed " + reward.amount + " XP for " + reward.playerName + " in job " + reward.jobId);
            }
            toProcess.values().stream().map(r -> r.playerUuid).distinct().forEach(uuid -> this.jobManager.savePlayerData((UUID)uuid));
        });
    }

    public void flushMoneyBatch() {
        if (this.moneyBatch.isEmpty()) {
            return;
        }
        ConcurrentHashMap<UUID, Double> toProcess = new ConcurrentHashMap<UUID, Double>(this.moneyBatch);
        this.moneyBatch.clear();
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
            if (this.economy != null) {
                for (Map.Entry entry : toProcess.entrySet()) {
                    Player player = Bukkit.getPlayer((UUID)((UUID)entry.getKey()));
                    if (player == null || !player.isOnline()) continue;
                    this.economy.depositPlayer((OfflinePlayer)player, ((Double)entry.getValue()).doubleValue());
                    if (!this.plugin.getConfigManager().isDebugEnabled()) continue;
                    this.plugin.getLogger().info("[BATCH] Processed $" + String.valueOf(entry.getValue()) + " for " + player.getName());
                }
            }
        });
    }

    public void flushCommandBatch() {
        if (this.commandBatch.isEmpty()) {
            return;
        }
        ConcurrentHashMap<String, BatchedCommand> toProcess = new ConcurrentHashMap<String, BatchedCommand>(this.commandBatch);
        this.commandBatch.clear();
        for (BatchedCommand cmd : toProcess.values()) {
            for (int i = 0; i < cmd.count; ++i) {
                Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> Bukkit.dispatchCommand((CommandSender)Bukkit.getConsoleSender(), (String)cmd.command), (long)(cmd.delay + i * 2));
            }
        }
    }

    private void startBatchProcessors() {
        if (this.xpBatchTicks > 0) {
            long xpInterval = (long)this.xpBatchTicks * 50L;
            this.batchExecutor.scheduleAtFixedRate(this::flushXpBatch, xpInterval, xpInterval, TimeUnit.MILLISECONDS);
        }
        if (this.moneyBatchTicks > 0) {
            long moneyInterval = (long)this.moneyBatchTicks * 50L;
            this.batchExecutor.scheduleAtFixedRate(this::flushMoneyBatch, moneyInterval, moneyInterval, TimeUnit.MILLISECONDS);
        }
        if (this.othersBatchTicks > 0) {
            long othersInterval = (long)this.othersBatchTicks * 50L;
            this.batchExecutor.scheduleAtFixedRate(this::flushCommandBatch, othersInterval, othersInterval, TimeUnit.MILLISECONDS);
        }
    }

    public void shutdown() {
        this.flushXpBatch();
        this.flushMoneyBatch();
        this.flushCommandBatch();
        this.batchExecutor.shutdown();
        try {
            if (!this.batchExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.batchExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.batchExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public void forceFlushAll() {
        this.flushXpBatch();
        this.flushMoneyBatch();
        this.flushCommandBatch();
    }

    public void reloadConfig() {
        int newXpBatchTicks = this.plugin.getConfig().getInt("performance.batching-xp", 60);
        int newMoneyBatchTicks = this.plugin.getConfig().getInt("performance.batching-money", 60);
        int newOthersBatchTicks = this.plugin.getConfig().getInt("performance.batching-others", 40);
        this.batchExecutor.shutdown();
        try {
            if (!this.batchExecutor.awaitTermination(2L, TimeUnit.SECONDS)) {
                this.batchExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.batchExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
        this.batchExecutor = Executors.newScheduledThreadPool(1, r -> {
            Thread thread = new Thread(r, "UniverseJobs-BatchProcessor");
            thread.setDaemon(true);
            return thread;
        });
        this.xpBatchTicks = newXpBatchTicks;
        this.moneyBatchTicks = newMoneyBatchTicks;
        this.othersBatchTicks = newOthersBatchTicks;
        this.startBatchProcessors();
        this.plugin.getLogger().info("BatchedRewardManager reloaded - XP: " + newXpBatchTicks + " ticks, Money: " + newMoneyBatchTicks + " ticks, Others: " + newOthersBatchTicks + " ticks");
    }

    public BatchStatistics getStatistics() {
        return new BatchStatistics(this.xpBatch.size(), this.moneyBatch.size(), this.commandBatch.size(), this.xpBatch.values().stream().mapToDouble(r -> r.amount).sum(), this.moneyBatch.values().stream().mapToDouble(Double::doubleValue).sum());
    }

    private Economy getVaultEconomy() {
        try {
            if (this.plugin.getServer().getServicesManager().getRegistration(Economy.class) != null) {
                return (Economy)this.plugin.getServer().getServicesManager().getRegistration(Economy.class).getProvider();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    private static class BatchedCommand {
        final String command;
        final int delay;
        int count = 1;

        BatchedCommand(String command, int delay) {
            this.command = command;
            this.delay = delay;
        }

        void incrementCount() {
            ++this.count;
        }
    }

    public static class BatchStatistics {
        public final int xpBatchSize;
        public final int moneyBatchSize;
        public final int commandBatchSize;
        public final double totalXpPending;
        public final double totalMoneyPending;

        BatchStatistics(int xpBatchSize, int moneyBatchSize, int commandBatchSize, double totalXpPending, double totalMoneyPending) {
            this.xpBatchSize = xpBatchSize;
            this.moneyBatchSize = moneyBatchSize;
            this.commandBatchSize = commandBatchSize;
            this.totalXpPending = totalXpPending;
            this.totalMoneyPending = totalMoneyPending;
        }
    }

    private static class BatchedReward {
        final UUID playerUuid;
        final String playerName;
        final String jobId;
        double amount;

        BatchedReward(UUID playerUuid, String playerName, String jobId, double amount) {
            this.playerUuid = playerUuid;
            this.playerName = playerName;
            this.jobId = jobId;
            this.amount = amount;
        }

        void addAmount(double additional) {
            this.amount += additional;
        }
    }
}

