/*
 * Decompiled with CFR 0.152.
 */
package fr.ax_dev.universejobs.storage.database;

import fr.ax_dev.universejobs.UniverseJobs;
import fr.ax_dev.universejobs.job.PlayerJobData;
import fr.ax_dev.universejobs.storage.DataStorage;
import fr.ax_dev.universejobs.storage.SqlIdentifierValidator;
import fr.ax_dev.universejobs.storage.dao.JobStatsDao;
import fr.ax_dev.universejobs.storage.dao.LeaderboardDao;
import fr.ax_dev.universejobs.storage.dao.PlayerDataDao;
import fr.ax_dev.universejobs.storage.dao.RewardDao;
import fr.ax_dev.universejobs.storage.database.DatabaseConfig;
import fr.ax_dev.universejobs.storage.database.DatabaseSchema;
import fr.ax_dev.universejobs.storage.database.HikariConnectionPool;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;

public class DatabaseDataStorage
implements DataStorage {
    private final UniverseJobs plugin;
    private final DatabaseConfig config;
    private final HikariConnectionPool connectionPool;
    private final DatabaseSchema schema;
    private final PlayerDataDao playerDataDao;
    private final RewardDao rewardDao;
    private final JobStatsDao jobStatsDao;
    private final LeaderboardDao leaderboardDao;
    private final Map<UUID, PlayerJobData> cache = new ConcurrentHashMap<UUID, PlayerJobData>();
    private final Map<UUID, Set<String>> rewardCache = new ConcurrentHashMap<UUID, Set<String>>();
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final AtomicBoolean shutdown = new AtomicBoolean(false);
    private long cacheHits = 0L;
    private long cacheMisses = 0L;
    private long totalOperations = 0L;

    public DatabaseDataStorage(UniverseJobs plugin) {
        this.plugin = plugin;
        this.config = new DatabaseConfig(plugin.getConfig().getConfigurationSection("database"));
        this.connectionPool = new HikariConnectionPool(plugin, this.config);
        this.schema = new DatabaseSchema(plugin, this.config, this.connectionPool);
        this.playerDataDao = new PlayerDataDao(this.connectionPool, this.config);
        this.rewardDao = new RewardDao(this.connectionPool, this.config);
        this.jobStatsDao = new JobStatsDao(this.connectionPool, this.config);
        this.leaderboardDao = new LeaderboardDao(plugin, this.connectionPool, this.config);
    }

    @Override
    public CompletableFuture<Void> initializeAsync() {
        return CompletableFuture.runAsync(() -> {
            if (this.initialized.compareAndSet(false, true)) {
                try {
                    this.connectionPool.initialize();
                    this.schema.initializeSchema();
                    String playerDataTable = SqlIdentifierValidator.buildSafeTableName(this.config.getPrefix(), "player_data");
                    this.leaderboardDao.syncLeaderboardFromPlayerData(playerDataTable).join();
                    this.plugin.getLogger().info("Database storage initialized successfully");
                }
                catch (Exception e) {
                    this.initialized.set(false);
                    this.plugin.getLogger().log(Level.SEVERE, "Failed to initialize database storage", e);
                    throw new RuntimeException("Failed to initialize database storage", e);
                }
            }
        });
    }

    @Override
    public CompletableFuture<Void> shutdownAsync() {
        return CompletableFuture.runAsync(() -> {
            if (this.shutdown.compareAndSet(false, true)) {
                try {
                    this.cache.clear();
                    this.rewardCache.clear();
                    this.connectionPool.shutdown();
                }
                catch (Exception e) {
                    this.plugin.getLogger().log(Level.WARNING, "Error during database storage shutdown", e);
                }
            }
        });
    }

    @Override
    public CompletableFuture<Void> savePlayerDataAsync(UUID playerId, PlayerJobData data) {
        if (this.shutdown.get()) {
            return CompletableFuture.completedFuture(null);
        }
        ++this.totalOperations;
        this.cache.put(playerId, data);
        CompletableFuture<Void> saveFuture = this.playerDataDao.savePlayerData(playerId, data);
        saveFuture.thenRun(() -> {
            String playerName = this.plugin.getServer().getOfflinePlayer(playerId).getName();
            if (playerName == null) {
                playerName = "Unknown";
            }
            for (String jobId : data.getJobs()) {
                double xp = data.getXp(jobId);
                int level = data.getLevel(jobId);
                this.leaderboardDao.updatePlayerLeaderboardEntry(playerId, playerName, jobId, xp, level);
            }
        });
        return saveFuture;
    }

    @Override
    public CompletableFuture<PlayerJobData> loadPlayerDataAsync(UUID playerId) {
        if (this.shutdown.get()) {
            return CompletableFuture.failedFuture(new IllegalStateException("Storage is shutdown"));
        }
        ++this.totalOperations;
        PlayerJobData cachedData = this.cache.get(playerId);
        if (cachedData != null) {
            ++this.cacheHits;
            return CompletableFuture.completedFuture(cachedData);
        }
        ++this.cacheMisses;
        return this.playerDataDao.loadPlayerData(playerId).thenApply(data -> {
            this.cache.put(playerId, (PlayerJobData)data);
            return data;
        });
    }

    @Override
    public CompletableFuture<Void> saveBatchPlayerData(Map<UUID, PlayerJobData> playerDataMap) {
        if (this.shutdown.get()) {
            return CompletableFuture.failedFuture(new IllegalStateException("Storage is shutdown"));
        }
        CompletableFuture[] futures = (CompletableFuture[])playerDataMap.entrySet().stream().map(entry -> this.savePlayerDataAsync((UUID)entry.getKey(), (PlayerJobData)entry.getValue())).toArray(CompletableFuture[]::new);
        return CompletableFuture.allOf(futures);
    }

    @Override
    public CompletableFuture<Map<UUID, PlayerJobData>> loadBatchPlayerData(Set<UUID> playerIds) {
        if (this.shutdown.get()) {
            return CompletableFuture.failedFuture(new IllegalStateException("Storage is shutdown"));
        }
        HashMap<UUID, PlayerJobData> result = new HashMap<UUID, PlayerJobData>();
        HashSet<UUID> toLoad = new HashSet<UUID>();
        for (UUID playerId : playerIds) {
            PlayerJobData cachedData = this.cache.get(playerId);
            if (cachedData != null) {
                result.put(playerId, cachedData);
                ++this.cacheHits;
                continue;
            }
            toLoad.add(playerId);
            ++this.cacheMisses;
        }
        if (toLoad.isEmpty()) {
            return CompletableFuture.completedFuture(result);
        }
        return this.playerDataDao.loadPlayerDataBatch(toLoad).thenApply(loadedData -> {
            this.cache.putAll((Map<UUID, PlayerJobData>)loadedData);
            result.putAll((Map<UUID, PlayerJobData>)loadedData);
            return result;
        });
    }

    @Override
    public CompletableFuture<Void> preloadPlayerData(Set<UUID> playerIds) {
        return this.loadBatchPlayerData(playerIds).thenAccept(data -> this.plugin.getLogger().info("Preloaded " + data.size() + " player data entries"));
    }

    @Override
    public void evictFromCache(UUID playerId) {
        this.cache.remove(playerId);
        this.rewardCache.remove(playerId);
    }

    @Override
    public void clearCache() {
        this.cache.clear();
        this.rewardCache.clear();
        this.cacheHits = 0L;
        this.cacheMisses = 0L;
    }

    @Override
    public Map<String, Object> getCacheStats() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("cached_players", this.cache.size());
        stats.put("cached_rewards", this.rewardCache.size());
        stats.put("cache_hits", this.cacheHits);
        stats.put("cache_misses", this.cacheMisses);
        stats.put("hit_ratio", this.totalOperations > 0L ? (double)this.cacheHits / (double)this.totalOperations : 0.0);
        return stats;
    }

    @Override
    public Map<String, Object> getPerformanceMetrics() {
        HashMap<String, Object> metrics = new HashMap<String, Object>();
        metrics.put("total_operations", this.totalOperations);
        metrics.put("cache_hits", this.cacheHits);
        metrics.put("cache_misses", this.cacheMisses);
        metrics.put("pool_active", this.connectionPool.getDataSource() != null ? this.connectionPool.getDataSource().getHikariPoolMXBean().getActiveConnections() : 0);
        metrics.put("pool_idle", this.connectionPool.getDataSource() != null ? this.connectionPool.getDataSource().getHikariPoolMXBean().getIdleConnections() : 0);
        return metrics;
    }

    @Override
    public void resetPerformanceMetrics() {
        this.totalOperations = 0L;
        this.cacheHits = 0L;
        this.cacheMisses = 0L;
    }

    @Override
    public boolean isHealthy() {
        return this.initialized.get() && !this.shutdown.get() && this.connectionPool.isInitialized();
    }

    @Override
    public Map<String, Object> getHealthInfo() {
        HashMap<String, Object> health = new HashMap<String, Object>();
        health.put("initialized", this.initialized.get());
        health.put("shutdown", this.shutdown.get());
        health.put("pool_initialized", this.connectionPool.isInitialized());
        health.put("database_type", this.config.getType().getName());
        health.put("cache_size", this.cache.size());
        return health;
    }

    @Override
    public void initialize() {
        this.initializeAsync().join();
    }

    @Override
    public void shutdown() {
        this.shutdownAsync().join();
    }

    @Override
    public boolean hasClaimedReward(UUID playerId, String jobId, String rewardId) {
        Set<String> playerRewards = this.rewardCache.get(playerId);
        if (playerRewards != null) {
            return playerRewards.contains(jobId + ":" + rewardId);
        }
        try {
            return this.rewardDao.hasClaimedReward(playerId, jobId, rewardId).get();
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to check reward claim", e);
            return false;
        }
    }

    @Override
    public void claimReward(UUID playerId, String jobId, String rewardId, long claimTime) {
        this.rewardCache.computeIfAbsent(playerId, k -> ConcurrentHashMap.newKeySet()).add(jobId + ":" + rewardId);
        this.rewardDao.claimReward(playerId, jobId, rewardId, claimTime).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.WARNING, "Failed to save reward claim to database", (Throwable)throwable);
            return null;
        });
    }

    @Override
    public long getClaimTime(UUID playerId, String jobId, String rewardId) {
        try {
            return this.rewardDao.getClaimTime(playerId, jobId, rewardId).get();
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to get claim time", e);
            return -1L;
        }
    }

    @Override
    public Set<String> getClaimedRewards(UUID playerId, String jobId) {
        try {
            return this.rewardDao.getClaimedRewards(playerId, jobId).get();
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to get claimed rewards", e);
            return new HashSet<String>();
        }
    }

    @Override
    public Set<String> getAllClaimedRewards(UUID playerId) {
        try {
            return this.rewardDao.getAllClaimedRewards(playerId).get();
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to get all claimed rewards", e);
            return new HashSet<String>();
        }
    }

    @Override
    public void resetRewardClaim(UUID playerId, String jobId, String rewardId) {
        Set<String> playerRewards = this.rewardCache.get(playerId);
        if (playerRewards != null) {
            playerRewards.remove(jobId + ":" + rewardId);
        }
        this.rewardDao.resetRewardClaim(playerId, jobId, rewardId).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.WARNING, "Failed to reset reward claim in database", (Throwable)throwable);
            return null;
        });
    }

    @Override
    public void resetJobRewards(UUID playerId, String jobId) {
        Set<String> playerRewards = this.rewardCache.get(playerId);
        if (playerRewards != null) {
            playerRewards.removeIf(reward -> reward.startsWith(jobId + ":"));
        }
        this.rewardDao.resetJobRewards(playerId, jobId).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.WARNING, "Failed to reset job rewards in database", (Throwable)throwable);
            return null;
        });
    }

    @Override
    public void resetAllRewards(UUID playerId) {
        this.rewardCache.remove(playerId);
        this.rewardDao.resetAllRewards(playerId).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.WARNING, "Failed to reset all rewards in database", (Throwable)throwable);
            return null;
        });
    }

    @Override
    public void save() {
    }

    @Override
    public void loadPlayerData(UUID playerId) {
        this.loadPlayerDataAsync(playerId).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.WARNING, "Failed to load player data for " + String.valueOf(playerId), (Throwable)throwable);
            return new PlayerJobData(playerId);
        });
    }

    @Override
    public void unloadPlayerData(UUID playerId) {
        this.evictFromCache(playerId);
    }

    public PlayerDataDao getPlayerDataDao() {
        return this.playerDataDao;
    }

    public RewardDao getRewardDao() {
        return this.rewardDao;
    }

    public LeaderboardDao getLeaderboardDao() {
        return this.leaderboardDao;
    }

    public JobStatsDao getJobStatsDao() {
        return this.jobStatsDao;
    }

    @Override
    public double getJobUserCount(String jobId) {
        try {
            return this.playerDataDao.getJobUserCount(jobId).get();
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to get job user count for " + jobId, e);
            return 0.0;
        }
    }

    @Override
    public PlayerJobData getPlayerData(UUID playerId) {
        PlayerJobData cached = this.cache.get(playerId);
        if (cached != null) {
            ++this.cacheHits;
            return cached;
        }
        ++this.cacheMisses;
        try {
            PlayerJobData data = this.loadPlayerDataAsync(playerId).get();
            this.cache.put(playerId, data);
            return data;
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to load player data for " + String.valueOf(playerId), e);
            PlayerJobData newData = new PlayerJobData(playerId);
            this.cache.put(playerId, newData);
            return newData;
        }
    }

    @Override
    public CompletableFuture<Void> deletePlayerData(UUID playerId) {
        return CompletableFuture.runAsync(() -> {
            this.cache.remove(playerId);
            this.rewardCache.remove(playerId);
            try {
                this.playerDataDao.deletePlayerData(playerId).get();
                this.rewardDao.deletePlayerRewards(playerId).get();
                this.leaderboardDao.removeFromLeaderboard(playerId).get();
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to delete player data for " + String.valueOf(playerId), e);
                throw new RuntimeException("Failed to delete player data", e);
            }
        });
    }

    @Override
    public CompletableFuture<List<LeaderboardDao.LeaderboardEntry>> getJobLeaderboard(String jobId, int limit) {
        return this.leaderboardDao.getJobLeaderboard(jobId, limit);
    }
}

