/*
 * Decompiled with CFR 0.152.
 */
package fr.ax_dev.universejobs.utils;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.File;
import java.io.FileReader;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Method;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Base64;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.bukkit.Bukkit;
import org.bukkit.profile.PlayerProfile;

public class PlayerTextureCache {
    private static final Map<UUID, PlayerProfile> PROFILE_CACHE = new ConcurrentHashMap<UUID, PlayerProfile>();
    private static final Map<String, UUID> NAME_TO_UUID_CACHE = new ConcurrentHashMap<String, UUID>();
    private static final Map<UUID, String> UUID_TO_NAME_CACHE = new ConcurrentHashMap<UUID, String>();
    private static final Map<UUID, Long> CACHE_TIMESTAMPS = new ConcurrentHashMap<UUID, Long>();
    private static final long CACHE_DURATION = TimeUnit.HOURS.toMillis(24L);
    private static final int MAX_CACHE_SIZE = 1000;
    private static volatile boolean userCacheLoaded = false;
    private static volatile long lastUserCacheLoad = 0L;
    private static final long USER_CACHE_RELOAD_INTERVAL = TimeUnit.MINUTES.toMillis(5L);
    private static volatile Boolean hasPaperProfileApi = null;
    private static Method paperCreateProfileMethod = null;
    private static Method paperFillFromCacheMethod = null;

    public static void init() {
        PlayerTextureCache.loadUserCache();
        PlayerTextureCache.detectPaperProfileApi();
    }

    private static void detectPaperProfileApi() {
        if (hasPaperProfileApi != null) {
            return;
        }
        try {
            Class serverClass = Bukkit.getServer().getClass();
            paperCreateProfileMethod = serverClass.getMethod("createProfile", UUID.class, String.class);
            Class<?> profileClass = Class.forName("com.destroystokyo.paper.profile.PlayerProfile");
            paperFillFromCacheMethod = profileClass.getMethod("complete", Boolean.TYPE);
            hasPaperProfileApi = true;
            Bukkit.getLogger().info("[UniverseJobs] Paper PlayerProfile API detected - using optimized player head caching");
        }
        catch (Exception e) {
            hasPaperProfileApi = false;
        }
    }

    private static PlayerProfile tryPaperCacheFirst(UUID playerId, String playerName) {
        if (hasPaperProfileApi == null) {
            PlayerTextureCache.detectPaperProfileApi();
        }
        if (!Boolean.TRUE.equals(hasPaperProfileApi)) {
            return null;
        }
        try {
            Method getTexturesMethod;
            Object textures;
            Method getSkinMethod;
            Object skin;
            Object paperProfile = paperCreateProfileMethod.invoke((Object)Bukkit.getServer(), playerId, playerName);
            Boolean completed = (Boolean)paperFillFromCacheMethod.invoke(paperProfile, false);
            if (completed.booleanValue() && (skin = (getSkinMethod = (textures = (getTexturesMethod = paperProfile.getClass().getMethod("getTextures", new Class[0])).invoke(paperProfile, new Object[0])).getClass().getMethod("getSkin", new Class[0])).invoke(textures, new Object[0])) != null) {
                PlayerProfile bukkitProfile = Bukkit.createPlayerProfile((UUID)playerId, (String)playerName);
                bukkitProfile.getTextures().setSkin((URL)skin);
                return bukkitProfile;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    private static void loadUserCache() {
        if (System.currentTimeMillis() - lastUserCacheLoad < USER_CACHE_RELOAD_INTERVAL && userCacheLoaded) {
            return;
        }
        CompletableFuture.runAsync(() -> {
            block11: {
                try {
                    File userCacheFile = new File(Bukkit.getWorldContainer(), "usercache.json");
                    if (!userCacheFile.exists()) {
                        userCacheFile = new File("usercache.json");
                    }
                    if (!userCacheFile.exists()) break block11;
                    try (FileReader reader = new FileReader(userCacheFile);){
                        JsonArray array = JsonParser.parseReader((Reader)reader).getAsJsonArray();
                        for (JsonElement element : array) {
                            JsonObject entry = element.getAsJsonObject();
                            if (!entry.has("uuid") || !entry.has("name")) continue;
                            String uuidStr = entry.get("uuid").getAsString();
                            String name = entry.get("name").getAsString();
                            try {
                                UUID uuid = UUID.fromString(uuidStr);
                                NAME_TO_UUID_CACHE.put(name.toLowerCase(), uuid);
                                UUID_TO_NAME_CACHE.put(uuid, name);
                            }
                            catch (IllegalArgumentException illegalArgumentException) {}
                        }
                    }
                    userCacheLoaded = true;
                    lastUserCacheLoad = System.currentTimeMillis();
                }
                catch (Exception e) {
                    Bukkit.getLogger().log(Level.WARNING, "Failed to load usercache.json", e);
                }
            }
        });
    }

    public static CompletableFuture<PlayerProfile> getPlayerProfile(UUID playerId, String playerName) {
        PlayerProfile cached;
        PlayerTextureCache.loadUserCache();
        Long timestamp = CACHE_TIMESTAMPS.get(playerId);
        if (timestamp != null && System.currentTimeMillis() - timestamp < CACHE_DURATION && (cached = PROFILE_CACHE.get(playerId)) != null && cached.getTextures().getSkin() != null) {
            return CompletableFuture.completedFuture(cached);
        }
        PlayerProfile paperCached = PlayerTextureCache.tryPaperCacheFirst(playerId, playerName);
        if (paperCached != null && paperCached.getTextures().getSkin() != null) {
            PlayerTextureCache.cacheProfile(playerId, paperCached);
            return CompletableFuture.completedFuture(paperCached);
        }
        return CompletableFuture.supplyAsync(() -> {
            try {
                PlayerProfile profile = Bukkit.createPlayerProfile((UUID)playerId, (String)playerName);
                profile.update().get(5L, TimeUnit.SECONDS);
                if (profile.getTextures().getSkin() != null) {
                    PlayerTextureCache.cacheProfile(playerId, profile);
                    return profile;
                }
                return PlayerTextureCache.fetchFromMojangApi(playerId, playerName);
            }
            catch (Exception e) {
                return PlayerTextureCache.fetchFromMojangApi(playerId, playerName);
            }
        });
    }

    public static CompletableFuture<PlayerProfile> getPlayerProfileByName(String playerName) {
        PlayerTextureCache.loadUserCache();
        UUID cachedUuid = NAME_TO_UUID_CACHE.get(playerName.toLowerCase());
        if (cachedUuid != null) {
            PlayerProfile cached;
            Long timestamp = CACHE_TIMESTAMPS.get(cachedUuid);
            if (timestamp != null && System.currentTimeMillis() - timestamp < CACHE_DURATION && (cached = PROFILE_CACHE.get(cachedUuid)) != null && cached.getTextures().getSkin() != null) {
                return CompletableFuture.completedFuture(cached);
            }
            return PlayerTextureCache.getPlayerProfile(cachedUuid, playerName);
        }
        return CompletableFuture.supplyAsync(() -> PlayerTextureCache.fetchFromMojangApiByName(playerName));
    }

    public static String getPlayerName(UUID playerId) {
        PlayerTextureCache.loadUserCache();
        return UUID_TO_NAME_CACHE.get(playerId);
    }

    public static UUID getPlayerUuid(String playerName) {
        PlayerTextureCache.loadUserCache();
        return NAME_TO_UUID_CACHE.get(playerName.toLowerCase());
    }

    private static PlayerProfile fetchFromMojangApi(UUID playerId, String playerName) {
        try {
            UUID mojangUuid = NAME_TO_UUID_CACHE.get(playerName.toLowerCase());
            if (mojangUuid == null) {
                mojangUuid = PlayerTextureCache.fetchMojangUuid(playerName);
            }
            if (mojangUuid != null) {
                NAME_TO_UUID_CACHE.put(playerName.toLowerCase(), mojangUuid);
                UUID_TO_NAME_CACHE.put(mojangUuid, playerName);
                return PlayerTextureCache.fetchProfileWithTextures(mojangUuid, playerName);
            }
        }
        catch (Exception mojangUuid) {
            // empty catch block
        }
        PlayerProfile fallback = Bukkit.createPlayerProfile((UUID)playerId, (String)playerName);
        PlayerTextureCache.cacheProfile(playerId, fallback);
        return fallback;
    }

    private static PlayerProfile fetchFromMojangApiByName(String playerName) {
        try {
            UUID mojangUuid = NAME_TO_UUID_CACHE.get(playerName.toLowerCase());
            if (mojangUuid == null) {
                mojangUuid = PlayerTextureCache.fetchMojangUuid(playerName);
            }
            if (mojangUuid != null) {
                NAME_TO_UUID_CACHE.put(playerName.toLowerCase(), mojangUuid);
                UUID_TO_NAME_CACHE.put(mojangUuid, playerName);
                return PlayerTextureCache.fetchProfileWithTextures(mojangUuid, playerName);
            }
        }
        catch (Exception mojangUuid) {
            // empty catch block
        }
        UUID offlineUuid = UUID.nameUUIDFromBytes(("OfflinePlayer:" + playerName).getBytes());
        return Bukkit.createPlayerProfile((UUID)offlineUuid, (String)playerName);
    }

    private static UUID fetchMojangUuid(String playerName) {
        block7: {
            UUID uUID;
            URL url = new URL("https://api.mojang.com/users/profiles/minecraft/" + playerName);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestMethod("GET");
            connection.setConnectTimeout(3000);
            connection.setReadTimeout(3000);
            if (connection.getResponseCode() != 200) break block7;
            InputStreamReader reader = new InputStreamReader(connection.getInputStream());
            try {
                JsonObject json = JsonParser.parseReader((Reader)reader).getAsJsonObject();
                String uuidStr = json.get("id").getAsString();
                uUID = PlayerTextureCache.parseUuidWithoutDashes(uuidStr);
            }
            catch (Throwable throwable) {
                try {
                    try {
                        reader.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            reader.close();
            return uUID;
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static PlayerProfile fetchProfileWithTextures(UUID mojangUuid, String playerName) {
        block10: {
            try {
                URL url = new URL("https://sessionserver.mojang.com/session/minecraft/profile/" + mojangUuid.toString().replace("-", "") + "?unsigned=false");
                HttpURLConnection connection = (HttpURLConnection)url.openConnection();
                connection.setRequestMethod("GET");
                connection.setConnectTimeout(3000);
                connection.setReadTimeout(3000);
                if (connection.getResponseCode() != 200) break block10;
                try (InputStreamReader reader = new InputStreamReader(connection.getInputStream());){
                    JsonObject json = JsonParser.parseReader((Reader)reader).getAsJsonObject();
                    if (json.has("properties")) {
                        for (JsonElement prop : json.getAsJsonArray("properties")) {
                            JsonObject propObj = prop.getAsJsonObject();
                            if (!"textures".equals(propObj.get("name").getAsString())) continue;
                            String textureValue = propObj.get("value").getAsString();
                            PlayerProfile playerProfile = PlayerTextureCache.createProfileFromTexture(mojangUuid, playerName, textureValue);
                            return playerProfile;
                        }
                    }
                }
            }
            catch (Exception url) {
                // empty catch block
            }
        }
        PlayerProfile profile = Bukkit.createPlayerProfile((UUID)mojangUuid, (String)playerName);
        PlayerTextureCache.cacheProfile(mojangUuid, profile);
        return profile;
    }

    private static PlayerProfile createProfileFromTexture(UUID uuid, String name, String textureValue) {
        try {
            String decoded = new String(Base64.getDecoder().decode(textureValue));
            JsonObject textureJson = JsonParser.parseString((String)decoded).getAsJsonObject();
            JsonObject textures = textureJson.getAsJsonObject("textures");
            if (textures.has("SKIN")) {
                String skinUrl = textures.getAsJsonObject("SKIN").get("url").getAsString();
                PlayerProfile profile = Bukkit.createPlayerProfile((UUID)uuid, (String)name);
                profile.getTextures().setSkin(new URL(skinUrl));
                PlayerTextureCache.cacheProfile(uuid, profile);
                return profile;
            }
        }
        catch (Exception decoded) {
            // empty catch block
        }
        PlayerProfile profile = Bukkit.createPlayerProfile((UUID)uuid, (String)name);
        PlayerTextureCache.cacheProfile(uuid, profile);
        return profile;
    }

    private static UUID parseUuidWithoutDashes(String uuidStr) {
        String withDashes = uuidStr.replaceFirst("(\\w{8})(\\w{4})(\\w{4})(\\w{4})(\\w{12})", "$1-$2-$3-$4-$5");
        return UUID.fromString(withDashes);
    }

    private static void cacheProfile(UUID playerId, PlayerProfile profile) {
        if (PROFILE_CACHE.size() >= 1000) {
            PlayerTextureCache.cleanOldestEntries();
        }
        PROFILE_CACHE.put(playerId, profile);
        CACHE_TIMESTAMPS.put(playerId, System.currentTimeMillis());
    }

    private static void cleanOldestEntries() {
        long oldestTime = System.currentTimeMillis();
        UUID oldestUuid = null;
        for (Map.Entry<UUID, Long> entry : CACHE_TIMESTAMPS.entrySet()) {
            if (entry.getValue() >= oldestTime) continue;
            oldestTime = entry.getValue();
            oldestUuid = entry.getKey();
        }
        if (oldestUuid != null) {
            PROFILE_CACHE.remove(oldestUuid);
            CACHE_TIMESTAMPS.remove(oldestUuid);
        }
    }

    public static void clearCache() {
        PROFILE_CACHE.clear();
        CACHE_TIMESTAMPS.clear();
        NAME_TO_UUID_CACHE.clear();
        UUID_TO_NAME_CACHE.clear();
        userCacheLoaded = false;
    }

    public static void removeFromCache(UUID playerId) {
        PROFILE_CACHE.remove(playerId);
        CACHE_TIMESTAMPS.remove(playerId);
    }

    public static int getCacheSize() {
        return PROFILE_CACHE.size();
    }

    public static int getNameCacheSize() {
        return NAME_TO_UUID_CACHE.size();
    }
}

