-- SQL Server compatibility settings (uncomment if using SQL Server)
-- SET QUOTED_IDENTIFIER ON;
-- GO

-- UniverseJobs Database Schema
-- Cross-database compatible schema for player job data and rewards
-- 
-- Supported databases: H2, SQLite, PostgreSQL, MySQL/MariaDB, SQL Server
-- This schema uses standard SQL syntax for maximum compatibility

-- Create database (if using MySQL/PostgreSQL)
-- CREATE DATABASE IF NOT EXISTS UniverseJobs;
-- USE UniverseJobs;

-- Main player job data table
CREATE TABLE IF NOT EXISTS player_job_data (
    player_uuid VARCHAR(36) PRIMARY KEY,
    data TEXT NOT NULL,
    compressed BOOLEAN DEFAULT FALSE,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Player reward claim data table
CREATE TABLE IF NOT EXISTS player_reward_claims (
    id BIGINT GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
    player_uuid VARCHAR(36) NOT NULL,
    job_id VARCHAR(64) NOT NULL,
    reward_id VARCHAR(128) NOT NULL,
    claim_time TIMESTAMP NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Unique constraint to prevent duplicate claims (unless repeatable)
    CONSTRAINT uk_player_job_reward UNIQUE (player_uuid, job_id, reward_id)
);

-- Performance statistics table
CREATE TABLE IF NOT EXISTS performance_stats (
    id BIGINT GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
    metric_name VARCHAR(128) NOT NULL,
    metric_value DECIMAL(15,4),
    metric_type VARCHAR(20) DEFAULT 'gauge' CHECK (metric_type IN ('counter', 'gauge', 'timer', 'histogram')),
    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Server configuration table
CREATE TABLE IF NOT EXISTS server_config (
    config_key VARCHAR(128) PRIMARY KEY,
    config_value TEXT,
    config_type VARCHAR(20) DEFAULT 'string' CHECK (config_type IN ('string', 'number', 'boolean', 'json')),
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Migration version tracking
CREATE TABLE IF NOT EXISTS schema_migrations (
    version VARCHAR(32) PRIMARY KEY,
    applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    description TEXT
);

-- Insert initial migration record (handled in Java code to avoid syntax differences)

-- Create indexes for performance
CREATE INDEX IF NOT EXISTS idx_player_job_data_last_updated ON player_job_data (last_updated);
CREATE INDEX IF NOT EXISTS idx_player_job_data_created_at ON player_job_data (created_at);

CREATE INDEX IF NOT EXISTS idx_player_reward_claims_player_uuid ON player_reward_claims (player_uuid);
CREATE INDEX IF NOT EXISTS idx_player_reward_claims_job_id ON player_reward_claims (job_id);
CREATE INDEX IF NOT EXISTS idx_player_reward_claims_claim_time ON player_reward_claims (claim_time);
CREATE INDEX IF NOT EXISTS idx_player_reward_claims_player_job ON player_reward_claims (player_uuid, job_id);

CREATE INDEX IF NOT EXISTS idx_performance_stats_metric_name ON performance_stats (metric_name);
CREATE INDEX IF NOT EXISTS idx_performance_stats_timestamp ON performance_stats (timestamp);
CREATE INDEX IF NOT EXISTS idx_performance_stats_metric_type ON performance_stats (metric_type);

CREATE INDEX IF NOT EXISTS idx_server_config_last_updated ON server_config (last_updated);

-- Database initialization complete
-- All tables and indexes created with cross-database compatibility