package com.debugger.entity.screen;

import com.debugger.entity.EntityDebugger;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.minecraft.class_124;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_2479;
import net.minecraft.class_2481;
import net.minecraft.class_2487;
import net.minecraft.class_2489;
import net.minecraft.class_2494;
import net.minecraft.class_2495;
import net.minecraft.class_2497;
import net.minecraft.class_2499;
import net.minecraft.class_2501;
import net.minecraft.class_2503;
import net.minecraft.class_2516;
import net.minecraft.class_2519;
import net.minecraft.class_2520;
import net.minecraft.class_2522;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_342;
import net.minecraft.class_410;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import net.minecraft.class_746;
import net.minecraft.class_7919;
import net.minecraft.nbt.*;
import org.lwjgl.glfw.GLFW;
import java.util.*;

public class EntityDebugScreen extends class_437 {
    private DropdownWidget activeDropdown;
    // 特殊NBT标签集合
    private static final Set<String> SPECIAL_TAGS = Set.of(
            "Pos", "Rotation", "Motion",
            "Health", "Air", "Fire",
            "Invulnerable", "PortalCooldown",
            "CustomName", "NoGravity"
    );
    // 类型颜色配置 - 更新为更现代化的颜色
    private static final Map<Byte, Integer> TYPE_COLORS = Map.ofEntries(
            Map.entry(class_2520.field_33250, 0xFF555555),
            Map.entry(class_2520.field_33251, 0xFF4287f5),
            Map.entry(class_2520.field_33252, 0xFF4CAF50),
            Map.entry(class_2520.field_33253, 0xFFFFC107),
            Map.entry(class_2520.field_33254, 0xFF9C27B0),
            Map.entry(class_2520.field_33255, 0xFFE91E63),
            Map.entry(class_2520.field_33256, 0xFFF44336),
            Map.entry(class_2520.field_33257, 0xFF2196F3),
            Map.entry(class_2520.field_33258, 0xFF8BC34A),
            Map.entry(class_2520.field_33259, 0xFF00BCD4),
            Map.entry(class_2520.field_33260, 0xFF795548),
            Map.entry(class_2520.field_33261, 0xFFFF9800),
            Map.entry(class_2520.field_33262, 0xFF673AB7)
    );

    // UI组件
    private class_342 activeEditor;
    private class_4185 prevButton, nextButton;

    // 数据状态
    private String editingKey;
    private final Map<String, class_2520> nbtMap = new LinkedHashMap<>();
    private final List<String> nbtKeys = new ArrayList<>();
    private int currentPage = 1;
    private int itemsPerPage;
    private final class_1297 entity;

    // 类型选择相关
    private byte selectedType = class_2520.field_33258;
    private final List<String> visibleKeys = new ArrayList<>();
    private class_342 addKeyField;
    private class_342 addValueField;

    // 修改状态
    private final Map<String, class_2520> modifiedValues = new HashMap<>();
    private boolean hasUnsavedChanges = false;

    // 图标资源

    // 颜色常量
    private static final int PANEL_COLOR = 0xFF2D2D2D;
    private static final int HIGHLIGHT_COLOR = 0xFF3A3A3A;
    private static final int TEXT_COLOR = 0xFFFFFFFF;
    private static final int SECONDARY_TEXT_COLOR = 0xFFAAAAAA;
    private static final int ACCENT_COLOR = 0xFF4CAF50;
    private static final int WARNING_COLOR = 0xFFFFA000;
    // 在类顶部添加这些常量
    private static final int MARGIN = 10;
    private static final int PANEL_PADDING = 5;
    private static final int ENTRY_HEIGHT = 20;
    private static final int HEADER_HEIGHT = 45;

    public EntityDebugScreen(class_1297 entity) {
        super(class_2561.method_43471("entity_debugger.screen.title"));
        this.entity = entity;
    }

    @Override
    protected void method_25426() {
        super.method_25426();
        // 计算每页显示的项目数
        calculatePageSize();

        // 初始化分页控件
        createPaginationControls();

        // 添加NBT按钮 - 使用更现代的风格
        this.method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.add_nbt"), button -> openAddNbtDialog())
                .method_46434(field_22789 - 110, 15, 90, 24)
                .method_46436(class_7919.method_47407(class_2561.method_43471("entity_debugger.screen.add_nbt.tooltip")))
                .method_46431());

        // 保存所有按钮 - 使用强调色
        this.method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.save_all"), button -> {
                    if (activeEditor != null) {
                        saveEditedValue();
                    }
                    saveAllChanges();
                })
                .method_46434(field_22789 - 110, 45, 90, 24)
                .method_46436(class_7919.method_47407(class_2561.method_43471("entity_debugger.screen.save_all.tooltip")))
                .method_46431());

        // 加载数据
        reloadNbtData();
    }

    @Override
    public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
        // 渲染自定义背景
        renderCustomBackground(context);

        // 渲染标题面板
        renderHeaderPanel(context);

        // 渲染NBT条目
        renderNbtEntries(context);

        // 渲染分页信息
        renderPagination(context);

        // 如果有活动的编辑器，渲染它
        if (activeEditor != null) {
            activeEditor.method_25394(context, mouseX, mouseY, delta);
        }

        super.method_25394(context, mouseX, mouseY, delta);
    }

    private void renderCustomBackground(class_332 context) {
        // 半透明暗色背景
        context.method_25294(0, 0, field_22789, field_22790, 0xCC000000);

        // 主内容区域背景
        int panelX = 10;
        int panelY = 10;
        int panelWidth = field_22789 - 20;
        int panelHeight = field_22790 - 20;
        context.method_25294(panelX, panelY, panelX + panelWidth, panelY + panelHeight, PANEL_COLOR);

        // 边框
        context.method_25294(panelX, panelY, panelX + panelWidth, panelY + 1, ACCENT_COLOR); // 上边框
        context.method_25294(panelX, panelY + panelHeight - 1, panelX + panelWidth, panelY + panelHeight, ACCENT_COLOR); // 下边框
        context.method_25294(panelX, panelY, panelX + 1, panelY + panelHeight, ACCENT_COLOR); // 左边框
        context.method_25294(panelX + panelWidth - 1, panelY, panelX + panelWidth, panelY + panelHeight, ACCENT_COLOR); // 右边框
    }

    private void renderHeaderPanel(class_332 context) {
        // 标题背景
        context.method_25294(15, 15, field_22789 - 15, 45, HIGHLIGHT_COLOR);

        // 标题文本
        context.method_27534(field_22793,
                class_2561.method_43471("entity_debugger.screen.title").method_27692(class_124.field_1067),
                field_22789 / 2, 25, TEXT_COLOR);

        // 实体信息
        context.method_27535(field_22793,
                class_2561.method_43470(entity.method_5477().getString() + " - " + entity.method_5845())
                        .method_27692(class_124.field_1056),
                25, 50, SECONDARY_TEXT_COLOR);

        // 如果有未保存的修改，显示提示
        if (hasUnsavedChanges) {
            context.method_27535(field_22793,
                    class_2561.method_43471("entity_debugger.screen.unsaved_changes").method_27692(class_124.field_1054),
                    field_22789 - 120, 75, WARNING_COLOR);
        }
    }

    private void renderNbtEntries(class_332 context) {
        int startY = HEADER_HEIGHT + 15; // 从标题下方开始

        int startIdx = (currentPage - 1) * itemsPerPage;
        int endIdx = Math.min(startIdx + itemsPerPage, nbtKeys.size());

        // 渲染条目背景
        for (int i = 0; i < endIdx - startIdx; i++) {
            int y = startY + i * ENTRY_HEIGHT;
            int color = i % 2 == 0 ? PANEL_COLOR : HIGHLIGHT_COLOR;
            context.method_25294(MARGIN + PANEL_PADDING, y,
                    field_22789 - MARGIN - PANEL_PADDING,
                    y + ENTRY_HEIGHT - 2, color);
        }

        // 渲染条目内容
        for (int i = startIdx; i < endIdx; i++) {
            String key = nbtKeys.get(i);
            class_2520 value = nbtMap.get(key);
            int yPos = startY + (i - startIdx) * ENTRY_HEIGHT + 6;

            // 渲染键名
            context.method_51439(field_22793,
                    class_2561.method_43470(key).method_27692(class_124.field_1054),
                    MARGIN + PANEL_PADDING + 5, yPos, TEXT_COLOR, false);

            // 渲染等号
            int equalsX = MARGIN + PANEL_PADDING + 5 + field_22793.method_1727(key) + 5;
            context.method_51433(field_22793, "=",
                    equalsX, yPos, SECONDARY_TEXT_COLOR, false);

            // 渲染值（带类型颜色）
            int valueX = equalsX + 10;
            int color = TYPE_COLORS.getOrDefault(value.method_10711(), TEXT_COLOR);
            context.method_51439(field_22793, formatNbtValue(value),
                    valueX, yPos, color, false);
        }
    }

    private void renderPagination(class_332 context) {
        // 分页控件背景
        context.method_25294(20, field_22790 - 40, field_22789 - 20, field_22790 - 15, HIGHLIGHT_COLOR);

        // 分页文本
        String pageInfo = class_2561.method_43469("entity_debugger.screen.page_info", currentPage, getTotalPages()).getString();
        context.method_25300(field_22793, pageInfo,
                field_22789 / 2, field_22790 - 30, TEXT_COLOR);
    }

    private void openAddNbtDialog() {
        class_437 addNbtScreen = new class_437(class_2561.method_43469("entity_debugger.screen.add_dialog.title", getTypeName(selectedType))) {
            // 尺寸常量
            private static final int DIALOG_WIDTH = 320;
            private static final int DIALOG_HEIGHT = 220; // 增加高度避免挤压
            private static final int PADDING = 20;

            // 坐标计算（动态获取）
            private int getDialogX() { return (field_22789 - DIALOG_WIDTH) / 2; }
            private int getDialogY() { return (field_22790 - DIALOG_HEIGHT) / 2; }

            // 垂直位置（从上到下计算）
            private int getTitleY() { return getDialogY() + 10; }
            private int getTypeButtonsY() { return getDialogY() + 40; }
            private int getKeyLabelY() { return getDialogY() + 75; } // 下移10px
            private int getKeyFieldY() { return getKeyLabelY() + 15; }
            private int getValueLabelY() { return getKeyFieldY() + 30; } // 增加间距
            private int getValueFieldY() { return getValueLabelY() + 15; }
            private int getButtonsY() { return getDialogY() + DIALOG_HEIGHT - 30; }

            @Override
            protected void method_25426() {
                // 类型按钮
                method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.type.byte"), b -> updateType(class_2520.field_33251))
                        .method_46434(getDialogX() + PADDING, getTypeButtonsY() - 5, 80, 20).method_46431());

                method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.type.int"), b -> updateType(class_2520.field_33253))
                        .method_46434(getDialogX() + PADDING + 100, getTypeButtonsY() - 5, 80, 20).method_46431());

                method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.type.float"), b -> updateType(class_2520.field_33255))
                        .method_46434(getDialogX() + PADDING + 200, getTypeButtonsY() - 5, 80, 20).method_46431());

                // 键名输入框（下移10px）
                addKeyField = new class_342(field_22793,
                        getDialogX() + PADDING,
                        getKeyFieldY(),
                        DIALOG_WIDTH - 2*PADDING,
                        20,
                        class_2561.method_43473()
                );
                addKeyField.method_47404(class_2561.method_43471("entity_debugger.screen.add_dialog.key_placeholder"));
                method_37063(addKeyField);

                // 值输入框
                addValueField = new class_342(field_22793,
                        getDialogX() + PADDING,
                        getValueFieldY(),
                        DIALOG_WIDTH - 2*PADDING,
                        20,
                        class_2561.method_43473()
                );
                updateValueHint();
                method_37063(addValueField);

                // 操作按钮
                method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.add_dialog.cancel"), b -> method_25419())
                        .method_46434(getDialogX() + PADDING, getButtonsY(), 120, 24).method_46431());

                method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.add_dialog.confirm"), b -> trySubmit())
                        .method_46434(getDialogX() + DIALOG_WIDTH - PADDING - 120, getButtonsY(), 120, 24).method_46431());
            }

            @Override
            public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
                // 背景
                method_25420(context);
                context.method_25294(0, 0, field_22789, field_22790, 0x80000000);

                // 对话框
                int x = getDialogX();
                int y = getDialogY();
                context.method_25294(x, y, x + DIALOG_WIDTH, y + DIALOG_HEIGHT, 0xFF222222);
                context.method_25294(x, y, x + DIALOG_WIDTH, y + 30, 0xFF333333);

                // 标题
                context.method_27534(field_22793,
                        class_2561.method_43469("entity_debugger.screen.add_dialog.title", getTypeName(selectedType)),
                        x + DIALOG_WIDTH/2, getTitleY(), 0xFFFFFF);

                // 标签
                context.method_51439(field_22793, class_2561.method_43471("entity_debugger.screen.add_dialog.key"),
                        x + PADDING, getKeyLabelY(), 0xAAAAAA, false);

                context.method_51439(field_22793, class_2561.method_43471("entity_debugger.screen.add_dialog.value"),
                        x + PADDING, getValueLabelY(), 0xAAAAAA, false);

                // 重要：必须调用super.render()来渲染所有子组件
                super.method_25394(context, mouseX, mouseY, delta);
            }

            private void trySubmit() {
                if (validateInput()) {
                    addNewNbtEntry(addKeyField.method_1882(), addValueField.method_1882());
                    method_25419();
                }
            }

            private boolean validateInput() {
                if (addKeyField.method_1882().isEmpty()) {
                    addKeyField.method_25355(class_2561.method_43471("entity_debugger.message.key_empty").method_27692(class_124.field_1061));
                    return false;
                }

                try {
                    switch (selectedType) {
                        case class_2520.field_33251 -> {
                            if (addValueField.method_1882().equalsIgnoreCase("true") ||
                                    addValueField.method_1882().equalsIgnoreCase("false")) return true;
                            Byte.parseByte(addValueField.method_1882());
                        }
                        case class_2520.field_33253 -> Integer.parseInt(addValueField.method_1882());
                        case class_2520.field_33255 -> Float.parseFloat(addValueField.method_1882());
                    }
                } catch (NumberFormatException e) {
                    addValueField.method_25355(class_2561.method_43469("entity_debugger.message.invalid_value", getTypeName(selectedType))
                            .method_27692(class_124.field_1061));
                    return false;
                }
                return true;
            }

            private void updateType(byte type) {
                selectedType = type;
                updateValueHint();
            }

            private void updateValueHint() {
                String hint = class_2561.method_43471("entity_debugger.screen.add_dialog.value_hint." + selectedType).getString();
                addValueField.method_47404(class_2561.method_43470(hint));
            }

            private String getTypeName(byte nbtType) {
                return class_2561.method_43469("nbt.type." + switch (nbtType) {
                    case class_2520.field_33251 -> "byte";
                    case class_2520.field_33253 -> "int";
                    case class_2520.field_33255 -> "float";
                    default -> "unknown";
                }, nbtType).getString();
            }

            // ... (其他方法保持不变，包括trySubmit/validateInput等)
        };

        // 重要：必须在设置屏幕前获取client实例
        if (this.field_22787 != null) {
            this.field_22787.method_1507(addNbtScreen);
        }
    }
    private void openListEditor(class_2499 list, String key) {
        class_2499 workingCopy = list.method_10612();
        field_22787.method_1507(new class_437(class_2561.method_43469("entity_debugger.screen.list_editor.title", key, list.size())) {
            private NbtListEditorWidget editor;
            private class_4185 formatToggle;
            private boolean prettyFormat = true;

            @Override
            protected void method_25426() {
                super.method_25426();

                // 添加格式化切换按钮
                this.formatToggle = class_4185.method_46430(
                        class_2561.method_43471(prettyFormat ? "entity_debugger.screen.list_editor.raw_format" : "entity_debugger.screen.list_editor.pretty_format"),
                        button -> {
                            prettyFormat = !prettyFormat;
                            formatToggle.method_25355(class_2561.method_43471(prettyFormat ? "entity_debugger.screen.list_editor.raw_format" : "entity_debugger.screen.list_editor.pretty_format"));
                            updateEditorContent();
                        }
                ).method_46434(field_22789 - 120, 10, 100, 20).method_46431();
                method_37063(formatToggle);

                // 创建增强版多行文本编辑器
                this.editor = new NbtListEditorWidget(
                        field_22793,
                        20, 40,
                        field_22789 - 40, field_22790 - 100,
                        class_2561.method_43470("编辑NBT列表内容")
                );
                editor.method_1880(Integer.MAX_VALUE);
                updateEditorContent();
                method_37063(editor);

                // 取消按钮
                method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.list_editor.cancel"), button -> returnToMainScreen())
                        .method_46434(field_22789/2 - 105, field_22790 - 50, 100, 20).method_46431());

                // 保存按钮
                method_37063(class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.list_editor.save"), button -> {
                    try {
                        class_2520 parsed = parseInputValue(editor.method_1882(), class_2520.field_33259);
                        if (parsed instanceof class_2499) {
                            workingCopy.clear();
                            workingCopy.addAll((class_2499)parsed);
                            returnToMainScreen();
                        }
                    } catch (Exception e) {
                        showError(class_2561.method_43469("entity_debugger.message.parse_error", e.getMessage()).getString());
                    }
                }).method_46434(field_22789/2 + 5, field_22790 - 50, 100, 20).method_46431());
            }

            @Override
            public void method_25394(class_332 context, int mouseX, int mouseY, float delta) {
                // 半透明背景
                context.method_25294(0, 0, field_22789, field_22790, 0x80000000);

                // 主编辑器背景
                context.method_25294(10, 10, field_22789 - 10, field_22790 - 10, PANEL_COLOR);

                // 标题栏
                context.method_25294(10, 10, field_22789 - 10, 40, HIGHLIGHT_COLOR);
                context.method_27534(field_22793,
                        class_2561.method_43469("entity_debugger.screen.list_editor.title", key, list.size()),
                        field_22789 / 2, 20, TEXT_COLOR);

                // 编辑器背景
                context.method_25294(20, 40, field_22789 - 20, field_22790 - 60, 0xFF1E1E1E);

                // 帮助文本
                context.method_51439(field_22793,
                        class_2561.method_43471("entity_debugger.screen.list_editor.help"),
                        20, field_22790 - 30, SECONDARY_TEXT_COLOR, false);

                super.method_25394(context, mouseX, mouseY, delta);
            }

            private void updateEditorContent() {
                if (prettyFormat) {
                    editor.method_1852(formatNbtListPretty(workingCopy));
                } else {
                    editor.method_1852(workingCopy.method_10714());
                }
            }

            private void returnToMainScreen() {
                // 检查是否有修改
                if (!workingCopy.equals(list)) {
                    modifiedValues.put(key, workingCopy.method_10612());
                    hasUnsavedChanges = true;
                    showSuccess(class_2561.method_43471("entity_debugger.message.list_changes_staged").getString());
                }
                // 返回到主编辑界面
                field_22787.method_1507(EntityDebugScreen.this);
            }

            @Override
            public void method_25419() {
                returnToMainScreen();
            }

            private String formatNbtListPretty(class_2499 list) {
                StringBuilder sb = new StringBuilder("[\n");
                for (int i = 0; i < list.size(); i++) {
                    sb.append("  ").append(formatNbtElementPretty(list.method_10534(i), 1));
                    if (i < list.size() - 1) sb.append(",");
                    sb.append("\n");
                }
                return sb.append("]").toString();
            }

            private String formatNbtElementPretty(class_2520 element, int indent) {
                String indentStr = "  ".repeat(indent);
                if (element instanceof class_2487 compound) {
                    StringBuilder sb = new StringBuilder("{\n");
                    int i = 0;
                    for (String key : compound.method_10541()) {
                        sb.append(indentStr).append("  ").append(key).append(": ")
                                .append(formatNbtElementPretty(compound.method_10580(key), indent + 1));
                        if (++i < compound.method_10546()) sb.append(",");
                        sb.append("\n");
                    }
                    return sb.append(indentStr).append("}").toString();
                } else if (element instanceof class_2499 list) {
                    StringBuilder sb = new StringBuilder("[\n");
                    for (int i = 0; i < list.size(); i++) {
                        sb.append(indentStr).append("  ")
                                .append(formatNbtElementPretty(list.method_10534(i), indent + 1));
                        if (i < list.size() - 1) sb.append(",");
                        sb.append("\n");
                    }
                    return sb.append(indentStr).append("]").toString();
                } else {
                    return element.method_10714();
                }
            }
        });
    }

    private void addNewNbtEntry(String key, String value) {
        if (key.isEmpty()) {
            showError(class_2561.method_43471("entity_debugger.message.key_empty").getString());
            return;
        }
        if (nbtMap.containsKey(key)) {
            showError(class_2561.method_43469("entity_debugger.message.key_exists", key).getString());
            return;
        }

        try {
            class_2520 newValue = createNbtElement(value, selectedType);

            // 更新本地缓存
            nbtMap.put(key, newValue);
            nbtKeys.add(key);

            // 标记为已修改
            modifiedValues.put(key, newValue);  // 添加到修改列表
            hasUnsavedChanges = true;          // 标记有未保存修改

            // 更新UI但不立即保存到实体
            updateVisibleItems();

            showSuccess(class_2561.method_43469("entity_debugger.message.add_success", key).getString());
        } catch (InvalidNbtFormatException e) {
            showError(e.getMessage());
        }
    }

    private class_2520 createNbtElement(String value, byte type) throws InvalidNbtFormatException {
        value = value.trim();
        try {
            switch (type) {
                case class_2520.field_33251:
                    if (value.equalsIgnoreCase("true")) return class_2481.field_21027;
                    if (value.equalsIgnoreCase("false")) return class_2481.field_21026;
                    return class_2481.method_23233(Byte.parseByte(value));

                case class_2520.field_33252:
                    if (value.endsWith("s") || value.endsWith("S")) {
                        value = value.substring(0, value.length() - 1);
                    }
                    return class_2516.method_23254(Short.parseShort(value));

                case class_2520.field_33253:
                    return class_2497.method_23247(Integer.parseInt(value));

                case class_2520.field_33254:
                    if (value.endsWith("L") || value.endsWith("l")) {
                        value = value.substring(0, value.length() - 1);
                    }
                    return class_2503.method_23251(Long.parseLong(value));

                case class_2520.field_33255:
                    if (value.endsWith("F") || value.endsWith("f")) {
                        value = value.substring(0, value.length() - 1);
                    }
                    return class_2494.method_23244(Float.parseFloat(value));

                case class_2520.field_33256:
                    return class_2489.method_23241(Double.parseDouble(value));

                case class_2520.field_33258:
                    return class_2519.method_23256(value);

                case class_2520.field_33257:
                    if (!value.startsWith("[B;")) value = "[B;" + value;
                    if (!value.endsWith("]")) value = value + "]";
                    return class_2522.method_10718(value);

                case class_2520.field_33261:
                    if (!value.startsWith("[I;")) value = "[I;" + value;
                    if (!value.endsWith("]")) value = value + "]";
                    return class_2522.method_10718(value);

                case class_2520.field_33262:
                    if (!value.startsWith("[L;")) value = "[L;" + value;
                    if (!value.endsWith("]")) value = value + "]";
                    return class_2522.method_10718(value);

                case class_2520.field_33259:
                    value = value.trim();
                    if (!value.startsWith("[")) value = "[" + value;
                    if (!value.endsWith("]")) value = value + "]";

                    // 处理浮点数f后缀
                    value = value.replaceAll("([0-9\\.]+)[fF]([,}\\]])", "$1$2");

                    // 处理简单数组默认转为整数列表
                    if (!value.startsWith("[I;") && !value.startsWith("[B;") &&
                            !value.startsWith("[L;") && !value.startsWith("[{") &&
                            !value.startsWith("[\"")) {
                        value = "[I;" + value.substring(1);
                    }

                    return class_2522.method_10718(value);

                case class_2520.field_33260:
                    if (!value.startsWith("{")) value = "{" + value;
                    if (!value.endsWith("}")) value = value + "}";
                    return class_2522.method_10718(value);

                default:
                    throw new InvalidNbtFormatException("不支持的NBT类型: " + getTypeName(type));
            }
        } catch (Exception e) {
            throw new InvalidNbtFormatException(getTypeName(type) + "类型解析失败: " + e.getMessage());
        }
    }

    private class_2520 parseNbtList(String input) throws Exception {
        input = input.trim();

        // 自动补全方括号
        if (!input.startsWith("[")) input = "[" + input;
        if (!input.endsWith("]")) input = input + "]";

        // 处理空列表
        if (input.equals("[]")) return new class_2499();

        // 如果是标准NBT格式（如[I;1,2,3]），直接解析
        if (input.startsWith("[I;") || input.startsWith("[B;") ||
                input.startsWith("[L;") || input.startsWith("[D;") ||
                input.startsWith("[F;") || input.startsWith("[{") ||
                input.startsWith("[\"")) {
            return class_2522.method_10718(input);
        }

        // 处理嵌套结构和复杂元素
        return parseUniversalList(input);
    }

    private class_2520 parseUniversalList(String input) throws Exception {
        String content = input.substring(1, input.length() - 1).trim();
        if (content.isEmpty()) return new class_2499();

        class_2499 result = new class_2499();
        int position = 0;
        int braceLevel = 0;  // 大括号层级
        int bracketLevel = 0; // 方括号层级
        StringBuilder currentElement = new StringBuilder();

        while (position < content.length()) {
            char c = content.charAt(position);

            // 处理嵌套结构
            if (c == '{') braceLevel++;
            if (c == '}') braceLevel--;
            if (c == '[') bracketLevel++;
            if (c == ']') bracketLevel--;

            // 遇到逗号且不在嵌套结构中，分割元素
            if (c == ',' && braceLevel == 0 && bracketLevel == 0) {
                addUniversalElement(result, currentElement.toString().trim());
                currentElement = new StringBuilder();
            } else {
                currentElement.append(c);
            }

            position++;
        }

        // 添加最后一个元素
        if (currentElement.length() > 0) {
            addUniversalElement(result, currentElement.toString().trim());
        }

        return result;
    }

    private void addUniversalElement(class_2499 list, String element) throws Exception {
        if (element.isEmpty()) return;

        try {
            // 处理嵌套列表
            if (element.startsWith("[")) {
                list.add(parseNbtList(element));
                return;
            }

            // 处理复合标签
            if (element.startsWith("{")) {
                list.add(class_2522.method_10718(element));
                return;
            }

            // 处理所有基本类型
            if (element.equalsIgnoreCase("true")) {
                list.add(class_2481.field_21027);
            } else if (element.equalsIgnoreCase("false")) {
                list.add(class_2481.field_21026);
            }
            // 处理字节(带b后缀)
            else if (element.matches("^-?\\d+[bB]$")) {
                list.add(class_2481.method_23233(Byte.parseByte(element.substring(0, element.length()-1))));
            }
            // 处理短整型(带s后缀)
            else if (element.matches("^-?\\d+[sS]$")) {
                list.add(class_2516.method_23254(Short.parseShort(element.substring(0, element.length()-1))));
            }
            // 处理长整型(带l后缀)
            else if (element.matches("^-?\\d+[lL]$")) {
                list.add(class_2503.method_23251(Long.parseLong(element.substring(0, element.length()-1))));
            }
            // 处理double类型(带d后缀或科学计数法)
            else if (element.matches("^-?\\d+\\.\\d+([dD]|([eE][-+]?\\d+))?$")) {
                list.add(class_2489.method_23241(Double.parseDouble(element.replaceAll("[dD]$", ""))));
            }
            // 处理float类型(带f后缀)
            else if (element.matches("^-?\\d+\\.?\\d*[fF]$")) {
                list.add(class_2494.method_23244(Float.parseFloat(element.substring(0, element.length()-1))));
            }
            // 处理普通整数
            else if (element.matches("^-?\\d+$")) {
                list.add(class_2497.method_23247(Integer.parseInt(element)));
            }
            // 处理普通浮点数(无后缀)
            else if (element.matches("^-?\\d+\\.\\d+$")) {
                list.add(class_2489.method_23241(Double.parseDouble(element)));
            }
            // 处理字符串
            else if (element.startsWith("\"") && element.endsWith("\"")) {
                list.add(class_2519.method_23256(element.substring(1, element.length()-1)));
            } else {
                throw new Exception("无法识别的列表元素: " + element);
            }
        } catch (NumberFormatException e) {
            throw new Exception("数字格式错误: " + element);
        }
    }


    private void createPaginationControls() {
        // 统一按钮样式
        int buttonWidth = 30;  // 加宽按钮
        int buttonHeight = 20;
        int spacing = 10;     // 按钮间距

        // 底部居中布局
        int totalWidth = buttonWidth * 2 + spacing + field_22793.method_1727("第1/1页");
        int startX = (field_22789 - totalWidth) / 2;
        int buttonY = field_22790 - 37;

        // 前一页按钮
        this.prevButton = class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.previous_page"), btn -> {
                    cleanupEditing();
                    currentPage = Math.max(1, currentPage - 1);
                    updateVisibleItems();
                })
                .method_46434(startX, buttonY, buttonWidth, buttonHeight)
                .method_46436(class_7919.method_47407(class_2561.method_43471("entity_debugger.screen.previous_page.tooltip")))
                .method_46431();

        // 页码文本位置
        int textX = startX + buttonWidth + spacing/2;

        // 后一页按钮
        this.nextButton = class_4185.method_46430(class_2561.method_43471("entity_debugger.screen.next_page"), btn -> {
                    cleanupEditing();
                    currentPage = Math.min(getTotalPages(), currentPage + 1);
                    updateVisibleItems();
                })
                .method_46434(textX + field_22793.method_1727(class_2561.method_43469("entity_debugger.screen.page_info", 1, 1).getString()) + spacing/2,
                        buttonY, buttonWidth, buttonHeight)
                .method_46436(class_7919.method_47407(class_2561.method_43471("entity_debugger.screen.next_page.tooltip")))
                .method_46431();

        method_37063(prevButton);
        method_37063(nextButton);
    }

    private void updateButtonStates() {
        // 添加空指针检查
        if (prevButton != null && nextButton != null) {
            prevButton.field_22763 = currentPage > 1;
            nextButton.field_22763 = currentPage < getTotalPages();
        }
    }

    private void updateVisibleItems() {
        // 重新排序键列表
        nbtKeys.sort(String::compareToIgnoreCase);

        // 更新分页
        int start = (currentPage - 1) * itemsPerPage;
        int end = Math.min(start + itemsPerPage, nbtKeys.size());

        visibleKeys.clear();
        for (int i = start; i < end; i++) {
            visibleKeys.add(nbtKeys.get(i));
        }
        updateButtonStates();
    }

    // === 核心功能方法 ===

    /**
     * 重新从实体加载NBT数据
     */
    private void reloadNbtData() {
        class_2487 nbt = entity.method_5647(new class_2487());
        nbtMap.clear();
        nbtKeys.clear();

        // 保持原始顺序但排序关键字段
        nbt.method_10541().stream()
                .sorted(Comparator.comparing(key -> {
                    if (key.equals("id")) return 0;
                    if (key.equals("Pos")) return 1;
                    if (key.equals("Rotation")) return 2;
                    return 3;
                }))
                .forEach(key -> {
                    nbtMap.put(key, nbt.method_10580(key));
                    nbtKeys.add(key);
                });

        calculatePageSize();
        updateVisibleItems();
    }

    /**
     * 计算每页显示项数
     */
    private void calculatePageSize() {
        itemsPerPage = Math.max(1, (field_22790 - 80) / 20 - 1); // 保留底部空间
    }

    private class_2561 formatNbtValue(class_2520 value) {
        class_124 color = class_124.method_534(TYPE_COLORS.getOrDefault(value.method_10711(), 0xFFFFFFFF));
        String text = switch (value.method_10711()) {
            case class_2520.field_33258 -> "\"" + value.method_10714() + "\"";
            case class_2520.field_33260 -> "{...}";
            case class_2520.field_33259 -> "List[" + ((class_2499)value).size() + "]";
            case class_2520.field_33257 -> "ByteArray[" + ((class_2479)value).method_10521().length + "]";
            case class_2520.field_33261 -> "IntArray[" + ((class_2495)value).method_10588().length + "]";
            case class_2520.field_33262 -> "LongArray[" + ((class_2501)value).method_10615().length + "]";
            case class_2520.field_33251 ->
                    value.equals(class_2481.field_21027) ? "true" :
                            value.equals(class_2481.field_21026) ? "false" :
                                    value.method_10714();
            case class_2520.field_33252 -> value.method_10714(); // 已经包含s后缀
            case class_2520.field_33255 -> value.method_10714();
            case class_2520.field_33254 -> value.method_10714();
            default -> value.method_10714();
        };
        return class_2561.method_43470(text).method_27692(color);
    }

    // === 交互处理方法 ===

    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        // 先检查是否点击了UI组件（包括保存按钮）
        if (super.method_25402(mouseX, mouseY, button)) {
            return true;
        }

        // 处理下拉框点击
        if (activeDropdown != null && activeDropdown.parentMouseClicked(mouseX, mouseY, button)) {
            return true;
        }

        // 检查是否点击了保存按钮区域 - 先于NBT条目检查
        if (mouseX >= field_22789 - 110 && mouseX <= field_22789 - 20 &&
                mouseY >= 15 && mouseY <= 69) { // 覆盖添加NBT和保存所有按钮区域
            // 让按钮正常处理点击
            return false;
        }

        // 检查是否点击了NBT条目区域
        int startY = HEADER_HEIGHT + 15;
        int endY = startY + itemsPerPage * ENTRY_HEIGHT;

        if (mouseY >= startY && mouseY <= endY &&
                mouseX >= MARGIN + PANEL_PADDING &&
                mouseX <= field_22789 - MARGIN - PANEL_PADDING) {

            int clickedIndex = (int) ((mouseY - startY) / ENTRY_HEIGHT);
            if (clickedIndex >= 0 && clickedIndex < itemsPerPage) {
                startEditing(clickedIndex);
                return true;
            }
        }

        // 如果点击了非编辑区域且当前有活动的编辑器，保存并清理
        if (button == 0 && activeEditor != null && !activeEditor.method_25405(mouseX, mouseY)) {
            saveEditedValue(); // 尝试保存当前编辑
            return true;
        }

        // 如果点击了分页按钮区域，让分页按钮处理
        if (mouseY >= field_22790 - 40 && mouseY <= field_22790 - 15) {
            return false;
        }

        return false;
    }

    // 添加辅助方法
    private boolean isMouseOverEditor(double mouseX, double mouseY) {
        // 检查是否在活动文本编辑器上方
        if (activeEditor != null && activeEditor.method_25405(mouseX, mouseY)) {
            return true;
        }
        // 检查是否在活动下拉框上方
        if (activeDropdown != null && activeDropdown.method_25405(mouseX, mouseY)) {
            return true;
        }
        // 检查是否在保存按钮区域
        if (mouseX >= field_22789 - 100 && mouseX <= field_22789 - 20 &&
                mouseY >= 45 && mouseY <= 65) {
            return true;
        }
        // 检查是否在其他UI组件上方（按钮等）
        for (var child : method_25396()) {
            if (child instanceof class_339 widget && widget.method_25405(mouseX, mouseY)) {
                return true;
            }
        }
        return false;
    }

    // 添加清理方法
    public void clearActiveEditors() {
        if (activeEditor != null) {
            method_37066(activeEditor);
            activeEditor = null;
        }
        editingKey = null;
        method_25395(null);
    }

    private int getClickedIndex(double mouseX, double mouseY) {
        int startY = 60; // 与renderNbtEntries一致
        int entryHeight = 20; // 与renderNbtEntries一致

        if (mouseX < 20 || mouseX > field_22789 - 20 || mouseY < startY || mouseY > startY + itemsPerPage * entryHeight) {
            return -1;
        }

        int relativeY = (int)(mouseY - startY);
        int index = relativeY / entryHeight;

        int startIdx = (currentPage - 1) * itemsPerPage;
        if (index >= 0 && (startIdx + index) < nbtKeys.size()) {
            return index;
        }
        return -1;
    }

    private void startEditing(int relativeIndex) {
        clearActiveEditors();

        int absoluteIndex = (currentPage - 1) * itemsPerPage + relativeIndex;
        if (absoluteIndex >= nbtKeys.size()) return;

        editingKey = nbtKeys.get(absoluteIndex);
        class_2520 value = nbtMap.get(editingKey);

        if (value instanceof class_2499) {
            openListEditor((class_2499) value, editingKey);
            return;
        }

        // 动态计算位置
        int startY = HEADER_HEIGHT + 15;
        int yPos = startY + relativeIndex * ENTRY_HEIGHT;

        // 计算键名宽度 - 添加额外边距
        int keyWidth = field_22793.method_1727(editingKey) + 15;

        // 计算编辑器位置和大小 - 确保有足够空间
        int editorX = MARGIN + PANEL_PADDING + 5 + keyWidth;
        int editorWidth = field_22789 - editorX - MARGIN - PANEL_PADDING - 20; // 增加右边距

        // 确保编辑器宽度足够
        if (editorWidth < 100) {
            editorWidth = 100;
        }

        // 特殊处理布尔值 - 使用下拉框
        if (value.method_10711() == class_2520.field_33251 &&
                (value.equals(class_2481.field_21027) || value.equals(class_2481.field_21026))) {
            // 创建下拉框
            DropdownWidget dropdown = new DropdownWidget(
                    this, editorX, yPos, editorWidth, ENTRY_HEIGHT - 2,
                    class_2561.method_43471("entity_debugger.screen.boolean_dropdown"),
                    List.of(
                            new DropdownWidget.Option(class_2561.method_43471("entity_debugger.screen.true"), class_2481.field_21027),
                            new DropdownWidget.Option(class_2561.method_43471("entity_debugger.screen.false"), class_2481.field_21026)
                    ),
                    value.equals(class_2481.field_21027) ? 0 : 1
            );

            dropdown.setChangedListener(selected -> {
                modifiedValues.put(editingKey, selected.value());
                hasUnsavedChanges = true;
                syncToEntity(editingKey, selected.value());
                activeDropdown = null;
                clearActiveEditors();
                showSuccess(class_2561.method_43469("entity_debugger.message.boolean_updated", editingKey).getString());
            });

            method_37063(dropdown);
            method_25395(dropdown);
            activeDropdown = dropdown; // 设置当前活动下拉框
            return;
        }

        // 其他类型保持原样
        // 文本编辑器
        activeEditor = new class_342(
                field_22793, editorX, yPos, editorWidth, ENTRY_HEIGHT - 2,
                class_2561.method_43469("entity_debugger.screen.edit", editingKey)
        );

        activeEditor.method_1852(value.method_10714());
        activeEditor.method_1863(this::validateInput);
        method_25429(activeEditor);
        method_25395(activeEditor);
    }

    private void validateInput(String text) {
        if (activeEditor == null || editingKey == null) {
            return;
        }

        try {
            class_2520 original = nbtMap.get(editingKey);
            if (original == null) {
                return;
            }

            byte expectedType = original.method_10711();

            // 布尔值特殊验证
            if (expectedType == class_2520.field_33251) {
                if (text.equalsIgnoreCase("true") || text.equalsIgnoreCase("false")) {
                    activeEditor.method_1868(0xFFFFFF);
                    return;
                }
            }

            parseInputValue(text, expectedType);
            activeEditor.method_1868(0xFFFFFF);
        } catch (Exception e) {
            if (activeEditor != null) {
                activeEditor.method_1868(0xFF5555);
            }
        }
    }

    private void saveAllChanges() {
        if (!hasUnsavedChanges || modifiedValues.isEmpty()) {
            showError(class_2561.method_43471("entity_debugger.message.no_changes").getString());
            return;
        }

        try {
            class_2487 entityNbt = entity.method_5647(new class_2487());

            // 应用所有修改
            for (Map.Entry<String, class_2520> entry : modifiedValues.entrySet()) {
                String key = entry.getKey();
                class_2520 value = entry.getValue().method_10707();

                // 更新NBT
                entityNbt.method_10566(key, value);
                EntityDebugger.LOGGER.info("保存键 {}: {}", key, value.method_10714());
            }

            // 读取回实体以确保所有更改生效
            entity.method_5651(entityNbt);

            // 重新同步所有特殊标签到实体属性
            for (Map.Entry<String, class_2520> entry : modifiedValues.entrySet()) {
                syncToEntity(entry.getKey(), entry.getValue());
            }

            showSuccess(class_2561.method_43469("entity_debugger.message.save_all_success", modifiedValues.size()).getString());
            modifiedValues.clear();
            hasUnsavedChanges = false;
            reloadNbtData();
        } catch (Exception e) {
            EntityDebugger.LOGGER.error("保存所有更改失败: {}", e.getMessage(), e);
            showError("保存失败: " + e.getMessage());
        }
    }

    private void syncToEntity(String key, class_2520 value) {
        try {
            switch (key) {
                case "Pos":
                    if (value instanceof class_2499 list && list.size() >= 3) {
                        double x = list.method_10611(0);
                        double y = list.method_10611(1);
                        double z = list.method_10611(2);
                        entity.method_30634(x, y, z);
                        entity.method_23327(x, y, z); // 确保位置完全更新
                    }
                    break;

                case "Rotation":
                    if (value instanceof class_2499 list && list.size() >= 2) {
                        float yaw = list.method_10604(0);
                        float pitch = list.method_10604(1);
                        entity.method_36456(yaw);
                        entity.method_36457(pitch);
                        // 对于LivingEntity还需要更新头部旋转
                        if (entity instanceof class_1309 living) {
                            living.method_5847(yaw);
                            living.field_6283 = yaw;
                        }
                    }
                    break;

                case "Health":
                    if (entity instanceof class_1309 living) {
                        if (value instanceof class_2494 health) {
                            living.method_6033(health.method_10700());
                        } else if (value instanceof class_2499 list && !list.isEmpty()) {
                            living.method_6033(list.method_10604(0));
                        }
                    }
                    break;

                case "Motion":
                    if (value instanceof class_2499 list && list.size() >= 3) {
                        double mx = list.method_10611(0);
                        double my = list.method_10611(1);
                        double mz = list.method_10611(2);
                        entity.method_18800(mx, my, mz);
                    }
                    break;

                case "Air":
                    if (entity instanceof class_1309 living) {
                        if (value instanceof class_2516 air) {
                            living.method_5855(air.method_10696());
                        }
                    }
                    break;

                case "Fire":
                    entity.method_20803(value instanceof class_2516 fire ? fire.method_10696() : 0);
                    break;

                case "Invulnerable":
                    entity.method_5684(value instanceof class_2481 invul && invul.method_10698() != 0);
                    break;

                case "PortalCooldown":
                    entity.method_51850(value instanceof class_2497 cooldown ? cooldown.method_10701() : 0);
                    break;

                case "CustomName":
                    if (value instanceof class_2519 name) {
                        entity.method_5665(class_2561.method_43470(name.method_10714()));
                    }
                    break;

                case "NoGravity":
                    entity.method_5875(value instanceof class_2481 noGrav && noGrav.method_10698() != 0);
                    break;
            }
        } catch (Exception e) {
            EntityDebugger.LOGGER.error("同步{}到实体失败: {}", key, e.getMessage());
        }
    }

    @Override
    public void method_25410(class_310 client, int width, int height) {
        super.method_25410(client, width, height);

        // 重新计算每页项目数
        calculatePageSize();

        // 重新初始化UI组件
        this.method_25423(client, width, height);

        // 如果有活动的编辑器，需要重新定位
        if (activeEditor != null) {
            int relativeIndex = nbtKeys.indexOf(editingKey) % itemsPerPage;
            cleanupEditing();
            startEditing(relativeIndex);
        }
    }

    private void saveEditedValue() {
        try {
            if (editingKey == null || activeEditor == null) {
                EntityDebugger.LOGGER.warn("保存失败: 没有活动的编辑器");
                return;
            }

            String input = activeEditor.method_1882().trim();
            EntityDebugger.LOGGER.info("尝试保存键 {}: {}", editingKey, input);

            class_2520 original = nbtMap.get(editingKey);
            if (original == null) {
                EntityDebugger.LOGGER.warn("原始NBT元素不存在: {}", editingKey);
                return;
            }

            // 特殊处理布尔值 - 允许true/false文本输入
            if (original.method_10711() == class_2520.field_33251) {
                if (input.equalsIgnoreCase("true")) {
                    modifiedValues.put(editingKey, class_2481.field_21027);
                    nbtMap.put(editingKey, class_2481.field_21027);
                    hasUnsavedChanges = true;
                    syncToEntity(editingKey, class_2481.field_21027);
                    showSuccess(class_2561.method_43469("entity_debugger.message.save_success", editingKey).getString());
                    cleanupEditing();
                    return;
                } else if (input.equalsIgnoreCase("false")) {
                    modifiedValues.put(editingKey, class_2481.field_21026);
                    nbtMap.put(editingKey, class_2481.field_21026);
                    hasUnsavedChanges = true;
                    syncToEntity(editingKey, class_2481.field_21026);
                    showSuccess(class_2561.method_43469("entity_debugger.message.save_success", editingKey).getString());
                    cleanupEditing();
                    return;
                }
            }

            class_2520 newValue = parseInputValue(input, original.method_10711());
            EntityDebugger.LOGGER.info("解析成功: {}", newValue.method_10714());

            // 更新本地缓存
            modifiedValues.put(editingKey, newValue);
            nbtMap.put(editingKey, newValue);
            hasUnsavedChanges = true;

            // 即时同步到实体
            syncToEntity(editingKey, newValue);

            showSuccess(class_2561.method_43469("entity_debugger.message.save_success", editingKey).getString());
        } catch (Exception e) {
            EntityDebugger.LOGGER.error("保存失败: {}", e.getMessage(), e);
            showError("保存失败: " + e.getMessage());
        } finally {
            cleanupEditing();
        }
    }

    private class_2520 parseInputValue(String input, byte expectedType) throws Exception {
        input = input.trim();
        try {
            // 处理空输入
            if (input.isEmpty()) {
                throw new Exception("输入不能为空");
            }

            // 特殊处理布尔值
            if (expectedType == class_2520.field_33251) {
                if (input.equalsIgnoreCase("true") || input.equals("1")) {
                    return class_2481.field_21027;
                }
                if (input.equalsIgnoreCase("false") || input.equals("0")) {
                    return class_2481.field_21026;
                }
            }

            // 对于基本类型，不使用StringNbtReader.parse()，而是手动解析
            switch (expectedType) {
                case class_2520.field_33251:
                    // 处理字节后缀 (b/B)
                    if (input.endsWith("b") || input.endsWith("B")) {
                        input = input.substring(0, input.length() - 1);
                    }
                    return class_2481.method_23233(Byte.parseByte(input));

                case class_2520.field_33252:
                    // 处理短整型后缀 (s/S)
                    if (input.endsWith("s") || input.endsWith("S")) {
                        input = input.substring(0, input.length() - 1);
                    }
                    return class_2516.method_23254(Short.parseShort(input));

                case class_2520.field_33253:
                    return class_2497.method_23247(Integer.parseInt(input));

                case class_2520.field_33254:
                    // 处理长整型后缀 (l/L)
                    if (input.endsWith("l") || input.endsWith("L")) {
                        input = input.substring(0, input.length() - 1);
                    }
                    return class_2503.method_23251(Long.parseLong(input));

                case class_2520.field_33255:
                    // 处理浮点数后缀 (f/F)
                    if (input.endsWith("f") || input.endsWith("F")) {
                        input = input.substring(0, input.length() - 1);
                    }
                    return class_2494.method_23244(Float.parseFloat(input));

                case class_2520.field_33256:
                    // 处理双精度后缀 (d/D)
                    if (input.endsWith("d") || input.endsWith("D")) {
                        input = input.substring(0, input.length() - 1);
                    }
                    return class_2489.method_23241(Double.parseDouble(input));

                case class_2520.field_33258:
                    // 去除字符串引号（如果用户输入了）
                    if (input.startsWith("\"") && input.endsWith("\"")) {
                        input = input.substring(1, input.length() - 1);
                    }
                    return class_2519.method_23256(input);

                case class_2520.field_33259:
                    return parseNbtList(input);

                case class_2520.field_33260:
                    // 自动补全复合标签格式
                    if (!input.startsWith("{")) input = "{" + input;
                    if (!input.endsWith("}")) input = input + "}";
                    return class_2522.method_10718(input);

                case class_2520.field_33257:
                    if (!input.startsWith("[B;")) input = "[B;" + input;
                    if (!input.endsWith("]")) input = input + "]";
                    return class_2522.method_10718(input);

                case class_2520.field_33261:
                    if (!input.startsWith("[I;")) input = "[I;" + input;
                    if (!input.endsWith("]")) input = input + "]";
                    return class_2522.method_10718(input);

                case class_2520.field_33262:
                    if (!input.startsWith("[L;")) input = "[L;" + input;
                    if (!input.endsWith("]")) input = input + "]";
                    return class_2522.method_10718(input);

                default:
                    throw new Exception("不支持的NBT类型: " + getTypeName(expectedType));
            }
        } catch (NumberFormatException e) {
            throw new Exception(class_2561.method_43469("entity_debugger.error.number_format", e.getMessage()).getString());
        } catch (CommandSyntaxException e) {
            // 提供更友好的错误信息
            String msg = e.getMessage();
            if (msg.contains("expected key")) {
                throw new Exception(class_2561.method_43471("entity_debugger.error.compound_missing_key").getString());
            } else if (msg.contains("at line") && msg.contains("column")) {
                throw new Exception(class_2561.method_43469("entity_debugger.error.syntax_error",
                        msg.substring(msg.indexOf("line") + 5, msg.indexOf(",")),
                        msg.substring(msg.indexOf("column") + 7, msg.indexOf(")"))).getString());
            } else if (msg.contains("Expected value")) {
                throw new Exception(class_2561.method_43471("entity_debugger.error.missing_value").getString());
            } else if (msg.contains("Invalid escape sequence")) {
                throw new Exception(class_2561.method_43471("entity_debugger.error.invalid_escape").getString());
            }
            throw new Exception(class_2561.method_43469("entity_debugger.error.nbt_format", msg).getString());
        } catch (Exception e) {
            // 添加额外上下文信息
            throw new Exception(class_2561.method_43469("entity_debugger.error.parse_failed", getTypeName(expectedType), e.getMessage()).getString());
        }
    }

    // 显示保存成功
    // 显示错误（红色）
    private void showError(String message) {
        class_746 player = class_310.method_1551().field_1724;
        if (player != null) {
            player.method_7353(class_2561.method_43469("entity_debugger.message.error", message), false);
        }
        EntityDebugger.LOGGER.error("NBT Error: {}", message);
    }

    private void showSuccess(String message) {
        class_746 player = class_310.method_1551().field_1724;
        if (player != null) {
            player.method_7353(class_2561.method_43469("entity_debugger.message.success", message), false);
        }
    }

    // 在文件顶部添加自定义异常（可选）
    static class InvalidNbtFormatException extends RuntimeException {
        public InvalidNbtFormatException(String message) {
            super(message);
        }
    }

    private String getTypeName(byte nbtType) {
        return class_2561.method_43469("nbt.type." + switch (nbtType) {
            case class_2520.field_33250 -> "end";
            case class_2520.field_33251 -> "byte";
            case class_2520.field_33252 -> "short";
            case class_2520.field_33253 -> "int";
            case class_2520.field_33254 -> "long";
            case class_2520.field_33255 -> "float";
            case class_2520.field_33256 -> "double";
            case class_2520.field_33257 -> "byte_array";
            case class_2520.field_33258 -> "string";
            case class_2520.field_33259 -> "list";
            case class_2520.field_33260 -> "compound";
            case class_2520.field_33261 -> "int_array";
            case class_2520.field_33262 -> "long_array";
            default -> "unknown";
        }, nbtType).getString();
    }


    private int getTotalPages() {
        return (int) Math.ceil((double) nbtKeys.size() / itemsPerPage);
    }

    private void cleanupEditing() {
        if (activeEditor != null) {
            method_37066(activeEditor); // 从屏幕移除
            activeEditor = null;
        }
        editingKey = null;
        method_25395(null); // 移除焦点
    }

    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        if (activeEditor != null) {
            if (keyCode == GLFW.GLFW_KEY_ENTER) {
                saveEditedValue();
                return true;
            }
            if (keyCode == GLFW.GLFW_KEY_ESCAPE) {
                cleanupEditing();
                return true;
            }
            return activeEditor.method_25404(keyCode, scanCode, modifiers);
        }
        return super.method_25404(keyCode, scanCode, modifiers);
    }

    @Override
    public void method_25419() {
        if (hasUnsavedChanges) {
            this.field_22787.method_1507(new class_410(confirmed -> {
                if (confirmed) saveAllChanges();
                super.method_25419();
            }, class_2561.method_43471("entity_debugger.dialog.unsaved_changes.title"),
                    class_2561.method_43471("entity_debugger.dialog.unsaved_changes.message")));
        } else {
            super.method_25419();
        }
    }
}