package forestry.api.plugin;

import forestry.api.apiculture.IBeeJubilance;
import forestry.api.apiculture.genetics.IBeeSpecies;
import forestry.api.apiculture.genetics.IBeeSpeciesType;
import forestry.api.core.IProduct;
import forestry.api.core.Product;
import net.minecraft.network.chat.TextColor;
import net.minecraft.world.item.ItemStack;

import java.awt.*;
import java.util.List;

/**
 * Builder used to register new bee species and configure already existing ones.
 * Use {@link IApicultureRegistration#registerSpecies} to obtain instances of this class.
 */
public interface IBeeSpeciesBuilder extends ISpeciesBuilder<IBeeSpeciesType, IBeeSpecies, IBeeSpeciesBuilder> {
	/**
	 * Adds a product to this species.
	 *
	 * @param product A product that can be produced by this species.
	 */
	IBeeSpeciesBuilder addProduct(IProduct product);

	/**
	 * Adds a product to this bee species.
	 *
	 * @param stack  The item stack the product should produce.
	 * @param chance A float between 0 and 1. The chance that this product is produced during a single work cycle.
	 */
	default IBeeSpeciesBuilder addProduct(ItemStack stack, float chance) {
		return addProduct(new Product(stack.getItem(), stack.getCount(), stack.getTag(), chance));
	}

	/**
	 * Adds a specialty product to this species, only produced when the bee in a jubilant state.
	 *
	 * @param specialty A product that can only be produced by this species when in its jubilant state.
	 */
	IBeeSpeciesBuilder addSpecialty(IProduct specialty);

	/**
	 * Adds a specialty to the bee species, a product only produced when the bee is in a jubilant state.
	 *
	 * @param stack  The item stack the product should produce.
	 * @param chance A float between 0 and 1. The chance that this product is produced during a single work cycle.
	 */
	default IBeeSpeciesBuilder addSpecialty(ItemStack stack, float chance) {
		return addSpecialty(new Product(stack.getItem(), stack.getCount(), stack.getTag(), chance));
	}

	/**
	 * Sets the color of the bee's body. The default is yellow, {@code #ffdc16}, used by most bees.
	 *
	 * @since 2.3.3 Now accepts TextColor instead of java.awt.Color
	 */
	IBeeSpeciesBuilder setBody(TextColor color);

	/**
	 * Sets the color of the bee's stripes. The default is {@code #000000}.
	 *
	 * @since 2.3.3 Now accepts TextColor instead of java.awt.Color
	 */
	IBeeSpeciesBuilder setStripes(TextColor color);

	/**
	 * Overrides the default bee outlines set in {@link IApicultureRegistration#registerSpecies}.
	 *
	 * @since 2.3.3 Now accepts TextColor instead of java.awt.Color
	 */
	IBeeSpeciesBuilder setOutline(TextColor color);

	/**
	 * @deprecated Use variant that accepts TextColor
	 */
	@Deprecated(forRemoval = true)
	default IBeeSpeciesBuilder setBody(Color color) {
		return setBody(TextColor.fromRgb(color.getRGB()));
	}

	/**
	 * @deprecated Use variant that accepts TextColor
	 */
	@Deprecated(forRemoval = true)
	default IBeeSpeciesBuilder setStripes(Color color) {
		return setStripes(TextColor.fromRgb(color.getRGB()));
	}

	/**
	 * @deprecated Use variant that accepts TextColor
	 */
	@Deprecated(forRemoval = true)
	default IBeeSpeciesBuilder setOutline(Color color) {
		return setOutline(TextColor.fromRgb(color.getRGB()));
	}

	/**
	 * Specify the jubilance conditions for this bee species. The default returns true if the bee's ideal temperature and humidity are met.
	 * When {@link IBeeJubilance#isJubilant} returns true, a bee can produce its specialty products.
	 */
	IBeeSpeciesBuilder setJubilance(IBeeJubilance jubilance);

	List<IProduct> buildProducts();

	List<IProduct> buildSpecialties();

	int getBody();

	int getStripes();

	int getOutline();

	IBeeJubilance getJubilance();
}
