package forestry.core.genetics;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import forestry.api.genetics.IGenome;
import forestry.api.genetics.ISpecies;
import forestry.api.genetics.alleles.*;
import forestry.api.plugin.IGenomeBuilder;

import java.util.IdentityHashMap;
import java.util.Map;

public final class Genome implements IGenome {
	final ImmutableMap<IChromosome<?>, AllelePair<?>> chromosomes;
	private final IKaryotype karyotype;

	private boolean isDefaultGenome;
	private boolean hasCachedDefaultGenome;

	public Genome(IKaryotype karyotype, ImmutableMap<IChromosome<?>, AllelePair<?>> chromosomes) {
		this.karyotype = karyotype;
		this.chromosomes = chromosomes;
	}

	// Used by codec to sort alleles properly and populate missing chromosomes
	public static IGenome sanitizeAlleles(Karyotype karyotype, Map<IChromosome<?>, AllelePair<?>> map) {
		ImmutableMap.Builder<IChromosome<?>, AllelePair<?>> sorted = ImmutableMap.builderWithExpectedSize(map.size());
		for (IChromosome<?> chromosome : karyotype.getChromosomes()) {
			// Fix missing chromosomes for backwards compatibility
			AllelePair<?> pair = map.get(chromosome);
			if (pair == null) {
				ISpecies<?> species;

				AllelePair<?> speciesPair = map.get(karyotype.getSpeciesChromosome());
				if (speciesPair == null) {
					// If species was added/removed, revert to default
					species = karyotype.getSpeciesChromosome().get(karyotype.getDefaultSpecies());
				} else {
					species = speciesPair.active()
						.<IRegistryAllele<ISpecies<?>>>cast()
						.value();
				}

				pair = species
					.getDefaultGenome()
					.getAllelePair(chromosome);
			}

			sorted.put(chromosome, pair);
		}
		return new Genome(karyotype, sorted.buildOrThrow());
	}

	@Override
	public ImmutableList<AllelePair<?>> getAllelePairs() {
		return this.chromosomes.values().asList();
	}

	@Override
	public IKaryotype getKaryotype() {
		return this.karyotype;
	}

	@Override
	@SuppressWarnings("unchecked")
	public <A extends IAllele> AllelePair<A> getAllelePair(IChromosome<A> chromosomeType) {
		return (AllelePair<A>) this.chromosomes.get(chromosomeType);
	}

	@Override
	public boolean isDefaultGenome() {
		if (!this.hasCachedDefaultGenome) {
			Genome defaultGenome = (Genome) getActiveValue(this.karyotype.getSpeciesChromosome()).getDefaultGenome();

			this.isDefaultGenome = this == defaultGenome || (isSameAlleles(defaultGenome));
			this.hasCachedDefaultGenome = true;
		}

		return this.isDefaultGenome;
	}

	@Override
	public ImmutableMap<IChromosome<?>, AllelePair<?>> getChromosomes() {
		return this.chromosomes;
	}

	@Override
	public IGenome copyWithPairs(Map<IChromosome<?>, AllelePair<?>> alleles) {
		if (alleles.isEmpty()) {
			return this;
		} else {
			Genome.Builder builder = new Genome.Builder(getKaryotype());
			// avoid duplicate instances of default genome
			boolean isDefault = true;

			// copy over allele map or default allele
			for (Map.Entry<IChromosome<?>, AllelePair<?>> entry : this.chromosomes.entrySet()) {
				IChromosome<?> chromosome = entry.getKey();
				AllelePair<?> pair = entry.getValue();
				AllelePair<?> allele = alleles.get(chromosome);

				if (allele == null || allele.equals(pair)) {
					// copy default allele if missing or equal
					builder.setUnchecked(chromosome, pair);
				} else {
					// mark not default when there are non-default alleles
					builder.setUnchecked(chromosome, allele);
					isDefault = false;
				}
			}

			if (isDefault) {
				return this;
			} else {
				return builder.build();
			}
		}
	}

	@Override
	public boolean isSameAlleles(IGenome other) {
		return other.getKaryotype() == this.karyotype && this.chromosomes.equals(other.getChromosomes());
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		Genome genome = (Genome) o;

		if (!this.chromosomes.equals(genome.chromosomes)) {
			return false;
		}
		return this.karyotype.equals(genome.karyotype);
	}

	@Override
	public int hashCode() {
		int result = this.chromosomes.hashCode();
		result = 31 * result + this.karyotype.hashCode();
		return result;
	}

	public static class Builder implements IGenomeBuilder {
		private final IKaryotype karyotype;
		private final IdentityHashMap<IChromosome<?>, IAllele> active = new IdentityHashMap<>();
		private final IdentityHashMap<IChromosome<?>, IAllele> inactive = new IdentityHashMap<>();

		public Builder(IKaryotype karyotype) {
			Preconditions.checkNotNull(karyotype);

			this.karyotype = karyotype;
		}

		@Override
		public <A extends IAllele> void set(IChromosome<A> chromosome, A allele) {
			if (this.karyotype.isAlleleValid(chromosome, allele)) {
				this.active.put(chromosome, allele);
				this.inactive.put(chromosome, allele);
			} else {
				throw new IllegalArgumentException("Allele " + allele.alleleId() + " is not a valid value for chromosome " + chromosome.id() + " in the karyotype with species chromosome " + this.karyotype.getSpeciesChromosome().id());
			}
		}

		@Override
		public <A extends IAllele> void setActive(IChromosome<A> chromosome, A allele) {
			if (this.karyotype.isAlleleValid(chromosome, allele)) {
				this.active.put(chromosome, allele);
			} else {
				throw new IllegalArgumentException("Invalid allele " + allele.alleleId() + " for chromosome " + chromosome.id());
			}
		}

		@Override
		public <A extends IAllele> void setInactive(IChromosome<A> chromosome, A allele) {
			if (this.karyotype.isAlleleValid(chromosome, allele)) {
				this.inactive.put(chromosome, allele);
			}
		}

		@Override
		public boolean isEmpty() {
			return this.inactive.isEmpty() && this.active.isEmpty();
		}

		@Override
		public void setRemainingDefault() {
			for (Map.Entry<IChromosome<?>, ? extends IAllele> entry : this.karyotype.getDefaultAlleles().entrySet()) {
				IChromosome<?> chromosome = entry.getKey();
				IAllele defaultAllele = entry.getValue();

				if (!this.active.containsKey(chromosome)) {
					this.active.put(chromosome, defaultAllele);
				}
				if (!this.inactive.containsKey(chromosome)) {
					this.inactive.put(chromosome, defaultAllele);
				}
			}
		}

		@Override
		public IGenome build() {
			// Check for missing chromosomes and display which ones are missing
			if (this.karyotype.size() != this.active.size()) {
				StringBuilder msg = new StringBuilder("Tried to build genome, but the following chromosomes are missing from the karyotype: { ");

				for (IChromosome<?> chromosome : this.karyotype.getChromosomes()) {
					if (!this.active.containsKey(chromosome)) {
						msg.append(chromosome.id());
						msg.append(' ');
					}
				}
				msg.append('}');

				throw new IllegalStateException(msg.toString());
			} else {
				ImmutableMap.Builder<IChromosome<?>, AllelePair<?>> genome = new ImmutableMap.Builder<>();

				for (IChromosome<?> chromosome : this.karyotype.getChromosomes()) {
					IAllele active = this.active.get(chromosome);
					IAllele inactive = this.inactive.get(chromosome);

					// Check for incomplete pairs
					if (active == null || inactive == null) {
						throw new IllegalStateException("Tried to build genome, but the allele pair was incomplete for the following chromosome: " + chromosome.id());
					} else {
						genome.put(chromosome, new AllelePair<>(active, inactive));
					}
				}

				return new Genome(this.karyotype, genome.build());
			}
		}
	}
}
