/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.gui;

import com.google.common.base.Preconditions;
import com.google.common.collect.LinkedListMultimap;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.PoseStack;
import forestry.Forestry;
import forestry.api.ForestryConstants;
import forestry.api.climate.IClimateProvider;
import forestry.api.core.IErrorLogicSource;
import forestry.api.core.IErrorSource;
import forestry.core.config.ForestryConfig;
import forestry.core.gui.ledgers.*;
import forestry.core.gui.widgets.TankWidget;
import forestry.core.gui.widgets.Widget;
import forestry.core.gui.widgets.WidgetManager;
import forestry.core.owner.IOwnedTile;
import forestry.core.render.ColourProperties;
import forestry.energy.ForestryEnergyStorage;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.screens.inventory.AbstractContainerScreen;
import net.minecraft.client.renderer.Rect2i;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.inventory.AbstractContainerMenu;
import net.minecraft.world.inventory.Slot;

import javax.annotation.Nullable;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;

public abstract class GuiForestry<C extends AbstractContainerMenu> extends AbstractContainerScreen<C> implements IGuiSizable {
	// Used to display "Did you know?" ledgers in GUI
	public static final LinkedListMultimap<String, String> HINTS = readDefaultHints();

	public final ResourceLocation textureFile;
	protected final WidgetManager widgetManager;
	protected final LedgerManager ledgerManager;
	protected TextLayoutHelper textLayout;

	protected GuiForestry(String texture, C menu, Inventory inv, Component title) {
		this(ForestryConstants.forestry(texture), menu, inv, title);
	}

	protected GuiForestry(ResourceLocation texture, C menu, Inventory inv, Component title) {
		super(menu, inv, title);

		this.widgetManager = new WidgetManager(this);
		this.ledgerManager = new LedgerManager(this);

		this.textureFile = texture;
	}

	/* LEDGERS */
	@Override
	public void init() {
		super.init();

		int maxLedgerWidth = (this.width - this.imageWidth) / 2;

		this.ledgerManager.setMaxWidth(maxLedgerWidth);
		this.ledgerManager.clear();

		this.textLayout = new TextLayoutHelper(this, ColourProperties.INSTANCE);

		addLedgers();
	}

	@Override
	public void resize(Minecraft mc, int width, int height) {
		super.resize(mc, width, height);
	}

	@Override
	public void containerTick() {
		super.containerTick();
	}

	@Override
	public void render(GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
		renderBackground(graphics);
		super.render(graphics, mouseX, mouseY, partialTicks);
		renderTooltip(graphics, mouseX, mouseY);
	}

	protected abstract void addLedgers();

	protected final void addErrorLedger(IErrorSource errorSource) {
        this.ledgerManager.add(errorSource);
	}

	protected final void addErrorLedger(IErrorLogicSource errorSource) {
        this.ledgerManager.add(errorSource.getErrorLogic());
	}

	protected final void addClimateLedger(IClimateProvider climatised) {
        this.ledgerManager.add(new ClimateLedger(this.ledgerManager, climatised));
	}

	protected final void addPowerLedger(ForestryEnergyStorage energyStorage) {
        this.ledgerManager.add(new PowerLedger(this.ledgerManager, energyStorage));
	}

	protected final void addHintLedger(String hintsKey) {
		if (ForestryConfig.CLIENT.enableHints.get()) {
			List<String> hints = HINTS.get(hintsKey);
			addHintLedger(hints);
		}
	}

	protected final void addHintLedger(List<String> hints) {
		if (ForestryConfig.CLIENT.enableHints.get()) {
			if (!hints.isEmpty()) {
                this.ledgerManager.add(new HintLedger(this.ledgerManager, hints));
			}
		}
	}

	protected final void addOwnerLedger(IOwnedTile ownedTile) {
        this.ledgerManager.add(new OwnerLedger(this.ledgerManager, ownedTile));
	}

	@Override
	public void onClose() {
		super.onClose();
        this.ledgerManager.onClose();
	}

	public ColourProperties getFontColor() {
		return ColourProperties.INSTANCE;
	}

	public Font font() {
		return this.font;
	}

	@Override
	public boolean mouseClicked(double mouseX, double mouseY, int mouseButton) {
		// / Handle ledger clicks
		this.ledgerManager.handleMouseClicked(mouseX, mouseY, mouseButton);
		this.widgetManager.handleMouseClicked(mouseX, mouseY, mouseButton);
		return super.mouseClicked(mouseX, mouseY, mouseButton);
	}

	@Override
	public boolean mouseReleased(double mouseX, double mouseY, int mouseButton) {
		if (this.widgetManager.handleMouseRelease(mouseX, mouseY, mouseButton)) {
			return true;
		}
		return super.mouseReleased(mouseX, mouseY, mouseButton);
	}

	@Nullable
	public TankWidget getTankAtPosition(double mouseX, double mouseY) {
		for (Widget widget : this.widgetManager.getWidgets()) {
			if (widget instanceof TankWidget tankWidget && widget.isMouseOver(mouseX - this.leftPos, mouseY - this.topPos)) {
				return tankWidget;
			}
		}
		return null;
	}

	@Nullable
	protected Slot getSlotAtPosition(double mouseX, double mouseY) {
		for (int k = 0; k < this.menu.slots.size(); ++k) {
			Slot slot = this.menu.slots.get(k);

			if (isMouseOverSlot(slot, mouseX, mouseY)) {
				return slot;
			}
		}

		return null;
	}

	private boolean isMouseOverSlot(Slot par1Slot, double mouseX, double mouseY) {
		return isHovering(par1Slot.x, par1Slot.y, 16, 16, mouseX, mouseY);
	}

	@Override
	protected void renderLabels(GuiGraphics graphics, int mouseX, int mouseY) {
        this.ledgerManager.drawTooltips(graphics, mouseX, mouseY);

		if (this.menu.getCarried().isEmpty()) {
			GuiUtil.drawToolTips(graphics, this, this.widgetManager.getWidgets(), mouseX, mouseY);
			GuiUtil.drawToolTips(graphics, this, this.renderables, mouseX, mouseY);
			GuiUtil.drawToolTips(graphics, this, this.menu.slots, mouseX, mouseY);
		}
	}

	protected void renderBg(GuiGraphics graphics, float partialTicks, int mouseX, int mouseY) {
		drawBackground(graphics);

        this.widgetManager.updateWidgets(mouseX - this.leftPos, mouseY - this.topPos);

		graphics.setColor(1.0F, 1.0F, 1.0F, 1.0F);
		PoseStack transform = graphics.pose();
		transform.pushPose();
		transform.translate(this.leftPos, this.topPos, 0.0F);
		drawWidgets(graphics);
		transform.popPose();

		bindTexture(this.textureFile);
	}

	protected void drawBackground(GuiGraphics transform) {
		transform.blit(this.textureFile, this.leftPos, this.topPos, 0, 0, this.imageWidth, this.imageHeight);
	}

	protected void drawWidgets(GuiGraphics graphics) {
        this.ledgerManager.drawLedgers(graphics);
        this.widgetManager.drawWidgets(graphics);
	}

	protected void bindTexture(ResourceLocation texturePath) {
		RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1.0F);
		RenderSystem.setShaderTexture(0, texturePath);
	}

	@Override
	public int getSizeX() {
		return this.imageWidth;
	}

	@Override
	public int getSizeY() {
		return this.imageHeight;
	}

	@Override
	public Minecraft getGameInstance() {
		return Preconditions.checkNotNull(this.minecraft);
	}

	public List<Rect2i> getExtraGuiAreas() {
		return this.ledgerManager.getLedgerAreas();
	}

	public TextLayoutHelper getTextLayout() {
		return this.textLayout;
	}

	private static LinkedListMultimap<String, String> readDefaultHints() {
		LinkedListMultimap<String, String> map = LinkedListMultimap.create();
		Properties prop = new Properties();

		try {
			InputStream hintStream = GuiForestry.class.getResourceAsStream("/config/forestry/hints.properties");
			prop.load(hintStream);
		} catch (IOException | NullPointerException e) {
			Forestry.LOGGER.error("Failed to load hints file.", e);
		}

		for (String key : prop.stringPropertyNames()) {
			String list = prop.getProperty(key);

			if (!list.isEmpty()) {
				for (String parsedHint : list.split(";+")) {
					map.put(key, parsedHint);
				}
			}
		}

		return map;
	}
}
