package forestry.core.multiblock;

import forestry.Forestry;
import forestry.api.multiblock.IMultiblockComponent;
import forestry.api.multiblock.IMultiblockLogic;
import forestry.core.tiles.TileUtil;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import net.minecraft.core.BlockPos;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.chunk.ChunkSource;

import java.util.*;

/**
 * This class manages all the multiblock controllers that exist in a given world,
 * either client- or server-side.
 * You must create different registries for server and client worlds.
 *
 * @author Erogenous Beef
 */
public class MultiblockWorldRegistry {
	private final Level world;

	private final Set<IMultiblockControllerInternal> controllers;        // Active controllers
	private final Set<IMultiblockControllerInternal> dirtyControllers;    // Controllers whose parts lists have changed
	private final Set<IMultiblockControllerInternal> deadControllers;    // Controllers which are empty

	// A list of orphan parts - parts which currently have no master, but should seek one this tick
	// Indexed by the hashed chunk coordinate
	// This can be added-to asynchronously via chunk loads!
	private Set<IMultiblockComponent> orphanedParts;

	// A list of parts which have been detached during internal operations
	private final Set<IMultiblockComponent> detachedParts;

	// A list of parts whose chunks have not yet finished loading
	// They will be added to the orphan list when they are finished loading.
	// Indexed by the hashed chunk coordinate
	// This can be added-to asynchronously via chunk loads!
	private final Long2ObjectMap<Set<IMultiblockComponent>> partsAwaitingChunkLoad;

	// Mutexes to protect lists which may be changed due to asynchronous events, such as chunk loads
	private final Object partsAwaitingChunkLoadMutex;
	private final Object orphanedPartsMutex;

	public MultiblockWorldRegistry(Level world) {
		this.world = world;

		this.controllers = new HashSet<>();
		this.deadControllers = new HashSet<>();
		this.dirtyControllers = new HashSet<>();

		this.detachedParts = new HashSet<>();
		this.orphanedParts = new HashSet<>();

		this.partsAwaitingChunkLoad = new Long2ObjectOpenHashMap<>();
		this.partsAwaitingChunkLoadMutex = new Object();
		this.orphanedPartsMutex = new Object();
	}

	/**
	 * Called before Tile Entities are ticked in the world. Run game logic.
	 */
	public void tickStart() {
		if (!this.controllers.isEmpty()) {
			for (IMultiblockControllerInternal controller : this.controllers) {
				if (controller.getWorldObj() == this.world && controller.getWorldObj().isClientSide == this.world.isClientSide) {
					if (controller.hasNoParts()) {
						// This happens on the server when the user breaks the last block. It's fine.
						// Mark 'er dead and move on.
                        this.deadControllers.add(controller);
					} else {
						// Run the game logic for this world
						controller.updateMultiblockEntity();
					}
				}
			}
		}
	}

	/**
	 * Called prior to processing multiblock controllers. Do bookkeeping.
	 */
	public void processMultiblockChanges() {
		ChunkSource chunkProvider = this.world.getChunkSource();
		BlockPos coord;

		// Merge pools - sets of adjacent machines which should be merged later on in processing
		List<Set<IMultiblockControllerInternal>> mergePools = null;
		if (!this.orphanedParts.isEmpty()) {
			Set<IMultiblockComponent> orphansToProcess = null;

			// Keep the synchronized block small. We can't iterate over orphanedParts directly
			// because the client does not know which chunks are actually loaded, so attachToNeighbors()
			// is not chunk-safe on the client, because Minecraft is stupid.
			// It's possible to polyfill this, but the polyfill is too slow for comfort.
			synchronized (this.orphanedPartsMutex) {
				if (!this.orphanedParts.isEmpty()) {
					orphansToProcess = this.orphanedParts;
                    this.orphanedParts = new HashSet<>();
				}
			}

			if (orphansToProcess != null && !orphansToProcess.isEmpty()) {
				Set<IMultiblockControllerInternal> compatibleControllers;

				// Process orphaned blocks
				// These are blocks that exist in a valid chunk and require a controller
				for (IMultiblockComponent orphan : orphansToProcess) {
					coord = orphan.getCoordinates();
					if (!chunkProvider.hasChunk(coord.getX() >> 4, coord.getZ() >> 4)) {
						continue;
					}

					// This can occur on slow machines.
					if (orphan instanceof BlockEntity entity && entity.isRemoved()) {
						continue;
					}

					if (TileUtil.getTile(this.world, coord) != orphan) {
						// This block has been replaced by another.
						continue;
					}

					// THIS IS THE ONLY PLACE WHERE PARTS ATTACH TO MACHINES
					// Try to attach to a neighbor's master controller
					compatibleControllers = attachToNeighbors(orphan);
					if (compatibleControllers.isEmpty()) {
						// FOREVER ALONE! Create and register a new controller.
						// THIS IS THE ONLY PLACE WHERE NEW CONTROLLERS ARE CREATED.
						MultiblockLogic<?> logic = (MultiblockLogic<?>) orphan.getMultiblockLogic();
						IMultiblockControllerInternal newController = logic.createNewController(this.world);
						newController.attachBlock(orphan);
						this.controllers.add(newController);
					} else if (compatibleControllers.size() > 1) {
						if (mergePools == null) {
							mergePools = new ArrayList<>();
						}

						// THIS IS THE ONLY PLACE WHERE MERGES ARE DETECTED
						// Multiple compatible controllers indicates an impending merge.
						// Locate the appropriate merge pool(s)
						List<Set<IMultiblockControllerInternal>> candidatePools = new ArrayList<>();
						for (Set<IMultiblockControllerInternal> candidatePool : mergePools) {
							if (!Collections.disjoint(candidatePool, compatibleControllers)) {
								// They share at least one element, so that means they will all touch after the merge
								candidatePools.add(candidatePool);
							}
						}

						if (candidatePools.isEmpty()) {
							// No pools nearby, create a new merge pool
							mergePools.add(compatibleControllers);
						} else if (candidatePools.size() == 1) {
							// Only one pool nearby, simply add to that one
							candidatePools.get(0).addAll(compatibleControllers);
						} else {
							// Multiple pools- merge into one, then add the compatible controllers
							Set<IMultiblockControllerInternal> masterPool = candidatePools.get(0);
							Set<IMultiblockControllerInternal> consumedPool;
							for (int i = 1; i < candidatePools.size(); i++) {
								consumedPool = candidatePools.get(i);
								masterPool.addAll(consumedPool);
								mergePools.remove(consumedPool);
							}
							masterPool.addAll(compatibleControllers);
						}
					}
				}
			}
		}

		if (mergePools != null && !mergePools.isEmpty()) {
			// Process merges - any machines that have been marked for merge should be merged
			// into the "master" machine.
			// To do this, we combine lists of machines that are touching one another and therefore
			// should voltron the fuck up.
			for (Set<IMultiblockControllerInternal> mergePool : mergePools) {
				// Search for the new master machine, which will take over all the blocks contained in the other machines
				IMultiblockControllerInternal newMaster = null;
				for (IMultiblockControllerInternal controller : mergePool) {
					if (newMaster == null || controller.shouldConsume(newMaster)) {
						newMaster = controller;
					}
				}

				if (newMaster == null) {
					Forestry.LOGGER.error("Multiblock system checked a merge pool of size {}, found no master candidates. This should never happen.", mergePool.size());
				} else {
					// Merge all the other machines into the master machine, then unregister them
					addDirtyController(newMaster);
					for (IMultiblockControllerInternal controller : mergePool) {
						if (controller != newMaster) {
							newMaster.assimilate(controller);
							addDeadController(controller);
							addDirtyController(newMaster);
						}
					}
				}
			}
		}

		// Process splits and assembly
		// Any controllers which have had parts removed must be checked to see if some parts are no longer
		// physically connected to their master.
		if (!this.dirtyControllers.isEmpty()) {
			for (IMultiblockControllerInternal controller : this.dirtyControllers) {
				if (controller == null) {
					continue;
				}
				// Tell the machine to check if any parts are disconnected.
				// It should return a set of parts which are no longer connected.
				// POSTCONDITION: The controller must have informed those parts that
				// they are no longer connected to this machine.
				Set<IMultiblockComponent> newlyDetachedParts = controller.checkForDisconnections();

				if (!controller.hasNoParts()) {
					controller.recalculateMinMaxCoords();
					controller.checkIfMachineIsWhole();
				} else {
					addDeadController(controller);
				}

				if (!newlyDetachedParts.isEmpty()) {
					// Controller has shed some parts - add them to the detached list for delayed processing
                    this.detachedParts.addAll(newlyDetachedParts);
				}
			}

            this.dirtyControllers.clear();
		}

		// Unregister dead controllers
		if (!this.deadControllers.isEmpty()) {
			for (IMultiblockControllerInternal controller : this.deadControllers) {
				// Go through any controllers which have marked themselves as potentially dead.
				// Validate that they are empty/dead, then unregister them.
				if (!controller.hasNoParts()) {
					Forestry.LOGGER.error("Found a non-empty controller. Forcing it to shed its blocks and die. This should never happen!");
                    this.detachedParts.addAll(controller.detachAllBlocks());
				}

				// THIS IS THE ONLY PLACE WHERE CONTROLLERS ARE UNREGISTERED.
				BlockPos destroyedCoord = controller.getDestroyedCoord();
				if (destroyedCoord != null) {
					controller.onDestroyed(destroyedCoord);
				}
				this.controllers.remove(controller);
			}

            this.deadControllers.clear();
		}

		// Process detached blocks
		// Any blocks which have been detached this tick should be moved to the orphaned
		// list, and will be checked next tick to see if their chunk is still loaded.
		for (IMultiblockComponent part : this.detachedParts) {
			// Ensure parts know they're detached
			MultiblockLogic<?> logic = (MultiblockLogic<?>) part.getMultiblockLogic();
			logic.assertDetached(part);
		}

		addAllOrphanedPartsThreadsafe(this.detachedParts);
        this.detachedParts.clear();
	}

	///// Multiblock Connection Base Logic
	private Set<IMultiblockControllerInternal> attachToNeighbors(IMultiblockComponent part) {
		Set<IMultiblockControllerInternal> controllers = new HashSet<>();
		IMultiblockControllerInternal bestController = null;

		MultiblockLogic<?> logic = (MultiblockLogic<?>) part.getMultiblockLogic();
		Class<?> controllerClass = logic.getControllerClass();
		// Look for a compatible controller in our neighboring parts.
		List<IMultiblockComponent> partsToCheck = MultiblockUtil.getNeighboringParts(this.world, part);
		for (IMultiblockComponent neighborPart : partsToCheck) {
			IMultiblockLogic neighborLogic = neighborPart.getMultiblockLogic();
			if (neighborLogic.isConnected()) {
				IMultiblockControllerInternal candidate = (IMultiblockControllerInternal) neighborLogic.getController();
				if (!controllerClass.isAssignableFrom(candidate.getClass())) {
					// Skip multiblocks with incompatible types
					continue;
				}

				if (!controllers.contains(candidate) && (bestController == null || candidate.shouldConsume(bestController))) {
					bestController = candidate;
				}

				controllers.add(candidate);
			}
		}

		// If we've located a valid neighboring controller, attach to it.
		if (bestController != null) {
			// attachBlock will call onAttached, which will set the controller.
			bestController.attachBlock(part);
		}

		return controllers;
	}

	/**
	 * Called when a multiblock part is added to the world, either via chunk-load or user action.
	 * If its chunk is loaded, it will be processed during the next tick.
	 * If the chunk is not loaded, it will be added to a list of objects waiting for a chunkload.
	 *
	 * @param part The part which is being added to this world.
	 */
	public void onPartAdded(IMultiblockComponent part) {
		BlockPos worldLocation = part.getCoordinates();

		if (!this.world.getChunkSource().hasChunk(worldLocation.getX() >> 4, worldLocation.getZ() >> 4)) {
			// Part goes into the waiting-for-chunk-load list
			Set<IMultiblockComponent> partSet;
			long chunkHash = ChunkPos.asLong(worldLocation.getX() >> 4, worldLocation.getZ() >> 4);
			synchronized (this.partsAwaitingChunkLoadMutex) {
				if (!this.partsAwaitingChunkLoad.containsKey(chunkHash)) {
					partSet = new HashSet<>();
                    this.partsAwaitingChunkLoad.put(chunkHash, partSet);
				} else {
					partSet = this.partsAwaitingChunkLoad.get(chunkHash);
				}

				partSet.add(part);
			}
		} else {
			// Part goes into the orphan queue, to be checked this tick
			addOrphanedPartThreadsafe(part);
		}
	}

	/**
	 * Called when a part is removed from the world, via user action or via chunk unloads.
	 * This part is removed from any lists in which it may be, and its machine is marked for recalculation.
	 *
	 * @param part The part which is being removed.
	 */
	public void onPartRemovedFromWorld(IMultiblockComponent part) {
		BlockPos coord = part.getCoordinates();
		long hash = ChunkPos.asLong(coord.getX() >> 4, coord.getZ() >> 4);

		if (this.partsAwaitingChunkLoad.containsKey(hash)) {
			synchronized (this.partsAwaitingChunkLoadMutex) {
				if (this.partsAwaitingChunkLoad.containsKey(hash)) {
                    this.partsAwaitingChunkLoad.get(hash).remove(part);
					if (this.partsAwaitingChunkLoad.get(hash).size() <= 0) {
                        this.partsAwaitingChunkLoad.remove(hash);
					}
				}
			}
		}

        this.detachedParts.remove(part);
		if (this.orphanedParts.contains(part)) {
			synchronized (this.orphanedPartsMutex) {
                this.orphanedParts.remove(part);
			}
		}

		MultiblockLogic<?> logic = (MultiblockLogic<?>) part.getMultiblockLogic();
		logic.assertDetached(part);
	}

	/**
	 * Called when the world which this World Registry represents is fully unloaded from the system.
	 * Does some housekeeping just to be nice.
	 */
	public void onWorldUnloaded() {
        this.controllers.clear();
        this.deadControllers.clear();
        this.dirtyControllers.clear();

        this.detachedParts.clear();

		synchronized (this.partsAwaitingChunkLoadMutex) {
            this.partsAwaitingChunkLoad.clear();
		}

		synchronized (this.orphanedPartsMutex) {
            this.orphanedParts.clear();
		}
	}

	/**
	 * Called when a chunk has finished loading. Adds all of the parts which are awaiting
	 * load to the list of parts which are orphans and therefore will be added to machines
	 * after the next world tick.
	 *
	 * @param chunkX Chunk X coordinate (world coordate >> 4) of the chunk that was loaded
	 * @param chunkZ Chunk Z coordinate (world coordate >> 4) of the chunk that was loaded
	 */
	public void onChunkLoaded(int chunkX, int chunkZ) {
		long chunkHash = ChunkPos.asLong(chunkX, chunkZ);
		if (this.partsAwaitingChunkLoad.containsKey(chunkHash)) {
			synchronized (this.partsAwaitingChunkLoadMutex) {
				if (this.partsAwaitingChunkLoad.containsKey(chunkHash)) {
					addAllOrphanedPartsThreadsafe(this.partsAwaitingChunkLoad.get(chunkHash));
                    this.partsAwaitingChunkLoad.remove(chunkHash);
				}
			}
		}
	}

	/**
	 * Registers a controller as dead. It will be cleaned up at the end of the next world tick.
	 * Note that a controller must shed all of its blocks before being marked as dead, or the system
	 * will complain at you.
	 *
	 * @param deadController The controller which is dead.
	 */
	public void addDeadController(IMultiblockControllerInternal deadController) {
		this.deadControllers.add(deadController);
	}

	/**
	 * Registers a controller as dirty - its list of attached blocks has changed, and it
	 * must be re-checked for assembly and, possibly, for orphans.
	 *
	 * @param dirtyController The dirty controller.
	 */
	public void addDirtyController(IMultiblockControllerInternal dirtyController) {
		this.dirtyControllers.add(dirtyController);
	}

	/**
	 * Use this only if you know what you're doing. You should rarely need to iterate
	 * over all controllers in a world!
	 *
	 * @return An (unmodifiable) set of controllers which are active in this world.
	 */
	public Set<IMultiblockControllerInternal> getControllers() {
		return Collections.unmodifiableSet(this.controllers);
	}

	/* *** PRIVATE HELPERS *** */

	private void addOrphanedPartThreadsafe(IMultiblockComponent part) {
		synchronized (this.orphanedPartsMutex) {
            this.orphanedParts.add(part);
		}
	}

	private void addAllOrphanedPartsThreadsafe(Collection<? extends IMultiblockComponent> parts) {
		if (parts.isEmpty()) {
			return;
		}
		synchronized (this.orphanedPartsMutex) {
            this.orphanedParts.addAll(parts);
		}
	}
}
