/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.factory.recipes;

import com.google.common.base.Preconditions;
import com.google.gson.JsonObject;
import forestry.api.recipes.IMoistenerRecipe;
import forestry.factory.features.FactoryRecipeTypes;
import net.minecraft.core.RegistryAccess;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.GsonHelper;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.Ingredient;
import net.minecraft.world.item.crafting.RecipeSerializer;
import net.minecraft.world.item.crafting.RecipeType;

public class MoistenerRecipe implements IMoistenerRecipe {
	private final ResourceLocation id;
	private final int timePerItem;
	private final Ingredient resource;
	private final ItemStack product;

	public MoistenerRecipe(ResourceLocation id, Ingredient resource, ItemStack product, int timePerItem) {
		Preconditions.checkNotNull(id, "Recipe identifier cannot be null");
		Preconditions.checkNotNull(resource, "Resource cannot be null");
		Preconditions.checkNotNull(product, "Product cannot be null");

		this.id = id;
		this.timePerItem = timePerItem;
		this.resource = resource;
		this.product = product;
	}

	@Override
	public int getTimePerItem() {
		return timePerItem;
	}

	@Override
	public Ingredient getInput() {
		return resource;
	}

	@Override
	public ItemStack getProduct() {
		return product;
	}

	@Override
	public ItemStack getResultItem(RegistryAccess registryAccess) {
		return this.product;
	}

	@Override
	public ResourceLocation getId() {
		return id;
	}

	@Override
	public RecipeSerializer<?> getSerializer() {
		return FactoryRecipeTypes.MOISTENER.serializer();
	}

	@Override
	public RecipeType<?> getType() {
		return FactoryRecipeTypes.MOISTENER.type();
	}

	public static class Serializer implements RecipeSerializer<MoistenerRecipe> {
		@Override
		public MoistenerRecipe fromJson(ResourceLocation recipeId, JsonObject json) {
			int timePerItem = GsonHelper.getAsInt(json, "time");
			Ingredient resource = RecipeSerializers.deserialize(json.get("resource"));
			ItemStack product = RecipeSerializers.item(GsonHelper.getAsJsonObject(json, "product"));

			return new MoistenerRecipe(recipeId, resource, product, timePerItem);
		}

		@Override
		public MoistenerRecipe fromNetwork(ResourceLocation recipeId, FriendlyByteBuf buffer) {
			int timePerItem = buffer.readVarInt();
			Ingredient resource = Ingredient.fromNetwork(buffer);
			ItemStack product = buffer.readItem();

			return new MoistenerRecipe(recipeId, resource, product, timePerItem);
		}

		@Override
		public void toNetwork(FriendlyByteBuf buffer, MoistenerRecipe recipe) {
			buffer.writeVarInt(recipe.timePerItem);
			recipe.resource.toNetwork(buffer);
			buffer.writeItem(recipe.product);
		}
	}
}
