package io.github.apace100.origins.command;

import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import io.github.apace100.origins.Origins;
import io.github.apace100.origins.component.OriginComponent;
import io.github.apace100.origins.networking.ModPackets;
import io.github.apace100.origins.origin.Origin;
import io.github.apace100.origins.origin.OriginLayer;
import io.github.apace100.origins.origin.OriginLayers;
import io.github.apace100.origins.origin.OriginRegistry;
import io.github.apace100.origins.registry.ModComponents;
import io.netty.buffer.Unpooled;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_2168;
import net.minecraft.class_2186;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_3222;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Random;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

public class OriginCommand {

	private enum TargetType {
		INVOKER,
		SPECIFY
	}

	public static void register(CommandDispatcher<class_2168> dispatcher) {
		dispatcher.register(
			method_9247("origin").requires(cs -> cs.method_9259(2))
				.then(method_9247("set")
					.then(method_9244("targets", class_2186.method_9308())
						.then(method_9244("layer", LayerArgumentType.layer())
							.then(method_9244("origin", OriginArgumentType.origin())
								.executes(OriginCommand::setOrigin))))
				)
				.then(method_9247("has")
					.then(method_9244("targets", class_2186.method_9308())
						.then(method_9244("layer", LayerArgumentType.layer())
							.then(method_9244("origin", OriginArgumentType.origin())
								.executes(OriginCommand::hasOrigin))))
				)
				.then(method_9247("get")
					.then(method_9244("target", class_2186.method_9305())
						.then(method_9244("layer", LayerArgumentType.layer())
							.executes(OriginCommand::getOrigin)
						)
					)
				)
				.then(method_9247("gui")
					.executes(commandContext -> OriginCommand.openMultipleLayerScreens(commandContext, TargetType.INVOKER))
					.then(method_9244("targets", class_2186.method_9308())
						.executes(commandContext -> OriginCommand.openMultipleLayerScreens(commandContext, TargetType.SPECIFY))
						.then(method_9244("layer", LayerArgumentType.layer())
							.executes(OriginCommand::openSingleLayerScreen)
						)
					)
				)
				.then(method_9247("random")
					.executes(commandContext -> OriginCommand.randomizeOrigins(commandContext, TargetType.INVOKER))
					.then(method_9244("targets", class_2186.method_9308())
						.executes(commandContext -> OriginCommand.randomizeOrigins(commandContext, TargetType.SPECIFY))
						.then(method_9244("layer", LayerArgumentType.layer())
							.executes(OriginCommand::randomizeOrigin)
						)
					)
				)
		);
	}

	/**
	 * 	Set the origin of the specified entities in the specified origin layer.
	 * 	@param commandContext the command context
	 * 	@return the number of players whose origin has been set
	 * 	@throws CommandSyntaxException if the entity is not found or if the entity is <b>not</b> an instance of {@link class_3222}
	 */
	private static int setOrigin(CommandContext<class_2168> commandContext) throws CommandSyntaxException {
		
		Collection<class_3222> targets = class_2186.method_9312(commandContext, "targets");
		OriginLayer originLayer = LayerArgumentType.getLayer(commandContext, "layer");
		Origin origin = OriginArgumentType.getOrigin(commandContext, "origin");
		class_2168 serverCommandSource = commandContext.getSource();
		
		int processedTargets = 0;
		
		if (origin.equals(Origin.EMPTY) || originLayer.getOrigins().contains(origin.getIdentifier())) {
			
			for (class_3222 target : targets) {
				
				OriginComponent originComponent = ModComponents.ORIGIN.get(target);
				boolean hadOriginBefore = originComponent.hadOriginBefore();
				
				originComponent.setOrigin(originLayer, origin);
				originComponent.sync();
				
				OriginComponent.partialOnChosen(target, hadOriginBefore, origin);
				
				processedTargets++;
				
			}
			
			if (processedTargets == 1) serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.set.success.single", targets.iterator().next().method_5476().getString(), class_2561.method_43471(originLayer.getTranslationKey()), origin.getName()), true);
			else {
				int finalProcessedTargets = processedTargets;
				serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.set.success.multiple", finalProcessedTargets, class_2561.method_43471(originLayer.getTranslationKey()), origin.getName()), true);
			}
			
		}
		
		else serverCommandSource.method_9213(class_2561.method_43469("commands.origin.unregistered_in_layer", origin.getIdentifier(), originLayer.getIdentifier()));
		
		return processedTargets;
		
	}

	/**
	 * 	Check if the specified entities has the specified origin in the specified origin layer.
	 * 	@param commandContext the command context
	 * 	@return the number of players that has the specified origin in the specified origin layer
	 * 	@throws CommandSyntaxException if the entity is not found or if the entity is <b>not</b> an instance of {@link class_3222}
	 */
	private static int hasOrigin(CommandContext<class_2168> commandContext) throws CommandSyntaxException {
		
		Collection<class_3222> targets = class_2186.method_9312(commandContext, "targets");
		OriginLayer originLayer = LayerArgumentType.getLayer(commandContext, "layer");
		Origin origin = OriginArgumentType.getOrigin(commandContext, "origin");
		class_2168 serverCommandSource = commandContext.getSource();
		
		int processedTargets = 0;
		
		if (origin.equals(Origin.EMPTY) || originLayer.getOrigins().contains(origin.getIdentifier())) {
			
			for (class_3222 target : targets) {
				OriginComponent originComponent = ModComponents.ORIGIN.get(target);
				if ((origin.equals(Origin.EMPTY) || originComponent.hasOrigin(originLayer)) && originComponent.getOrigin(originLayer).equals(origin)) processedTargets++;
			}
			
			if (processedTargets == 0) serverCommandSource.method_9213(class_2561.method_43471("commands.execute.conditional.fail"));
			else if (processedTargets == 1) serverCommandSource.method_9226(() -> class_2561.method_43471("commands.execute.conditional.pass"), true);
			else {
				int finalProcessedTargets = processedTargets;
				serverCommandSource.method_9226(() -> class_2561.method_43469("commands.execute.conditional.pass_count", finalProcessedTargets), true);
			}
			
		}
		
		else serverCommandSource.method_9213(class_2561.method_43469("commands.origin.unregistered_in_layer", origin.getIdentifier(), originLayer.getIdentifier()));
		
		return processedTargets;
		
	}

	/**
	 * 	Get the origin of the specified entity from the specified origin layer.
	 * 	@param commandContext the command context
	 * 	@return 1
	 * 	@throws CommandSyntaxException if the entity is not found or if the entity is <b>not</b> an instance of {@link class_3222}
	 */
	private static int getOrigin(CommandContext<class_2168> commandContext) throws CommandSyntaxException {
		
		class_3222 target = class_2186.method_9315(commandContext, "target");
		class_2168 serverCommandSource = commandContext.getSource();

		OriginComponent originComponent = ModComponents.ORIGIN.get(target);
		OriginLayer originLayer = LayerArgumentType.getLayer(commandContext, "layer");
		Origin origin = originComponent.getOrigin(originLayer);
		
		serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.get.result", target.method_5476().getString(), class_2561.method_43471(originLayer.getTranslationKey()), origin.getName(), origin.getIdentifier()), true);
		
		return 1;
		
	}

	/**
	 * 	Open the 'Choose Origin' screen for the specified origin layer to the specified entities.
	 * 	@param commandContext the command context
	 * 	@return the number of players that had the 'Choose Origin' screen opened for them
	 * 	@throws CommandSyntaxException if the entity is not found or if the entity is not an instance of {@link class_3222}
	 */
	private static int openSingleLayerScreen(CommandContext<class_2168> commandContext) throws CommandSyntaxException {

		class_2168 serverCommandSource = commandContext.getSource();
		Collection<class_3222> targets = class_2186.method_9312(commandContext, "targets");
		OriginLayer originLayer = LayerArgumentType.getLayer(commandContext, "layer");

		for (class_3222 target : targets) {
			openLayerScreen(target, originLayer);
		}

		serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.gui.layer", targets.size(), class_2561.method_43471(originLayer.getTranslationKey())), true);
		return targets.size();

	}

	/**
	 * 	Open the 'Choose Origin' screen for all the enabled origin layers to the specified entities.
	 * 	@param commandContext the command context
	 * 	@return the number of players that had the 'Choose Origin' screen opened for them
	 * 	@throws CommandSyntaxException if the entity is not found or if the entity is not an instance of {@link class_3222}
	 */
	private static int openMultipleLayerScreens(CommandContext<class_2168> commandContext, TargetType targetType) throws CommandSyntaxException {

		class_2168 serverCommandSource = commandContext.getSource();
		List<class_3222> targets = new ArrayList<>();
		List<OriginLayer> originLayers = OriginLayers.getLayers().stream().toList();

		switch (targetType) {
			case INVOKER -> targets.add(serverCommandSource.method_9207());
			case SPECIFY -> targets.addAll(class_2186.method_9312(commandContext, "targets"));
		}

		for (class_3222 target : targets) {
			for (OriginLayer originLayer : originLayers) {
				openLayerScreen(target, originLayer);
			}
		}

		serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.gui.all", targets.size()), false);
		return targets.size();

	}

	/**
	 * 	Randomize the origin of the specified entities in the specified origin layer.
	 * 	@param commandContext the command context
	 * 	@return the number of players that had their origin randomized in the specified origin layer
	 * 	@throws CommandSyntaxException if the entity is not found or if the entity is not an instance of {@link class_3222}
	 */
	private static int randomizeOrigin(CommandContext<class_2168> commandContext) throws CommandSyntaxException {

		class_2168 serverCommandSource = commandContext.getSource();
		Collection<class_3222> targets = class_2186.method_9312(commandContext, "targets");
		OriginLayer originLayer = LayerArgumentType.getLayer(commandContext, "layer");

		if (originLayer.isRandomAllowed()) {

			Origin origin = null;
			for (class_3222 target : targets) {
				origin = getRandomOrigin(target, originLayer);
			}

			if (targets.size() > 1) serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.random.success.multiple", targets.size(), class_2561.method_43471(originLayer.getTranslationKey())), true);
			else if (targets.size() == 1) {
				Origin finalOrigin = origin;
				serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.random.success.single", targets.iterator().next().method_5476().getString(), finalOrigin.getName(), class_2561.method_43471(originLayer.getTranslationKey())), false);
			}

			return targets.size();

		}

		else {
			serverCommandSource.method_9213(class_2561.method_43469("commands.origin.random.not_allowed", class_2561.method_43471(originLayer.getTranslationKey())));
			return 0;
		}

	}

	/**
	 * 	Randomize the origins of the specified entities in all of the origin layers that allows to be randomized.
	 * 	@param commandContext the command context
	 * 	@return the number of players that had their origins randomized in all of the origin layers that allows to be randomized
	 * 	@throws CommandSyntaxException if the entity is not found or if the entity is not an instance of {@link class_3222}
	 */
	private static int randomizeOrigins(CommandContext<class_2168> commandContext, TargetType targetType) throws CommandSyntaxException {

		class_2168 serverCommandSource = commandContext.getSource();
		List<class_3222> targets = new ArrayList<>();
		List<OriginLayer> originLayers = OriginLayers.getLayers().stream().filter(OriginLayer::isRandomAllowed).toList();

		switch (targetType) {
			case INVOKER -> targets.add(serverCommandSource.method_9207());
			case SPECIFY -> targets.addAll(class_2186.method_9312(commandContext, "targets"));
		}

		for (class_3222 target : targets) {
			for (OriginLayer originLayer : originLayers) {
				getRandomOrigin(target, originLayer);
			}
		}

		serverCommandSource.method_9226(() -> class_2561.method_43469("commands.origin.random.all", targets.size(), originLayers.size()), false);
		return targets.size();

	}

	private static void openLayerScreen(class_3222 target, OriginLayer originLayer) {

		OriginComponent originComponent = ModComponents.ORIGIN.get(target);
		class_2540 buffer = new class_2540(Unpooled.buffer());

		if (originLayer.isEnabled()) originComponent.setOrigin(originLayer, Origin.EMPTY);

		boolean originAutomaticallyAssigned = originComponent.checkAutoChoosingLayers(target, false);
		int originOptions = originLayer != null ? originLayer.getOriginOptionCount(target) : OriginLayers.getOriginOptionCount(target);
		originComponent.selectingOrigin(!originAutomaticallyAssigned || originOptions > 0);
		originComponent.sync();

		buffer.writeBoolean(false);
		ServerPlayNetworking.send(target, ModPackets.OPEN_ORIGIN_SCREEN, buffer);

	}

	private static Origin getRandomOrigin(class_3222 target, OriginLayer originLayer) {

		List<Origin> origins = originLayer.getRandomOrigins(target).stream().map(OriginRegistry::get).toList();
		OriginComponent originComponent = ModComponents.ORIGIN.get(target);
		Origin origin = origins.get(new Random().nextInt(origins.size()));

		boolean hadOriginBefore = originComponent.hadOriginBefore();
		boolean hadAllOrigins = originComponent.hasAllOrigins();

		originComponent.setOrigin(originLayer, origin);
		originComponent.checkAutoChoosingLayers(target, false);
		originComponent.sync();

		if (originComponent.hasAllOrigins() && !hadAllOrigins) OriginComponent.onChosen(target, hadOriginBefore);

		Origins.LOGGER.info(
			"Player {} was randomly assigned the origin {} for layer {}",
			target.method_5476().getString(),
			origin.getIdentifier().toString(),
			originLayer.getIdentifier().toString()
		);

		return origin;

	}
	
}
