package io.github.apace100.origins.origin;

import com.google.common.collect.Lists;
import com.google.gson.*;
import io.github.apace100.apoli.data.ApoliDataTypes;
import io.github.apace100.apoli.power.factory.condition.ConditionFactory;
import io.github.apace100.apoli.power.factory.condition.ConditionTypes;
import io.github.apace100.calio.data.SerializableData;
import io.github.apace100.calio.data.SerializableDataTypes;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

public class OriginLayer implements Comparable<OriginLayer> {

    private int order;
    private class_2960 identifier;
    private List<ConditionedOrigin> conditionedOrigins;
    private boolean enabled = false;

    private String nameTranslationKey;
    private String titleViewOriginTranslationKey;
    private String titleChooseOriginTranslationKey;
    private String missingOriginNameTranslationKey;
    private String missingOriginDescriptionTranslationKey;

    private boolean isRandomAllowed = false;
    private boolean doesRandomAllowUnchoosable = false;
    private List<class_2960> originsExcludedFromRandom = null;

    private class_2960 defaultOrigin = null;
    private boolean autoChooseIfNoChoice = false;

    private boolean hidden = false;
    private boolean overrideViewOriginTitle = false;
    private boolean overrideChooseOriginTitle = false;

    public String getOrCreateTranslationKey() {
        if(nameTranslationKey == null || nameTranslationKey.isEmpty()) {
            this.nameTranslationKey = "layer." + identifier.method_12836() + "." + identifier.method_12832() + ".name";
        }
        return nameTranslationKey;
    }

    public String getTranslationKey() {
        return getOrCreateTranslationKey();
    }

    public String getMissingOriginNameTranslationKey() {
        if(missingOriginNameTranslationKey == null || missingOriginNameTranslationKey.isEmpty()) {
            this.missingOriginNameTranslationKey = "layer." + identifier.method_12836() + "." + identifier.method_12832() + ".missing_origin.name";
        }
        return missingOriginNameTranslationKey;
    }

    public String getTitleViewOriginTranslationKey() {
        if(titleViewOriginTranslationKey == null || titleViewOriginTranslationKey.isEmpty()) {
            this.titleViewOriginTranslationKey = "layer." + identifier.method_12836() + "." + identifier.method_12832() + ".view_origin.name";
        }
        return titleViewOriginTranslationKey;
    }

    public boolean shouldOverrideViewOriginTitle() {
        return overrideViewOriginTitle;
    }

    public String getTitleChooseOriginTranslationKey() {
        if(titleChooseOriginTranslationKey == null || titleChooseOriginTranslationKey.isEmpty()) {
            this.titleChooseOriginTranslationKey = "layer." + identifier.method_12836() + "." + identifier.method_12832() + ".choose_origin.name";
        }
        return titleChooseOriginTranslationKey;
    }

    public boolean shouldOverrideChooseOriginTitle() {
        return overrideChooseOriginTitle;
    }

    public String getMissingOriginDescriptionTranslationKey() {
        if(missingOriginDescriptionTranslationKey == null || missingOriginDescriptionTranslationKey.isEmpty()) {
            this.missingOriginDescriptionTranslationKey = "layer." + identifier.method_12836() + "." + identifier.method_12832() + ".missing_origin.description";
        }
        return missingOriginDescriptionTranslationKey;
    }

    public class_2960 getIdentifier() {
        return identifier;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public boolean hasDefaultOrigin() {
        return defaultOrigin != null;
    }

    public class_2960 getDefaultOrigin() {
        return defaultOrigin;
    }

    public boolean shouldAutoChoose() {
        return autoChooseIfNoChoice;
    }

    public List<class_2960> getOrigins() {
        return conditionedOrigins.stream().flatMap(co -> co.getOrigins().stream()).filter(OriginRegistry::contains).collect(Collectors.toList());
    }

    public List<class_2960> getOrigins(class_1657 playerEntity) {
        return conditionedOrigins.stream().filter(co -> co.isConditionFulfilled(playerEntity)).flatMap(co -> co.getOrigins().stream()).filter(OriginRegistry::contains).collect(Collectors.toList());
    }

    public int getOriginOptionCount(class_1657 playerEntity) {
        long choosableOrigins = getOrigins(playerEntity).stream().map(OriginRegistry::get).filter(Origin::isChoosable).count();
        if(isRandomAllowed && getRandomOrigins(playerEntity).size() > 0) {
            choosableOrigins++;
        }
        return (int)choosableOrigins;
    }

    public boolean contains(Origin origin) {
        return conditionedOrigins.stream().anyMatch(co -> co.getOrigins().stream().anyMatch(o -> o.equals(origin.getIdentifier())));
    }

    public boolean contains(Origin origin, class_1657 playerEntity) {
        return conditionedOrigins.stream().filter(co -> co.isConditionFulfilled(playerEntity)).anyMatch(co -> co.getOrigins().stream().anyMatch(o -> o.equals(origin.getIdentifier())));
    }

    public boolean isRandomAllowed() {
        return isRandomAllowed;
    }

    public boolean isHidden() {
        return hidden;
    }

    public List<class_2960> getRandomOrigins(class_1657 playerEntity) {
        return conditionedOrigins.stream().filter(co -> co.isConditionFulfilled(playerEntity)).flatMap(co -> co.getOrigins().stream()).filter(OriginRegistry::contains).filter(o -> !originsExcludedFromRandom.contains(o)).filter(id -> doesRandomAllowUnchoosable || OriginRegistry.get(id).isChoosable()).collect(Collectors.toList());
    }

    public void merge(JsonObject json) {
        if(json.has("order")) {
            this.order = json.get("order").getAsInt();
        }
        if(json.has("enabled")) {
            this.enabled = json.get("enabled").getAsBoolean();
        }
        if(json.has("origins")) {
            JsonArray originArray = json.getAsJsonArray("origins");
            originArray.forEach(je -> this.conditionedOrigins.add(ConditionedOrigin.read(je)));
        }
        if(json.has("name")) {
            this.nameTranslationKey = class_3518.method_15253(json, "name", "");
        }
        if(json.has("gui_title")) {
            JsonObject guiTitleObj = json.getAsJsonObject("gui_title");
            if(guiTitleObj.has("view_origin")) {
                this.titleViewOriginTranslationKey = class_3518.method_15253(guiTitleObj, "view_origin", "");
                this.overrideViewOriginTitle = true;
            }
            if(guiTitleObj.has("choose_origin")) {
                this.titleChooseOriginTranslationKey = class_3518.method_15253(guiTitleObj, "choose_origin", "");
                this.overrideChooseOriginTitle = true;
            }
        }
        if(json.has("missing_name")) {
            this.missingOriginNameTranslationKey = class_3518.method_15253(json, "missing_name", "");
        }
        if(json.has("missing_description")) {
            this.missingOriginDescriptionTranslationKey = class_3518.method_15253(json, "missing_description", "");
        }
        if(json.has("allow_random")) {
            this.isRandomAllowed = class_3518.method_15270(json, "allow_random");
        }
        if(json.has("allow_random_unchoosable")) {
            this.doesRandomAllowUnchoosable = class_3518.method_15270(json, "allow_random_unchoosable");
        }
        if(json.has("exclude_random") && json.get("exclude_random").isJsonArray()) {
            boolean replaceExclude = class_3518.method_15258(json, "replace_exclude_random", false);
            if(replaceExclude) {
                originsExcludedFromRandom.clear();
            }
            JsonArray excludeRandomArray = json.getAsJsonArray("exclude_random");
            excludeRandomArray.forEach(je -> originsExcludedFromRandom.add(class_2960.method_12829(je.getAsString())));
        }
        if(json.has("default_origin")) {
            this.defaultOrigin = new class_2960(class_3518.method_15265(json, "default_origin"));
        }
        if(json.has("auto_choose")) {
            this.autoChooseIfNoChoice = class_3518.method_15270(json, "auto_choose");
        }
        if(json.has("hidden")) {
            this.hidden = class_3518.method_15270(json, "hidden");
        }
    }

    @Override
    public int hashCode() {
        return identifier.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if(obj == this) {
            return true;
        } else if(!(obj instanceof OriginLayer)) {
            return false;
        } else {
            return identifier.equals(((OriginLayer)obj).identifier);
        }
    }

    @Override
    public int compareTo(OriginLayer o) {
        return Integer.compare(order, o.order);
    }

    public void write(class_2540 buffer) {
        buffer.method_10814(identifier.toString());
        buffer.writeInt(order);
        buffer.writeBoolean(enabled);
        buffer.writeInt(conditionedOrigins.size());
        conditionedOrigins.forEach(co -> co.write(buffer));
        buffer.method_10814(getOrCreateTranslationKey());
        buffer.method_10814(getTitleViewOriginTranslationKey());
        buffer.method_10814(getTitleChooseOriginTranslationKey());
        buffer.method_10814(getMissingOriginNameTranslationKey());
        buffer.method_10814(getMissingOriginDescriptionTranslationKey());
        buffer.writeBoolean(isRandomAllowed());
        if(isRandomAllowed()) {
            buffer.writeBoolean(doesRandomAllowUnchoosable);
            buffer.writeInt(originsExcludedFromRandom.size());
            originsExcludedFromRandom.forEach(buffer::method_10812);
        }
        buffer.writeBoolean(hasDefaultOrigin());
        if(hasDefaultOrigin()) {
            buffer.method_10812(defaultOrigin);
        }
        buffer.writeBoolean(autoChooseIfNoChoice);
        buffer.writeBoolean(hidden);
        buffer.writeBoolean(overrideViewOriginTitle);
        buffer.writeBoolean(overrideChooseOriginTitle);
    }

    @Environment(EnvType.CLIENT)
    public static OriginLayer read(class_2540 buffer) {
        OriginLayer layer = new OriginLayer();
        layer.identifier = class_2960.method_12829(buffer.method_19772());
        layer.order = buffer.readInt();
        layer.enabled = buffer.readBoolean();
        int conditionedOriginCount = buffer.readInt();
        layer.conditionedOrigins = new ArrayList<>(conditionedOriginCount);
        for(int i = 0; i < conditionedOriginCount; i++) {
            layer.conditionedOrigins.add(ConditionedOrigin.read(buffer));
        }
        layer.nameTranslationKey = buffer.method_19772();
        layer.titleViewOriginTranslationKey = buffer.method_19772();
        layer.titleChooseOriginTranslationKey = buffer.method_19772();
        layer.missingOriginNameTranslationKey = buffer.method_19772();
        layer.missingOriginDescriptionTranslationKey = buffer.method_19772();
        layer.isRandomAllowed = buffer.readBoolean();
        if(layer.isRandomAllowed) {
            layer.doesRandomAllowUnchoosable = buffer.readBoolean();
            int excludedSize = buffer.readInt();
            layer.originsExcludedFromRandom = new LinkedList<>();
            for(int i = 0; i < excludedSize; i++) {
                layer.originsExcludedFromRandom.add(buffer.method_10810());
            }
        }
        if(buffer.readBoolean()) {
            layer.defaultOrigin = buffer.method_10810();
        }
        layer.autoChooseIfNoChoice = buffer.readBoolean();
        layer.hidden = buffer.readBoolean();
        layer.overrideViewOriginTitle = buffer.readBoolean();
        layer.overrideChooseOriginTitle = buffer.readBoolean();
        return layer;
    }

    public static OriginLayer fromJson(class_2960 id, JsonObject json) {
        int order = class_3518.method_15282(json, "order", OriginLayers.size());
        if(!json.has("origins") || !json.get("origins").isJsonArray()) {
            throw new JsonParseException("Origin layer JSON requires \"origins\" array of origin IDs to include in the layer.");
        }
        JsonArray originArray = json.getAsJsonArray("origins");
        List<ConditionedOrigin> list = new ArrayList<>(originArray.size());
        originArray.forEach(je -> list.add(ConditionedOrigin.read(je)));
        boolean enabled = class_3518.method_15258(json, "enabled", true);
        OriginLayer layer = new OriginLayer();
        layer.order = order;
        layer.conditionedOrigins = list;
        layer.enabled = enabled;
        layer.identifier = id;
        layer.nameTranslationKey = class_3518.method_15253(json, "name", "");
        if(json.has("gui_title") && json.get("gui_title").isJsonObject()) {
            JsonObject guiTitleObj = json.getAsJsonObject("gui_title");
            if(guiTitleObj.has("view_origin")) {
                layer.titleViewOriginTranslationKey = class_3518.method_15253(guiTitleObj, "view_origin", "");
                layer.overrideViewOriginTitle = true;
            }
            if(guiTitleObj.has("choose_origin")) {
                layer.titleChooseOriginTranslationKey = class_3518.method_15253(guiTitleObj, "choose_origin", "");
                layer.overrideChooseOriginTitle = true;
            }
        }
        layer.missingOriginNameTranslationKey = class_3518.method_15253(json, "missing_name", "");
        layer.missingOriginDescriptionTranslationKey = class_3518.method_15253(json, "missing_description", "");
        layer.isRandomAllowed = class_3518.method_15258(json, "allow_random", false);
        layer.doesRandomAllowUnchoosable = class_3518.method_15258(json, "allow_random_unchoosable", false);
        layer.originsExcludedFromRandom = new LinkedList<>();
        if(json.has("exclude_random") && json.get("exclude_random").isJsonArray()) {
            JsonArray excludeRandomArray = json.getAsJsonArray("exclude_random");
            excludeRandomArray.forEach(je -> layer.originsExcludedFromRandom.add(class_2960.method_12829(je.getAsString())));
        }
        if(json.has("default_origin")) {
            layer.defaultOrigin = new class_2960(class_3518.method_15265(json, "default_origin"));
        }
        layer.autoChooseIfNoChoice = class_3518.method_15258(json, "auto_choose", false);
        layer.hidden = class_3518.method_15258(json, "hidden", false);
        return layer;
    }

    public static class ConditionedOrigin {
        private final ConditionFactory<class_1297>.Instance condition;
        private final List<class_2960> origins;

        public ConditionedOrigin(ConditionFactory<class_1297>.Instance condition, List<class_2960> origins) {
            this.condition = condition;
            this.origins = origins;
        }

        public boolean isConditionFulfilled(class_1657 playerEntity) {
            return condition == null || condition.test(playerEntity);
        }

        public List<class_2960> getOrigins() {
            return origins;
        }
        private static final SerializableData conditionedOriginObjectData = new SerializableData()
            .add("condition", ApoliDataTypes.ENTITY_CONDITION)
            .add("origins", SerializableDataTypes.IDENTIFIERS);

        public void write(class_2540 buffer) {
            buffer.writeBoolean(condition != null);
            if(condition != null)
                condition.write(buffer);
            buffer.writeInt(origins.size());
            origins.forEach(buffer::method_10812);
        }

        @Environment(EnvType.CLIENT)
        public static ConditionedOrigin read(class_2540 buffer) {
            ConditionFactory<class_1297>.Instance condition = null;
            if(buffer.readBoolean()) {
                condition = ConditionTypes.ENTITY.read(buffer);
            }
            int originCount = buffer.readInt();
            List<class_2960> originList = new ArrayList<>(originCount);
            for(int i = 0; i < originCount; i++) {
                originList.add(buffer.method_10810());
            }
            return new ConditionedOrigin(condition, originList);
        }

        @SuppressWarnings("unchecked")
        public static ConditionedOrigin read(JsonElement element) {
            if(element.isJsonPrimitive()) {
                JsonPrimitive elemPrimitive = element.getAsJsonPrimitive();
                if(elemPrimitive.isString()) {
                    return new ConditionedOrigin(null, Lists.newArrayList(class_2960.method_12829(elemPrimitive.getAsString())));
                }
                throw new JsonParseException("Expected origin in layer to be either a string or an object.");
            } else if(element.isJsonObject()) {
                SerializableData.Instance data = conditionedOriginObjectData.read(element.getAsJsonObject());
                return new ConditionedOrigin((ConditionFactory<class_1297>.Instance)data.get("condition"), (List<class_2960>)data.get("origins"));
            }
            throw new JsonParseException("Expected origin in layer to be either a string or an object.");
        }
    }
}
