package foundry.veil.api.client.imgui;


import foundry.veil.Veil;
import foundry.veil.api.client.editor.EditorManager;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.api.client.render.framebuffer.AdvancedFbo;
import foundry.veil.impl.client.imgui.AdvancedFboImGuiAreaImpl;
import imgui.ImFont;
import imgui.ImGui;
import imgui.ImVec4;
import imgui.flag.ImGuiCol;
import imgui.flag.ImGuiStyleVar;
import net.minecraft.class_156;
import net.minecraft.class_2477;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_2568;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3544;
import net.minecraft.class_437;
import net.minecraft.class_5224;
import net.minecraft.class_5225;
import net.minecraft.class_5348;
import net.minecraft.class_5481;
import net.minecraft.class_746;
import net.minecraft.network.chat.*;
import org.jetbrains.annotations.ApiStatus;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Locale;
import java.util.function.Consumer;

/**
 * Extra components and helpers for ImGui.
 *
 * @author ryan, Ocelot
 */
public class VeilImGuiUtil {

    private static final ImGuiCharSink IM_GUI_CHAR_SINK = new ImGuiCharSink();
    private static final class_5225 IM_GUI_SPLITTER = new class_5225((charId, style) -> getStyleFont(style).getCharAdvance(charId));

    public static final class_2960 ICON_FONT = Veil.veilPath("remixicon");

    /**
     * Displays a (?) with a hover tooltip. Useful for example information.
     *
     * @param text The tooltip text
     */
    public static void tooltip(String text) {
        ImGui.textColored(0xFF555555, "(?)");
        if (ImGui.isItemHovered()) {
            ImGui.beginTooltip();
            ImGui.pushTextWrapPos(ImGui.getFontSize() * 35.0f);
            ImGui.textColored(0xFFFFFFFF, text);
            ImGui.popTextWrapPos();
            ImGui.endTooltip();
        }
    }

    /**
     * Sets the tooltip to the specified component
     *
     * @param text The text to render
     */
    public static void setTooltip(class_5348 text) {
        ImGui.beginTooltip();
        component(text);
        ImGui.endTooltip();
    }

    /**
     * Fully renders Minecraft text into ImGui.
     *
     * @param text The text to render
     */
    public static void component(class_5348 text) {
        component(text, Float.POSITIVE_INFINITY);
    }

    /**
     * Fully renders wrapped Minecraft text into ImGui.
     *
     * @param text      The text to render
     * @param wrapWidth The width to wrap to
     */
    public static void component(class_5348 text, float wrapWidth) {
        IM_GUI_CHAR_SINK.reset();
        for (class_5481 part : class_2477.method_10517().method_30933(IM_GUI_SPLITTER.method_27495(text, (int) wrapWidth, class_2583.field_24360))) {
            part.accept(IM_GUI_CHAR_SINK);
            IM_GUI_CHAR_SINK.finish();
            ImGui.newLine();
        }
    }

    /**
     * Renders an icon with the remixicon font
     *
     * @param code The icon code (ex. &#xED0F;)
     */
    public static void icon(int code) {
        ImGui.pushFont(VeilRenderSystem.renderer().getEditorManager().getFont(ICON_FONT, false, false));
        ImGui.text("" + (char) code);
        ImGui.popFont();
    }

    /**
     * Renders an icon with the remixicon font and a color
     *
     * @param code  The icon code (ex. &#xED0F;)
     * @param color The color of the icon
     */
    public static void icon(int code, int color) {
        ImGui.pushFont(VeilRenderSystem.renderer().getEditorManager().getFont(ICON_FONT, false, false));
        ImGui.textColored(color, "" + (char) code);
        ImGui.popFont();
    }

    /**
     * Helper to draw centered text.
     *
     * @param text  The text to render
     * @param width The width of the area to center on
     */
    public static void textCentered(String text, float width) {
        ImGui.setCursorPosX(ImGui.getCursorPosX() + (width - ImGui.getFont().calcTextSizeAX(ImGui.getFontSize(), Float.MAX_VALUE, 0, text)) / 2);
        ImGui.text(text);
    }

    /**
     * Displays a resource location with a dimmed namespace
     *
     * @param loc The resource location
     */
    public static void resourceLocation(class_2960 loc) {
        ImGui.beginGroup();
        ImGui.textColored(colorOf(loc.method_12836()), loc.method_12836() + ":");

        ImGui.pushStyleVar(ImGuiStyleVar.ItemSpacing, 0, 0);
        ImGui.sameLine();
        ImGui.text(loc.method_12832());
        ImGui.popStyleVar();

        ImGui.endGroup();

        if (ImGui.beginPopupContextItem("" + loc)) {
            if (ImGui.selectable("##copy_location")) {
                ImGui.setClipboardText(loc.toString());
            }

            ImGui.pushStyleVar(ImGuiStyleVar.ItemSpacing, 0, 0);
            ImGui.setItemAllowOverlap();
            ImGui.sameLine();
            VeilImGuiUtil.icon(0xEB91);
            ImGui.sameLine();
            ImGui.popStyleVar();
            ImGui.text("Copy Location");
            ImGui.endPopup();
        }
    }

    /**
     * Creates a rendering area of the specified size.
     *
     * @param width    The width of the area
     * @param height   The height of the area
     * @param renderer The renderer inside the area
     * @return A texture ID that can be displayed in ImGui
     */
    public static int renderArea(int width, int height, Consumer<AdvancedFbo> renderer) {
        ImVec4 colors = ImGui.getStyle().getColors()[ImGuiCol.FrameBg];
        AdvancedFbo fbo = AdvancedFboImGuiAreaImpl.allocate(width, height);
        fbo.bind(true);
        fbo.clear(colors.x, colors.y, colors.z, colors.w, fbo.getClearMask());
        renderer.accept(fbo);
        AdvancedFbo.unbind();
        return fbo.getColorTextureAttachment(0).method_4624();
    }

    /**
     * Obtains the color of the modid
     *
     * @param modid The modid to get the color of
     * @return color The color based on the hash of the modid
     */
    public static int colorOf(String modid) {
        int color = (modid.hashCode() & 0xAAAAAA) + 0x444444;

//        Color dark = new Color(0.6F, 0.6F, 0.6F);
//        Color c = new Color(0xFF | color << 8);
//        c.mix(dark, 0.35F);
//        return 0xFF | (c.rgb() & 0xFFFFFF) << 8;

        int r = (int) ((color & 0xFF) * 0.65F + 53);
        int g = (int) (((color >> 8) & 0xFF) * 0.65F + 53);
        int b = (int) (((color >> 16) & 0xFF) * 0.65F + 53);
        return 0xFF000000 | r << 16 | g << 8 | b;
    }

    /**
     * Retrieves the ImGui font to use for the specified Minecraft style.
     *
     * @param style The style to get the font for
     * @return The ImFont to use
     */
    public static ImFont getStyleFont(class_2583 style) {
        return VeilRenderSystem.renderer().getEditorManager().getFont(class_2583.field_24359.equals(style.method_27708()) ? EditorManager.DEFAULT_FONT : style.method_27708(), style.method_10984(), style.method_10966());
    }

    /**
     * Retrieves the ARGB color for the specified ImGui style color.
     *
     * @param color The
     * @return The ImFont to use
     */
    public static int getColor(int color) {
        ImVec4 colors = ImGui.getStyle().getColors()[color];
        return (int) (colors.w * 255) << 24 | (int) (colors.x * 255) << 16 | (int) (colors.y * 255) << 8 | (int) (colors.z * 255);
    }

    /**
     * @return A string splitter for ImGui fonts
     */
    public static class_5225 getStringSplitter() {
        return IM_GUI_SPLITTER;
    }

    @ApiStatus.Internal
    private static class ImGuiCharSink implements class_5224 {

        private ImFont font;
        private int textColor;
        private class_2568 hoverEvent;
        private class_2558 clickEvent;

        private final StringBuilder buffer;

        private ImGuiCharSink() {
            this.buffer = new StringBuilder();
            this.reset();
        }

        public void reset() {
            this.font = ImGui.getFont();
            this.textColor = getColor(ImGuiCol.Text);
            this.buffer.setLength(0);
            this.hoverEvent = null;
            this.clickEvent = null;
        }

        @Override
        public boolean accept(int positionInCurrentSequence, class_2583 style, int codePoint) {
            ImFont font = getStyleFont(style);
            int styleColor = style.method_10973() != null ? style.method_10973().method_27716() : this.textColor;
            if (font != this.font || styleColor != this.textColor || style.method_10969() != this.hoverEvent || style.method_10970() != this.clickEvent) {
                if (!this.buffer.isEmpty()) {
                    this.finish();
                }
                this.font = getStyleFont(style);
                this.textColor = styleColor;
                this.hoverEvent = style.method_10969();
                this.clickEvent = style.method_10970();
            }
            this.buffer.appendCodePoint(codePoint);
            return true;
        }

        public void finish() {
            if (!this.buffer.isEmpty()) {
                ImGui.pushStyleVar(ImGuiStyleVar.ItemSpacing, 0, 0);
                ImGui.pushFont(this.font);
                ImGui.textColored(0xFF000000 | (this.textColor & 0xFF0000) >> 16 | (this.textColor & 0xFF00) | (this.textColor & 0xFF) << 16, this.buffer.toString());

                if (ImGui.isItemClicked() && this.clickEvent != null) {
                    this.handleClick();
                }
                if (ImGui.isItemHovered() && this.hoverEvent != null) {
                    this.handleHover();
                }

                ImGui.sameLine();
                ImGui.popFont();
                ImGui.popStyleVar();
                this.buffer.setLength(0);
            }
        }

        private void handleClick() {
            class_310 minecraft = class_310.method_1551();
            String value = this.clickEvent.method_10844();
            if (this.clickEvent.method_10845() == class_2558.class_2559.field_11749) {
                try {
                    URI uri = new URI(value);
                    String scheme = uri.getScheme();
                    if (scheme == null) {
                        throw new URISyntaxException(value, "Missing protocol");
                    }

                    if (!scheme.equalsIgnoreCase("http") && !scheme.equalsIgnoreCase("https")) {
                        throw new URISyntaxException(value, "Unsupported protocol: " + scheme.toLowerCase(Locale.ROOT));
                    }

                    class_156.method_668().method_673(uri);
                } catch (URISyntaxException e) {
                    Veil.LOGGER.error("Can't open url for {}", this.clickEvent, e);
                }
                return;
            }

            if (this.clickEvent.method_10845() == class_2558.class_2559.field_11746) {
                class_156.method_668().method_673(new File(value).toURI());
                return;
            }

            // TODO
            if (this.clickEvent.method_10845() == class_2558.class_2559.field_11745) {
                return;
            }

            if (this.clickEvent.method_10845() == class_2558.class_2559.field_11750) {
                String s = class_3544.method_57180(this.clickEvent.method_10844());
                if (s.startsWith("/")) {
                    class_746 player = class_310.method_1551().field_1724;
                    if (player != null && !player.field_3944.method_45731(s.substring(1))) {
                        Veil.LOGGER.error("Not allowed to run command with signed argument from click event: '{}'", s);
                    }
                } else {
                    Veil.LOGGER.error("Failed to run command without '/' prefix from click event: '{}'", s);
                }
                return;
            }

            if (this.clickEvent.method_10845() == class_2558.class_2559.field_21462) {
                minecraft.field_1774.method_1455(value);
                return;
            }

            Veil.LOGGER.error("Don't know how to handle {}", this.clickEvent);
        }

        private void handleHover() {
            class_310 minecraft = class_310.method_1551();
            class_2568.class_5249 stack = this.hoverEvent.method_10891(class_2568.class_5247.field_24343);
            if (stack != null) {
                ImGui.beginTooltip();
                List<class_2561> tooltip = class_437.method_25408(minecraft, stack.method_27683());
                for (class_2561 line : tooltip) {
                    component(line, ImGui.getFontSize() * 35.0f);
                }
                ImGui.endTooltip();
                return;
            }

            class_2568.class_5248 entity = this.hoverEvent.method_10891(class_2568.class_5247.field_24344);
            if (entity != null) {
                if (minecraft.field_1690.field_1827) {
                    ImGui.beginTooltip();
                    List<class_2561> tooltip = entity.method_27682();
                    for (class_2561 line : tooltip) {
                        component(line, ImGui.getFontSize() * 35.0f);
                    }
                    ImGui.endTooltip();
                }
                return;
            }

            class_2561 showText = this.hoverEvent.method_10891(class_2568.class_5247.field_24342);
            if (showText != null) {
                ImGui.beginTooltip();
                component(showText, ImGui.getFontSize() * 35.0f);
                ImGui.endTooltip();
            }
        }
    }
}
