package foundry.veil.api.client.necromancer;

import org.jetbrains.annotations.Nullable;
import org.joml.*;

import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_2350;
import net.minecraft.class_3532;

public class Bone {

    public Vector3f position, previousPosition, basePosition;
    public Vector3f rotationPoint, previousRotationPoint, baseRotationPoint;
    public Quaternionf rotation, previousRotation, baseRotation;
    public Vector3f size, previousSize, baseSize;
    public Vector4f color, previousColor, baseColor;

    @Nullable
    public Bone parent;
    public List<Bone> children;

    public final String identifier;
    private int index;

    // list of all parents, starting from the root and going down
    public List<Bone> parentChain;

    public Bone(String identifier) {
        this.identifier = identifier;

        this.position = new Vector3f(0.0F);
        this.previousPosition = new Vector3f(0.0F);
        this.basePosition = new Vector3f(0.0F);

        this.rotationPoint = new Vector3f(0.0F);
        this.previousRotationPoint = new Vector3f(0.0F);
        this.baseRotationPoint = new Vector3f(0.0F);

        this.rotation = new Quaternionf();
        this.previousRotation = new Quaternionf();
        this.baseRotation = new Quaternionf();

        this.size = new Vector3f(1.0F);
        this.previousSize = new Vector3f(1.0F);
        this.baseSize = new Vector3f(1.0F);

        this.color = new Vector4f(1.0F);
        this.previousColor = new Vector4f(1.0F);
        this.baseColor = new Vector4f(1.0F);

        this.children = new ArrayList<>();
        this.parentChain = new ArrayList<>();
    }

    protected void setIndex(int index) {
        this.index = index;
    }

    public int getIndex() {
        return this.index;
    }

    public void setBaseAttributes(Vector3fc pos, Quaternionfc rotation, Vector3f rotationPoint, Vector3fc scale, Vector4fc color) {
        this.basePosition.set(pos);
        this.position.set(this.basePosition);
        this.previousPosition.set(this.basePosition);

        this.baseSize.set(scale);
        this.size.set(this.baseSize);
        this.previousSize.set(this.baseSize);

        this.baseRotation.set(rotation);
        this.rotation.set(this.baseRotation);
        this.previousRotation.set(this.baseRotation);

        this.baseRotationPoint.set(rotationPoint);
        this.rotationPoint.set(this.baseRotationPoint);
        this.previousRotationPoint.set(this.baseRotationPoint);

        this.baseColor.set(color);
        this.color.set(this.baseColor);
        this.previousColor.set(this.baseColor);
    }

    public void reset() {
        this.position.set(this.basePosition);
        this.rotation.set(this.baseRotation);
        this.rotationPoint.set(this.baseRotationPoint);
        this.size.set(this.baseSize);
        this.color.set(this.baseColor);
    }

    protected void updatePreviousAttributes() {
        this.previousPosition.set(this.position);
        this.previousRotation.set(this.rotation);
        this.previousRotationPoint.set(this.rotationPoint);
        this.previousSize.set(this.size);
        this.previousColor.set(this.color);
    }

    public Matrix4x3f getModelTransform(Matrix4x3f matrix, Quaternionf orientation, float partialTicks) {
        for (Bone bone : this.parentChain) {
            bone.getLocalTransform(matrix, orientation, partialTicks);
        }
        this.getLocalTransform(matrix, orientation, partialTicks);
        return matrix;
    }

    public Matrix4x3f getModelTransform(Matrix4x3f matrix, float partialTicks) {
        return this.getModelTransform(matrix, new Quaternionf(), partialTicks);
    }

    public void getLocalTransform(Matrix4x3f matrix, Quaternionf orientation, float partialTicks) {
        float x = class_3532.method_16439(partialTicks, this.previousPosition.x, this.position.x);
        float y = class_3532.method_16439(partialTicks, this.previousPosition.y, this.position.y);
        float z = class_3532.method_16439(partialTicks, this.previousPosition.z, this.position.z);
        matrix.translate(x, y, z);

        this.previousRotation.slerp(this.rotation, partialTicks, orientation);
        float rotationPointX = class_3532.method_16439(partialTicks, this.previousRotationPoint.x, this.rotationPoint.x);
        float rotationPointY = class_3532.method_16439(partialTicks, this.previousRotationPoint.y, this.rotationPoint.y);
        float rotationPointZ = class_3532.method_16439(partialTicks, this.previousRotationPoint.z, this.rotationPoint.z);

        matrix.translate(rotationPointX, rotationPointY, rotationPointZ);
        matrix.rotate(orientation.normalize());
        matrix.translate(-rotationPointX, -rotationPointY, -rotationPointZ);

        // technically wrong but whatever
        matrix.scale(
                class_3532.method_16439(partialTicks, this.previousSize.x, this.size.x),
                class_3532.method_16439(partialTicks, this.previousSize.y, this.size.y),
                class_3532.method_16439(partialTicks, this.previousSize.z, this.size.z)
        );
    }

    public void getLocalTransform(Matrix4x3f matrix, float partialTicks) {
        this.getLocalTransform(matrix, new Quaternionf(), partialTicks);
    }

    public void getColor(Vector4f color, float partialTicks) {
        this.previousColor.lerp(this.color, partialTicks, color);
    }

    protected void tick(float deltaTime) {
    }

    public void addChild(Bone child) {
        if (child.parent != null) {
            child.parent.children.remove(child);
        }

        this.children.add(child);
        child.parent = this;
    }

    public void setParent(Bone parent) {
        this.parent = parent;
        parent.children.add(this);
    }

    public Bone rotate(float angle, class_2350.class_2351 axis) {
        switch (axis) {
            case field_11048 -> this.rotation.rotateX(angle);
            case field_11052 -> this.rotation.rotateY(angle);
            case field_11051 -> this.rotation.rotateZ(angle);
        }
        return this;
    }

    public Bone rotateDeg(float angle, class_2350.class_2351 axis) {
        switch (axis) {
            case field_11048 -> this.rotation.rotateX(angle * class_3532.field_29847);
            case field_11052 -> this.rotation.rotateY(angle * class_3532.field_29847);
            case field_11051 -> this.rotation.rotateZ(angle * class_3532.field_29847);
        }
        return this;
    }

    public Bone offset(float x, float y, float z) {
        this.position.add(x, y, z);
        return this;
    }

    public Bone offsetX(float x) {
        this.position.add(x, 0, 0);
        return this;
    }

    public Bone offsetY(float y) {
        this.position.add(0, y, 0);
        return this;
    }

    public Bone offsetZ(float z) {
        this.position.add(0, 0, z);
        return this;
    }
}
