package foundry.veil.api.client.render;

import com.mojang.blaze3d.systems.RenderSystem;
import foundry.veil.api.client.render.framebuffer.AdvancedFbo;
import foundry.veil.api.compat.FlashbackCompat;
import foundry.veil.api.compat.SodiumCompat;
import foundry.veil.ext.RenderTargetExtension;
import foundry.veil.impl.client.render.perspective.FlashbackAccess;
import foundry.veil.impl.client.render.perspective.IrisPipelineAccess;
import foundry.veil.impl.client.render.perspective.LevelPerspectiveCamera;
import foundry.veil.mixin.perspective.accessor.GameRendererAccessor;
import foundry.veil.mixin.perspective.accessor.LevelRendererAccessor;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.joml.*;

import java.util.concurrent.atomic.AtomicInteger;
import net.minecraft.class_1041;
import net.minecraft.class_1297;
import net.minecraft.class_239;
import net.minecraft.class_243;
import net.minecraft.class_310;
import net.minecraft.class_3695;
import net.minecraft.class_4184;
import net.minecraft.class_4587;
import net.minecraft.class_4597;
import net.minecraft.class_4604;
import net.minecraft.class_6854;
import net.minecraft.class_757;
import net.minecraft.class_761;
import net.minecraft.class_9779;

/**
 * Renders the level from different perspectives.
 *
 * @author Ocelot
 */
public final class VeilLevelPerspectiveRenderer {

    private static final LevelPerspectiveCamera CAMERA = new LevelPerspectiveCamera();
    private static final Matrix4f TRANSFORM = new Matrix4f();
    private static final AtomicInteger ID = new AtomicInteger();

    private static final CameraMatrices BACKUP_CAMERA_MATRICES = new CameraMatrices();
    private static final Matrix4f BACKUP_PROJECTION = new Matrix4f();
    private static final Vector3f BACKUP_LIGHT0_POSITION = new Vector3f();
    private static final Vector3f BACKUP_LIGHT1_POSITION = new Vector3f();

    private static final Matrix4f BACKUP_FLASHBACK_PROJECTION = new Matrix4f();
    private static final Quaternionf BACKUP_FLASHBACK_CAMERA = new Quaternionf();

    private static boolean renderingPerspective = false;

    private VeilLevelPerspectiveRenderer() {
    }

    /**
     * Renders the level from another POV. Automatically prevents circular render references.
     *
     * @param framebuffer       The framebuffer to draw into
     * @param modelView         The base modelview matrix
     * @param projection        The projection matrix
     * @param cameraPosition    The position of the camera
     * @param cameraOrientation The orientation of the camera
     * @param renderDistance    The chunk render distance
     * @param deltaTracker      The delta tracker instance
     * @param drawLights        Whether to draw lights to the scene after
     * @return The full framebuffer including dynamic buffers. This framebuffer is owned by the render system
     */
    public static AdvancedFbo render(AdvancedFbo framebuffer, Matrix4fc modelView, Matrix4fc projection, Vector3dc cameraPosition, Quaternionfc cameraOrientation, float renderDistance, class_9779 deltaTracker, boolean drawLights) {
        return render(framebuffer, class_310.method_1551().field_1719, modelView, projection, cameraPosition, cameraOrientation, renderDistance, deltaTracker, drawLights);
    }

    /**
     * Renders the level from another POV. Automatically prevents circular render references.
     *
     * @param framebuffer       The framebuffer to draw into
     * @param cameraEntity      The entity to draw the camera in relation to. If unsure use {@link #render(AdvancedFbo, Matrix4fc, Matrix4fc, Vector3dc, Quaternionfc, float, class_9779, boolean)}
     * @param modelView         The base modelview matrix
     * @param projection        The projection matrix
     * @param cameraPosition    The position of the camera
     * @param cameraOrientation The orientation of the camera
     * @param renderDistance    The chunk render distance
     * @param deltaTracker      The delta tracker instance
     * @param drawLights        Whether to draw lights to the scene after
     * @return The full framebuffer including dynamic buffers. This framebuffer is owned by the render system
     */
    public static AdvancedFbo render(AdvancedFbo framebuffer, @Nullable class_1297 cameraEntity, Matrix4fc modelView, Matrix4fc projection, Vector3dc cameraPosition, Quaternionfc cameraOrientation, float renderDistance, class_9779 deltaTracker, boolean drawLights) {
        if (renderingPerspective) {
            return framebuffer;
        }

        // Finish anything previously being rendered for safety
        class_4597.class_4598 bufferSource = class_310.method_1551().method_22940().method_23000();
        bufferSource.method_22993();

        final class_310 minecraft = class_310.method_1551();
        final class_757 gameRenderer = minecraft.field_1773;
        final class_761 levelRenderer = minecraft.field_1769;
        final LevelRendererAccessor levelRendererAccessor = (LevelRendererAccessor) levelRenderer;
        final class_1041 window = minecraft.method_22683();
        final GameRendererAccessor accessor = (GameRendererAccessor) gameRenderer;
        final RenderTargetExtension renderTargetExtension = (RenderTargetExtension) minecraft.method_1522();
        final class_4587 poseStack = new class_4587();

        CAMERA.setup(cameraPosition, cameraEntity, minecraft.field_1687, cameraOrientation, renderDistance);

        poseStack.method_34425(TRANSFORM.set(modelView));
        poseStack.method_22907(CAMERA.method_23767());

        float backupRenderDistance = gameRenderer.method_3193();
        accessor.setRenderDistance(renderDistance * 16.0F);

        float backupFogStart = RenderSystem.getShaderFogStart();
        float backupFogEnd = RenderSystem.getShaderFogEnd();
        class_6854 backupFogShape = RenderSystem.getShaderFogShape();

        int backupWidth = window.method_4489();
        int backupHeight = window.method_4506();
        if (!FlashbackCompat.isLoaded()) {
            window.method_35642(framebuffer.getWidth());
            window.method_35643(framebuffer.getHeight());
        } else {
            FlashbackAccess.backup(BACKUP_FLASHBACK_PROJECTION, BACKUP_FLASHBACK_CAMERA);
        }

        final Object backupPipeline = IrisPipelineAccess.getPipeline(levelRenderer);

        final Object backupRenderLists;
        final Object backupTaskLists;
        if (SodiumCompat.isLoaded()) {
            backupRenderLists = SodiumCompat.INSTANCE.getSortedRenderLists();
            backupTaskLists = SodiumCompat.INSTANCE.getTaskLists();
            ID.getAndIncrement();
        } else {
            backupRenderLists = null;
            backupTaskLists = null;
        }

        BACKUP_PROJECTION.set(RenderSystem.getProjectionMatrix());
        gameRenderer.method_22709(TRANSFORM.set(projection));
        BACKUP_LIGHT0_POSITION.set(VeilRenderSystem.getLight0Direction());
        BACKUP_LIGHT1_POSITION.set(VeilRenderSystem.getLight1Direction());

        Matrix4fStack matrix4fstack = RenderSystem.getModelViewStack();
        matrix4fstack.pushMatrix();
        matrix4fstack.identity();
        RenderSystem.applyModelViewMatrix();

        class_239 backupHitResult = minecraft.field_1765;
        class_1297 backupCrosshairPickEntity = minecraft.field_1692;

        renderingPerspective = true;
        AdvancedFbo drawFbo = VeilRenderSystem.renderer().getDynamicBufferManger().getDynamicFbo(framebuffer);
        drawFbo.bind(true);
        renderTargetExtension.veil$setWrapper(drawFbo);

        class_4604 backupFrustum = levelRendererAccessor.getCullingFrustum();

        CameraMatrices matrices = VeilRenderSystem.renderer().getCameraMatrices();
        matrices.backup(BACKUP_CAMERA_MATRICES);

        try {
            levelRenderer.method_32133(new class_243(cameraPosition.x(), cameraPosition.y(), cameraPosition.z()), poseStack.method_23760().method_23761(), TRANSFORM);
            levelRenderer.method_22710(deltaTracker, false, CAMERA, gameRenderer, gameRenderer.method_22974(), poseStack.method_23760().method_23761(), TRANSFORM);
            // Make sure all buffers have been finished
            bufferSource.method_22993();
            levelRenderer.method_3254();

            // Draw lights
            if (drawLights) {
                class_3695 profiler = class_310.method_1551().method_16011();
                if (VeilRenderSystem.drawLights(profiler, VeilRenderSystem.getCullingFrustum())) {
                    VeilRenderSystem.compositeLights(profiler);
                } else {
                    AdvancedFbo.unbind();
                }
            }
        } finally {
            matrices.restore(BACKUP_CAMERA_MATRICES);

            levelRendererAccessor.setCullingFrustum(backupFrustum);

            renderTargetExtension.veil$setWrapper(null);
            AdvancedFbo.unbind();
            renderingPerspective = false;

            minecraft.field_1692 = backupCrosshairPickEntity;
            minecraft.field_1765 = backupHitResult;

            matrix4fstack.popMatrix();
            RenderSystem.applyModelViewMatrix();

            RenderSystem.setShaderLights(BACKUP_LIGHT0_POSITION, BACKUP_LIGHT1_POSITION);
            gameRenderer.method_22709(BACKUP_PROJECTION);

            IrisPipelineAccess.setPipeline(levelRenderer, backupPipeline);

            if (SodiumCompat.isLoaded()) {
                SodiumCompat.INSTANCE.setSortedRenderLists(backupRenderLists);
                SodiumCompat.INSTANCE.setTaskList(backupTaskLists);
            }

            RenderSystem.setShaderFogStart(backupFogStart);
            RenderSystem.setShaderFogEnd(backupFogEnd);
            RenderSystem.setShaderFogShape(backupFogShape);

            if (!FlashbackCompat.isLoaded()) {
                window.method_35642(backupWidth);
                window.method_35643(backupHeight);
            } else {
                FlashbackAccess.restore(BACKUP_FLASHBACK_PROJECTION, BACKUP_FLASHBACK_CAMERA);
            }

            accessor.setRenderDistance(backupRenderDistance);

            // Reset the renderers to what they used to be
            class_4184 mainCamera = gameRenderer.method_19418();
            minecraft.method_31975().method_3549(minecraft.field_1687, mainCamera, minecraft.field_1765);
            minecraft.method_1561().method_3941(minecraft.field_1687, mainCamera, minecraft.field_1692);
        }
        return drawFbo;
    }

    /**
     * @return Whether a perspective is being rendered
     */
    public static boolean isRenderingPerspective() {
        return renderingPerspective;
    }

    @ApiStatus.Internal
    public static int getID() {
        return ID.get();
    }
}
