package foundry.veil.api.client.render.shader;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSyntaxException;
import foundry.veil.Veil;
import foundry.veil.VeilClient;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.api.client.render.dynamicbuffer.DynamicBufferType;
import foundry.veil.api.client.render.shader.compiler.ShaderCompiler;
import foundry.veil.api.client.render.shader.compiler.ShaderException;
import foundry.veil.api.client.render.shader.compiler.VeilShaderSource;
import foundry.veil.api.client.render.shader.processor.*;
import foundry.veil.api.client.render.shader.program.ProgramDefinition;
import foundry.veil.api.client.render.shader.program.ShaderProgram;
import foundry.veil.impl.ThreadTaskScheduler;
import foundry.veil.impl.client.render.dynamicbuffer.DynamicBufferManager;
import foundry.veil.impl.client.render.dynamicbuffer.DynamicBufferProcessor;
import foundry.veil.impl.client.render.shader.ShaderImporterImpl;
import foundry.veil.impl.client.render.shader.processor.ShaderProcessorList;
import foundry.veil.impl.client.render.shader.program.DynamicShaderProgramImpl;
import foundry.veil.impl.client.render.shader.program.ShaderProgramImpl;
import io.github.ocelot.glslprocessor.api.GlslParser;
import io.github.ocelot.glslprocessor.api.GlslSyntaxException;
import io.github.ocelot.glslprocessor.api.node.GlslTree;
import io.github.ocelot.glslprocessor.lib.anarres.cpp.LexerException;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectArrayMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMaps;
import it.unimi.dsi.fastutil.objects.Object2IntArrayMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import org.apache.commons.io.IOUtils;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Async;
import org.jetbrains.annotations.Nullable;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Reader;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3302;
import net.minecraft.class_3518;
import net.minecraft.class_3695;
import net.minecraft.class_5912;
import net.minecraft.class_7654;

import static org.lwjgl.opengl.GL20C.GL_FRAGMENT_SHADER;
import static org.lwjgl.opengl.GL20C.GL_VERTEX_SHADER;
import static org.lwjgl.opengl.GL32C.GL_GEOMETRY_SHADER;
import static org.lwjgl.opengl.GL40C.GL_TESS_CONTROL_SHADER;
import static org.lwjgl.opengl.GL40C.GL_TESS_EVALUATION_SHADER;
import static org.lwjgl.opengl.GL43C.GL_COMPUTE_SHADER;

/**
 * Manages all shaders and compiles them automatically.
 *
 * @author Ocelot
 * @see ShaderCompiler
 */
public class ShaderManager implements class_3302, Closeable {

    private static final Gson GSON = new GsonBuilder()
            .registerTypeAdapter(class_2960.class, new class_2960.class_2961())
            .registerTypeAdapter(ProgramDefinition.class, new ProgramDefinition.Deserializer())
            .create();

    public static final class_7654 INCLUDE_LISTER = new class_7654("pinwheel/shaders/include", ".glsl");
    public static final ShaderSourceSet PROGRAM_SET = new ShaderSourceSet("pinwheel/shaders/program");

    private static final Map<Integer, String> TYPES = Map.of(
            GL_VERTEX_SHADER, "vertex",
            GL_TESS_CONTROL_SHADER, "tesselation_control",
            GL_TESS_EVALUATION_SHADER, "tesselation_evaluation",
            GL_GEOMETRY_SHADER, "geometry",
            GL_FRAGMENT_SHADER, "fragment",
            GL_COMPUTE_SHADER, "compute"
    );

    private final DynamicBufferManager dynamicBufferManager;
    private final ShaderSourceSet sourceSet;
    private final ShaderPreDefinitions definitions;
    private final Map<class_2960, ShaderProgramImpl> shaders;
    private final Map<class_2960, ShaderProgram> shadersView;
    private final Set<class_2960> dirtyShaders;
    private final long supportedFeatures;

    private CompletableFuture<Void> recompileFuture;
    private CompletableFuture<Void> updateBuffersFuture;

    /**
     * Creates a new shader manager.
     *
     * @param sourceSet            The source set to load all shaders from
     * @param shaderPreDefinitions The set of shader pre-definitions
     * @param dynamicBufferManager The manager for dynamic buffers
     */
    public ShaderManager(ShaderSourceSet sourceSet, ShaderPreDefinitions shaderPreDefinitions, DynamicBufferManager dynamicBufferManager) {
        this.dynamicBufferManager = dynamicBufferManager;
        this.sourceSet = sourceSet;
        this.definitions = shaderPreDefinitions;
        this.definitions.addListener(this::onDefinitionChanged);
        this.shaders = new HashMap<>();
        this.shadersView = Collections.unmodifiableMap(this.shaders);
        this.dirtyShaders = new HashSet<>();

        long supportedFeatures = 0;
        for (ShaderFeature feature : ShaderFeature.FEATURES) {
            if (feature.isSupported()) {
                supportedFeatures |= 1 << feature.ordinal();
            }
        }
        this.supportedFeatures = supportedFeatures;

        this.recompileFuture = CompletableFuture.completedFuture(null);
        this.updateBuffersFuture = CompletableFuture.completedFuture(null);
    }

    private void onDefinitionChanged(String definition) {
        this.shaders.values().forEach(shader -> {
            if (shader.getDefinitionDependencies().contains(definition)) {
                Veil.LOGGER.debug("{} changed, recompiling {}", definition, shader.getName());
                this.scheduleRecompile(shader.getName());
            }
        });
    }

    private void addProcessors(ShaderProcessorList processorList, class_5912 provider) {
        processorList.addPreprocessor(new ShaderImportProcessor());
        processorList.addPreprocessor(new ShaderBufferProcessor());
        processorList.addPreprocessor(new ShaderBindingProcessor());
        processorList.addPreprocessor(new ShaderVersionProcessor(), false);
        processorList.addPreprocessor(new ShaderModifyProcessor(), false);
        processorList.addPreprocessor(new DynamicBufferProcessor(), false);
        processorList.addPreprocessor(new ShaderFeatureProcessor(), false);
        VeilClient.clientPlatform().onRegisterShaderPreProcessors(provider, processorList);
    }

    private ProgramDefinition parseDefinition(class_2960 id, class_5912 provider) throws IOException {
        try (Reader reader = provider.openAsReader(this.sourceSet.getShaderDefinitionLister().method_45112(id))) {
            ProgramDefinition definition = class_3518.method_15276(GSON, reader, ProgramDefinition.class);
            if (definition.vertex() == null &&
                    definition.tesselationControl() == null &&
                    definition.tesselationEvaluation() == null &&
                    definition.geometry() == null &&
                    definition.fragment() == null &&
                    definition.compute() == null) {
                throw new JsonSyntaxException("Shader programs must define at least 1 shader type");
            }

            return definition;
        } catch (JsonParseException e) {
            throw new IOException(e);
        }
    }

    private void readShader(ShaderProcessorList processorList, class_5912 resourceProvider, Map<class_2960, ProgramDefinition> definitions, Map<class_2960, VeilShaderSource> shaderSources, class_2960 definitionId, int activeBuffers) {
        Set<class_2960> checkedSources = new HashSet<>();

        ShaderPreProcessor processor = processorList.getProcessor();
        ShaderPreProcessor importProcessor = processorList.getImportProcessor();

        try {
            ProgramDefinition definition = this.parseDefinition(definitionId, resourceProvider);
            if (definitions.put(definitionId, definition) != null) {
                throw new IllegalStateException("Duplicate shader ignored with ID " + definitionId);
            }

            Map<String, Object> customProgramData = new HashMap<>();
            for (Int2ObjectMap.Entry<class_2960> shader : definition.shaders().int2ObjectEntrySet()) {
                int type = shader.getIntKey();
                class_2960 shaderId = shader.getValue();

                class_7654 typeConverter = this.sourceSet.getTypeConverter(type);
                class_2960 location = typeConverter.method_45112(shaderId);

                if (!checkedSources.add(location)) {
                    continue;
                }

                class_3298 resource = resourceProvider.getResourceOrThrow(location);
                try (Reader reader = resource.method_43039()) {
                    String source = IOUtils.toString(reader);

                    processor.prepare();
                    importProcessor.prepare();

                    Set<String> dependencies = new HashSet<>();
                    Map<String, String> macros = definition.getMacros(dependencies, this.definitions);
                    DynamicBufferType.addMacros(activeBuffers, macros);
                    VeilRenderSystem.renderer().getShaderManager().addMacros(macros);
                    GlslTree tree = GlslParser.preprocessParse(source, macros);

                    Object2IntMap<String> uniformBindings = new Object2IntArrayMap<>();
                    PreProcessorContext preProcessorContext = new PreProcessorContext(
                            customProgramData,
                            importProcessor,
                            definition,
                            this.definitions,
                            processorList.getShaderImporter(),
                            activeBuffers,
                            type,
                            uniformBindings,
                            dependencies,
                            macros,
                            shaderId,
                            true);
                    processor.modify(preProcessorContext, tree);
                    GlslTree.stripGLMacros(macros);
                    Map<String, String> treeMacros = tree.getMacros();
                    treeMacros.putAll(macros);

                    for (String dependency : dependencies) {
                        String value = this.definitions.getDefinition(dependency);
                        if (value != null) {
                            treeMacros.putIfAbsent(dependency, value);
                        }
                    }

                    shaderSources.put(location, new VeilShaderSource(shaderId, tree.toSourceString(), uniformBindings, dependencies, new HashSet<>(processorList.getShaderImporter().addedImports())));
                } catch (Throwable t) {
                    throw new IOException("Failed to load " + getTypeName(type) + " shader", t);
                }
            }
        } catch (IOException | IllegalArgumentException | JsonParseException e) {
            Veil.LOGGER.error("Couldn't parse shader {} from {}", definitionId, this.sourceSet.getShaderDefinitionLister().method_45112(definitionId), e);
        }
    }

    private boolean isInvalid(class_2960 id, ProgramDefinition definition) {
        if (!this.hasFeatures(definition.requiredFeatures())) {
            List<String> requiredFeatures = new ArrayList<>();
            for (ShaderFeature feature : definition.requiredFeatures()) {
                if (!this.hasFeatures(feature)) {
                    requiredFeatures.add(feature.name().toLowerCase(Locale.ROOT));
                }
            }
            Veil.LOGGER.info("Skipping shader '{}' (missing required features: {})", id, String.join(", ", requiredFeatures));
            return true;
        }

        return false;
    }

    private void compile(ShaderProgramImpl program, @Nullable ProgramDefinition definition, ShaderCompiler compiler) {
        class_2960 id = program.getName();
        try {
            program.compile(this.dynamicBufferManager.getActiveBuffers(), this.sourceSet, definition, compiler);
        } catch (ShaderException e) {
            Veil.LOGGER.error("Failed to create shader {}: {}", id, e.getMessage());
            String error = e.getGlError();
            if (error != null) {
                Veil.LOGGER.warn(error);
            }
        } catch (Exception e) {
            Veil.LOGGER.error("Failed to create shader: {}", id, e);
        }
    }

    /**
     * Creates a new dynamic shader with the specified shader sources.
     *
     * @param id            The internal ID of the shader
     * @param shaderSources A map of all shader sources from GL shader enum values to GLSL source code
     * @return A future for when the shader is done compiling
     */
    public CompletableFuture<ShaderProgram> createDynamicProgram(class_2960 id, Int2ObjectMap<String> shaderSources) {
        DynamicShaderProgramImpl compileProgram;
        ShaderProgramImpl program = this.shaders.get(id);
        if (!(program instanceof DynamicShaderProgramImpl dynamicShaderProgram)) {
            if (program != null) {
                Veil.LOGGER.warn("Dynamic shader {} will overwrite the shader file until it is deleted!", id);
            }

            compileProgram = new DynamicShaderProgramImpl(id, () -> {
                if (program != null) {
                    this.shaders.put(id, program);
                } else {
                    this.shaders.remove(id);
                }
            });
            compileProgram.setOldShader(program);
            this.shaders.put(id, compileProgram);
        } else {
            compileProgram = dynamicShaderProgram;
        }

        compileProgram.setShaderSources(shaderSources);
        int activeBuffers = VeilRenderSystem.renderer().getDynamicBufferManger().getActiveBuffers();
        return CompletableFuture.runAsync(() -> {
            class_3300 resourceManager = class_310.method_1551().method_1478();
            ShaderProcessorList list = new ShaderProcessorList(resourceManager);
            this.addProcessors(list, resourceManager);
            compileProgram.processShaderSources(list, this.definitions, activeBuffers);
        }, class_156.method_18349()).thenApplyAsync(unused -> {
            try (ShaderCompiler compiler = ShaderCompiler.direct(null)) {
                this.compile(compileProgram, null, compiler);
            }
            return compileProgram;
        }, VeilRenderSystem.renderThreadExecutor());
    }

    /**
     * Sets a global shader value.
     *
     * @param setter The setter for shaders
     */
    public void setGlobal(Consumer<ShaderProgram> setter) {
        this.shaders.values().forEach(setter);
    }

    /**
     * Checks if the requested shader features are available.
     *
     * @param features The features to check for
     * @return Whether those features are supported
     * @since 2.0.0
     */
    public boolean hasFeatures(ShaderFeature... features) {
        int mask = 0;
        for (ShaderFeature feature : features) {
            mask |= 1 << feature.ordinal();
        }
        return (this.supportedFeatures & mask) == mask;
    }

    /**
     * Retrieves a shader by the specified id.
     *
     * @param id The id of the shader to retrieve
     * @return The retrieved shader or <code>null</code> if there is no valid shader with that id
     */
    public @Nullable ShaderProgram getShader(class_2960 id) {
        return this.shaders.get(id);
    }

    /**
     * @return All shader programs registered
     */
    public Map<class_2960, ShaderProgram> getShaders() {
        return this.shadersView;
    }

    /**
     * @return The source set all shaders are loaded from
     */
    public ShaderSourceSet getSourceSet() {
        return this.sourceSet;
    }

    private CompletableFuture<ReloadState> prepare(class_3300 resourceManager, Collection<DynamicShaderProgramImpl> dynamicShaders, Collection<class_2960> shaders, int activeBuffers, Executor executor) {
        Map<class_2960, ProgramDefinition> definitions = new ConcurrentHashMap<>();
        Map<class_2960, VeilShaderSource> shaderSources = new ConcurrentHashMap<>();

        Long2ObjectMap<ShaderProcessorList> processorList = Long2ObjectMaps.synchronize(new Long2ObjectArrayMap<>());
        Deque<class_2960> shaderQueue = new ConcurrentLinkedDeque<>(shaders);
        ThreadTaskScheduler scheduler = new ThreadTaskScheduler("VeilShaderCompiler", Math.max(1, Runtime.getRuntime().availableProcessors() / 4), () -> {
            class_2960 key = shaderQueue.poll();
            if (key == null) {
                return null;
            }

            return () -> {
                ShaderProcessorList shaderProcessor = processorList.computeIfAbsent(Thread.currentThread().threadId(), id -> {
                    ShaderProcessorList list = new ShaderProcessorList(resourceManager);
                    this.addProcessors(list, resourceManager);
                    return list;
                });
                shaderProcessor.getShaderImporter().reset();
                this.readShader(shaderProcessor, resourceManager, definitions, shaderSources, key, activeBuffers);
            };
        });

        return CompletableFuture.allOf(scheduler.getCompletedFuture(),
                CompletableFuture.runAsync(() -> {
                    ShaderProcessorList list = new ShaderProcessorList(resourceManager);
                    this.addProcessors(list, resourceManager);
                    ShaderImporterImpl shaderImporter = list.getShaderImporter();
                    for (DynamicShaderProgramImpl shader : dynamicShaders) {
                        shaderImporter.reset();
                        shader.processShaderSources(list, this.definitions, activeBuffers);
                    }
                }, executor)).thenApply(unused -> new ReloadState(definitions, shaderSources));
    }

    private void apply(ShaderManager.ReloadState reloadState) {
        Iterator<ShaderProgramImpl> iterator = this.shaders.values().iterator();
        while (iterator.hasNext()) {
            ShaderProgramImpl program = iterator.next();
            if (program instanceof DynamicShaderProgramImpl dynamicShaderProgram) {
                ShaderProgramImpl old = dynamicShaderProgram.getOldShader();
                if (old != null) {
                    old.free();
                }
            } else {
                program.free();
                iterator.remove();
            }
        }

        try (ShaderCompiler compiler = reloadState.createCompiler()) {
            for (Map.Entry<class_2960, ProgramDefinition> entry : reloadState.definitions().entrySet()) {
                ProgramDefinition definition = entry.getValue();
                if (this.isInvalid(entry.getKey(), definition)) {
                    continue;
                }

                class_2960 id = entry.getKey();
                ShaderProgramImpl program = new ShaderProgramImpl(id);
                this.compile(program, definition, compiler);
                DynamicShaderProgramImpl old = (DynamicShaderProgramImpl) this.shaders.put(id, program);
                if (old != null) {
                    old.setOldShader(program);
                }
            }
        }

        VeilClient.clientPlatform().onVeilCompileShaders(this, Collections.unmodifiableMap(this.shaders));

        Veil.LOGGER.info("Loaded {} shaders from: {}", this.shaders.size(), this.sourceSet.getFolder());
    }

    private void applyRecompile(ShaderManager.ReloadState reloadState, Map<class_2960, ShaderProgram> updatedShaders) {
        try (ShaderCompiler compiler = reloadState.createCompiler()) {
            for (Map.Entry<class_2960, ProgramDefinition> entry : reloadState.definitions().entrySet()) {
                class_2960 id = entry.getKey();
                ShaderProgramImpl program = this.shaders.get(id);
                if (program == null) {
                    Veil.LOGGER.warn("Failed to recompile unknown shader: {}", id);
                    continue;
                }

                ProgramDefinition definition = entry.getValue();
                if (this.isInvalid(id, definition)) {
                    continue;
                }

                this.compile(program, definition, compiler);
            }
        }

        VeilClient.clientPlatform().onVeilCompileShaders(this, updatedShaders);

        Veil.LOGGER.info("Recompiled {} shaders from: {}", updatedShaders.size(), this.sourceSet.getFolder());
    }

    private void scheduleRecompile(int attempt) {
        class_310 client = class_310.method_1551();
        client.method_18858(() -> {
            if (!this.recompileFuture.isDone()) {
                return;
            }

            Set<class_2960> shaders;
            synchronized (this.dirtyShaders) {
                shaders = new HashSet<>(this.dirtyShaders);
                this.dirtyShaders.clear();
            }

            Map<class_2960, ShaderProgram> updatedShaders = new HashMap<>(shaders.size());
            for (class_2960 id : shaders) {
                ShaderProgram shader = this.getShader(id);
                if (shader != null) {
                    updatedShaders.put(id, shader);
                }
            }

            Set<DynamicShaderProgramImpl> dynamicShaderPrograms = new HashSet<>();
            Iterator<class_2960> iterator = shaders.iterator();
            while (iterator.hasNext()) {
                class_2960 shader = iterator.next();
                ShaderProgramImpl program = this.shaders.get(shader);

                if (program instanceof DynamicShaderProgramImpl dynamicShaderProgram) {
                    iterator.remove();
                    dynamicShaderPrograms.add(dynamicShaderProgram);
                }
            }

            int activeBuffers = this.dynamicBufferManager.getActiveBuffers();
            this.recompileFuture = this.prepare(client.method_1478(), dynamicShaderPrograms, shaders, activeBuffers, class_156.method_18349())
                    .thenAcceptAsync(state -> this.applyRecompile(state, Collections.unmodifiableMap(updatedShaders)), client)
                    .handle((value, e) -> {
                        if (e != null) {
                            Veil.LOGGER.error("Error recompiling shaders", e);
                        }

                        synchronized (this.dirtyShaders) {
                            if (this.dirtyShaders.isEmpty()) {
                                return value;
                            }
                        }

                        if (attempt >= 3) {
                            Veil.LOGGER.error("Failed to recompile shaders after {} attempts", attempt);
                            return value;
                        }

                        this.scheduleRecompile(attempt + 1);
                        return value;
                    });
        });
    }

    /**
     * Schedules a shader recompilation on the next loop iteration.
     *
     * @param shader The shader to recompile
     */
    @Async.Schedule
    public void scheduleRecompile(class_2960 shader) {
        synchronized (this.dirtyShaders) {
            this.dirtyShaders.add(shader);
        }

        if (!this.recompileFuture.isDone()) {
            return;
        }

        this.scheduleRecompile(0);
    }

    @ApiStatus.Internal
    public void setActiveBuffers(int activeBuffers) {
        ShaderProgram active = null;

        try {
            Set<DynamicShaderProgramImpl> dynamicShaders = new HashSet<>();
            Set<class_2960> shaders = new HashSet<>(this.shaders.size());
            Map<class_2960, ShaderProgram> updatedShaders = new HashMap<>();
            for (ShaderProgram program : this.shaders.values()) {
                active = program;
                if (program instanceof ShaderProgramImpl impl) {
                    if (impl.setActiveBuffers(activeBuffers)) {
                        if (program instanceof DynamicShaderProgramImpl dynamicShaderProgram) {
                            dynamicShaders.add(dynamicShaderProgram);
                        } else {
                            shaders.add(program.getName());
                        }
                        updatedShaders.put(program.getName(), program);
                    }
                }
            }

            if (!shaders.isEmpty()) {
                this.updateBuffersFuture = this.updateBuffersFuture
                        .thenCompose(unused -> this.prepare(class_310.method_1551().method_1478(), dynamicShaders, shaders, activeBuffers, class_156.method_18349()))
                        .thenAcceptAsync(reloadState -> {
                            try (ShaderCompiler compiler = reloadState.createCompiler()) {
                                for (Map.Entry<class_2960, ProgramDefinition> entry : reloadState.definitions().entrySet()) {
                                    class_2960 id = entry.getKey();
                                    ShaderProgram program = this.getShader(id);
                                    if (!(program instanceof ShaderProgramImpl impl)) {
                                        Veil.LOGGER.warn("Failed to set shader active buffers: {}", id);
                                        continue;
                                    }

                                    try {
                                        impl.recompile(activeBuffers, this.sourceSet, compiler);
                                    } catch (ShaderException e) {
                                        Veil.LOGGER.error("Failed to update shader active buffers: {}. {}", id, e.getMessage());
                                        String error = e.getGlError();
                                        if (error != null) {
                                            Veil.LOGGER.warn(error);
                                        }
                                    } catch (Exception e) {
                                        Veil.LOGGER.error("Failed to update shader active buffers: {}", id, e);
                                    }
                                }
                            }

                            VeilClient.clientPlatform().onVeilCompileShaders(this, Collections.unmodifiableMap(updatedShaders));

                            Veil.LOGGER.info("Compiled {} shaders from: {}", updatedShaders.size(), this.sourceSet.getFolder());
                        }, class_310.method_1551());
            }
        } catch (ShaderException e) {
            Veil.LOGGER.error("Failed to set shader active buffers {}: {}", active.getName(), e.getMessage());
            String error = e.getGlError();
            if (error != null) {
                Veil.LOGGER.warn(error);
            }
        }
    }

    @Override
    public CompletableFuture<Void> method_25931(class_4045 preparationBarrier, class_3300 resourceManager, class_3695 preparationsProfiler, class_3695 reloadProfiler, Executor backgroundExecutor, Executor gameExecutor) {
        Set<class_2960> dynamicShaders = new HashSet<>();
        for (ShaderProgramImpl program : this.shaders.values()) {
            if (program instanceof DynamicShaderProgramImpl) {
                dynamicShaders.add(program.getName());
            }
        }
        int activeBuffers = this.dynamicBufferManager.getActiveBuffers();
        return CompletableFuture.allOf(this.recompileFuture, this.updateBuffersFuture).thenComposeAsync(
                unused -> {
                    class_7654 lister = this.sourceSet.getShaderDefinitionLister();
                    Set<class_2960> shaderIds = lister.method_45113(resourceManager).keySet()
                            .stream()
                            .map(lister::method_45115)
                            .filter(id -> !dynamicShaders.contains(id))
                            .collect(Collectors.toSet());
                    return this.prepare(resourceManager, Collections.emptySet(), shaderIds, activeBuffers, backgroundExecutor)
                            .thenCompose(preparationBarrier::method_18352)
                            .thenAcceptAsync(this::apply, gameExecutor);
                }, backgroundExecutor);
    }

    @Override
    public String method_22322() {
        return this.getClass().getSimpleName() + " " + this.getSourceSet().getFolder();
    }

    /**
     * @return The current future for dirty shader recompilation status
     */
    public CompletableFuture<Void> getRecompileFuture() {
        return this.recompileFuture;
    }

    /**
     * @return The current future for updating dynamic buffers
     */
    public CompletableFuture<Void> getUpdateBuffersFuture() {
        return this.updateBuffersFuture;
    }

    /**
     * Retrieves a readable name for a shader type. Supports all shader types instead of just vertex and fragment.
     *
     * @param type The GL enum for the type
     * @return The readable name or a hex value if the type is unknown
     */
    public static String getTypeName(int type) {
        String value = TYPES.get(type);
        return value != null ? value : "0x" + Integer.toHexString(type);
    }

    @Override
    public void close() {
        this.shaders.values().forEach(ShaderProgramImpl::freeInternal);
        this.shaders.clear();
    }

    @ApiStatus.Internal
    public void addMacros(Map<String, String> macros) {
        long mask = this.supportedFeatures;
        while (mask != 0) {
            int ordinal = Long.numberOfTrailingZeros(mask);
            macros.put(ShaderFeature.FEATURES[ordinal].getDefinitionName(), "1");
            mask &= ~(1L << ordinal);
        }
    }

    private record PreProcessorContext(Map<String, Object> customProgramData,
                                       ShaderPreProcessor preProcessor,
                                       @Nullable ProgramDefinition definition,
                                       ShaderPreDefinitions preDefinitions,
                                       ShaderImporter shaderImporter,
                                       int activeBuffers,
                                       int type,
                                       Object2IntMap<String> uniformBindings,
                                       Set<String> definitionDependencies,
                                       Map<String, String> macros,
                                       @Nullable class_2960 name,
                                       boolean sourceFile) implements ShaderPreProcessor.VeilContext {

        @Override
        public GlslTree modifyInclude(@Nullable class_2960 name, String source) throws IOException, GlslSyntaxException, LexerException {
            GlslTree tree = GlslParser.preprocessParse(source, this.macros);
            PreProcessorContext context = new PreProcessorContext(this.customProgramData, this.preProcessor, this.definition, this.preDefinitions, this.shaderImporter, this.activeBuffers, this.type, this.uniformBindings, this.definitionDependencies, this.macros, name, false);
            this.preProcessor.modify(context, tree);
            return tree;
        }

        @Override
        public void addUniformBinding(String name, int binding) {
            this.uniformBindings.put(name, binding);
        }

        @Override
        public void addDefinitionDependency(String name) {
            this.definitionDependencies.add(name);
        }

        @Override
        public boolean isDynamic() {
            return false;
        }

        @Override
        public boolean isSourceFile() {
            return this.sourceFile;
        }
    }

    private record ReloadState(Map<class_2960, ProgramDefinition> definitions,
                               Map<class_2960, VeilShaderSource> shaderSources) {

        public ShaderCompiler createCompiler() {
            return ShaderCompiler.direct(name -> {
                VeilShaderSource source = this.shaderSources.get(name);
                if (source == null) {
                    throw new FileNotFoundException("Unknown shader source: " + name);
                }
                return source;
            });
        }
    }
}
