package foundry.veil.api.client.render.shader.program;

import foundry.veil.api.client.render.framebuffer.AdvancedFbo;
import foundry.veil.api.client.render.framebuffer.AdvancedFboTextureAttachment;
import foundry.veil.api.client.render.shader.texture.ShaderTextureSource;
import foundry.veil.impl.client.render.shader.program.ShaderProgramImpl;
import net.minecraft.class_1044;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_5944;
import org.jetbrains.annotations.Nullable;

/**
 * Provides write access to all textures in a shader program.
 *
 * @author Ocelot
 */
public interface TextureUniformAccess {

    /**
     * Sets <code>DiffuseSampler0</code>-<code>DiffuseSamplerMax</code> to the color buffers in the specified framebuffer.
     * <br>
     * Also sets <code>DiffuseDepthSampler</code> if the framebuffer has a depth attachment.
     *
     * @param framebuffer The framebuffer to bind samplers from
     */
    default void setFramebufferSamplers(AdvancedFbo framebuffer) {
        boolean setDiffuseSampler = false;
        for (int i = 0; i < framebuffer.getColorAttachments(); i++) {
            if (!framebuffer.isColorTextureAttachment(i)) {
                continue;
            }

            AdvancedFboTextureAttachment attachment = framebuffer.getColorTextureAttachment(i);
            this.setSampler("DiffuseSampler" + i, attachment.method_4624());
            if (attachment.getName() != null) {
                this.setSampler(attachment.getName(), attachment.method_4624());
            }
            if (!setDiffuseSampler) {
                this.setSampler("DiffuseSampler", attachment.method_4624());
                setDiffuseSampler = true;
            }
        }

        if (framebuffer.isDepthTextureAttachment()) {
            AdvancedFboTextureAttachment attachment = framebuffer.getDepthTextureAttachment();
            this.setSampler("DiffuseDepthSampler", attachment.method_4624());
            if (attachment.getName() != null) {
                this.setSampler(attachment.getName(), attachment.method_4624());
            }
        }
    }

    /**
     * Adds a texture that is dynamically bound and sets texture units.
     *
     * @param name     The name of the texture to set
     * @param location The name of the texture in the texture manager to bind and assign a texture unit
     * @since 2.5.0
     */
    default void setSampler(CharSequence name, class_2960 location) {
        class_1044 abstractTexture = class_310.method_1551().method_1531().method_4619(location);
        this.setSampler(name, abstractTexture.method_4624(), 0);
    }

    /**
     * Adds a texture that is dynamically bound and sets texture units.
     *
     * @param name      The name of the texture to set
     * @param location  The name of the texture in the texture manager to bind and assign a texture unit
     * @param samplerId The id of the sampler assign a texture unit
     * @since 2.5.0
     */
    default void setSampler(CharSequence name, class_2960 location, int samplerId) {
        class_1044 abstractTexture = class_310.method_1551().method_1531().method_4619(location);
        this.setSampler(name, abstractTexture.method_4624(), samplerId);
    }

    /**
     * Adds a texture that is dynamically bound and sets texture units.
     *
     * @param name      The name of the texture to set
     * @param textureId The id of the texture to bind and assign a texture unit
     */
    default void setSampler(CharSequence name, int textureId) {
        this.setSampler(name, textureId, 0);
    }

    /**
     * Adds a texture that is dynamically bound and sets texture units.
     *
     * @param name      The name of the texture to set
     * @param textureId The id of the texture to bind and assign a texture unit
     * @param samplerId The id of the sampler assign a texture unit
     */
    void setSampler(CharSequence name, int textureId, int samplerId);

    /**
     * Removes the specified sampler binding. This will effectively make it a missing texture.
     *
     * @param name The name of the sampler to remove
     */
    void removeSampler(CharSequence name);

    /**
     * Loads the samplers set by {@link #setSampler(CharSequence, int)} into the shader.
     *
     * @param samplerStart The sampler to start binding to
     */
    default void bindSamplers(int samplerStart) {
        this.bindSamplers(ShaderTextureSource.GLOBAL_CONTEXT, samplerStart);
    }

    /**
     * Loads the samplers set by {@link #setSampler(CharSequence, int)} into the shader.
     *
     * @param context      The context for setting built-in shader samplers or <code>null</code> to ignore normal samplers
     * @param samplerStart The sampler to start binding to
     */
    void bindSamplers(@Nullable ShaderTextureSource.Context context, int samplerStart);

    /**
     * Clears all samplers.
     */
    void clearSamplers();

    /**
     * Sets <code>DiffuseSampler0</code>-<code>DiffuseSamplerMax</code> to the color buffers in the specified framebuffer.
     * <br>
     * Also sets <code>DiffuseDepthSampler</code> if the framebuffer has a depth attachment.
     *
     * @param framebuffer The framebuffer to bind samplers from
     */
    static void setFramebufferSamplers(class_5944 instance, AdvancedFbo framebuffer) {
        if (instance instanceof ShaderProgramImpl.Wrapper wrapper) {
            wrapper.program().setFramebufferSamplers(framebuffer);
            return;
        }

        boolean setDiffuseSampler = false;
        for (int i = 0; i < framebuffer.getColorAttachments(); i++) {
            if (!framebuffer.isColorTextureAttachment(i)) {
                continue;
            }

            AdvancedFboTextureAttachment attachment = framebuffer.getColorTextureAttachment(i);
            instance.method_34583("DiffuseSampler" + i, attachment);
            if (attachment.getName() != null) {
                instance.method_34583(attachment.getName(), attachment);
            }
            if (!setDiffuseSampler) {
                instance.method_34583("DiffuseSampler", attachment);
                setDiffuseSampler = true;
            }
        }

        if (framebuffer.isDepthTextureAttachment()) {
            AdvancedFboTextureAttachment attachment = framebuffer.getDepthTextureAttachment();
            instance.method_34583("DiffuseDepthSampler", attachment);
            if (attachment.getName() != null) {
                instance.method_34583(attachment.getName(), attachment);
            }
        }
    }
}
