package foundry.veil.impl.client.render.dynamicbuffer;

import com.google.common.base.Stopwatch;
import foundry.veil.Veil;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.ext.ShaderInstanceExtension;
import foundry.veil.impl.ThreadTaskScheduler;
import foundry.veil.impl.client.render.shader.processor.VanillaShaderProcessor;
import foundry.veil.impl.client.render.shader.program.ShaderProgramImpl;
import org.apache.commons.io.IOUtils;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.Reader;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3300;
import net.minecraft.class_4239;
import net.minecraft.class_5913;
import net.minecraft.class_5944;

import static org.lwjgl.opengl.GL20C.GL_FRAGMENT_SHADER;
import static org.lwjgl.opengl.GL20C.GL_VERTEX_SHADER;

/**
 * Compiles vanilla shaders asynchronously.
 */
public class VanillaShaderCompiler {

    private static final Set<String> LAST_FRAME_SHADERS = ConcurrentHashMap.newKeySet();

    private ThreadTaskScheduler scheduler;

    public VanillaShaderCompiler() {
    }

    private void compileShader(class_5944 shader, int activeBuffers) {
        ShaderInstanceExtension extension = (ShaderInstanceExtension) shader;
        Collection<class_2960> shaderSources = extension.veil$getShaderSources();
        class_293 vertexFormat = shader.method_35786();
        Map<String, Object> customProgramData = new HashMap<>();
        class_3300 resourceManager = class_310.method_1551().method_1478();

        VanillaShaderProcessor.setup(resourceManager);
        for (class_2960 path : shaderSources) {
            try (Reader reader = resourceManager.openAsReader(path)) {
                String source = IOUtils.toString(reader);
                class_5913 preprocessor = new class_5913() {
                    private final Set<String> importedPaths = new HashSet<>();

                    @Override
                    public String method_34233(boolean useFullPath, @NotNull String directory) {
                        if (useFullPath) {
                            directory = class_4239.method_34676(path.method_12832() + directory);
                        } else {
                            directory = class_4239.method_34676("shaders/include/" + directory);
                            if (directory.indexOf(class_2960.field_33380) != -1) {
                                class_2960 contained = class_2960.method_60654(directory);
                                String finalDirectory = directory;
                                directory = contained.method_45134(path -> path.replace(finalDirectory, path)).toString();
                            }
                        }

                        if (!this.importedPaths.add(directory)) {
                            return null;
                        }

                        class_2960 resourcelocation = class_2960.method_60654(directory);
                        try (Reader reader = resourceManager.openAsReader(resourcelocation)) {
                            return IOUtils.toString(reader);
                        } catch (IOException e) {
                            Veil.LOGGER.error("Could not open GLSL import {}: {}", directory, e.getMessage());
                            return "#error " + e.getMessage();
                        }
                    }
                };
                source = String.join("", preprocessor.method_34229(source));

                boolean vertex = path.method_12832().endsWith(".vsh");
                String processed = VanillaShaderProcessor.modify(customProgramData, shader.method_35787(), path, vertexFormat, activeBuffers, vertex ? GL_VERTEX_SHADER : GL_FRAGMENT_SHADER, source);
                class_310.method_1551().execute(() -> extension.veil$recompile(vertex, processed, activeBuffers));
            } catch (Throwable t) {
                Veil.LOGGER.error("Couldn't load vanilla shader from {}", path, t);
            }
        }
        VanillaShaderProcessor.free();
    }

    /**
     * Attempts to preload all vanilla minecraft shader files before creating the shaders on the CPU.
     *
     * @param shaders The shaders to reload
     * @return A future for when vanilla shaders have reloaded
     */
    public CompletableFuture<?> reload(Collection<class_5944> shaders) {
        if (this.scheduler != null) {
            // Cancel the previous tasks and move on
            this.scheduler.cancel();
        }

        int shaderCount = shaders.size();
        Map<String, class_5944> shaderMap = new ConcurrentHashMap<>(shaderCount);
        for (class_5944 shader : shaders) {
            shaderMap.put(shader.method_35787(), shader);
        }

        int activeBuffers = VeilRenderSystem.renderer().getDynamicBufferManger().getActiveBuffers();
        Stopwatch stopwatch = Stopwatch.createStarted();
        ThreadTaskScheduler scheduler = new ThreadTaskScheduler("VeilVanillaShaderCompile", Math.max(1, Runtime.getRuntime().availableProcessors() / 6), () -> {
            for (String lastFrameShader : LAST_FRAME_SHADERS) {
                class_5944 shader = shaderMap.remove(lastFrameShader);
                if (shader != null) {
                    return () -> this.compileShader(shader, activeBuffers);
                }
            }

            Iterator<class_5944> iterator = shaderMap.values().iterator();
            if (iterator.hasNext()) {
                class_5944 shader = iterator.next();
                iterator.remove();
                return () -> this.compileShader(shader, activeBuffers);
            }
            return null;
        });
        this.scheduler = scheduler;

        CompletableFuture<?> future = scheduler.getCompletedFuture();
        future.thenRunAsync(() -> {
            if (!scheduler.isCancelled()) {
                Veil.LOGGER.info("Compiled {} vanilla shaders in {}", shaderCount, stopwatch.stop());
            }
        }, class_310.method_1551());
        return future.isDone() ? CompletableFuture.completedFuture(null) : future;
    }

    /**
     * @return Whether shaders are currently being rendered
     */
    public boolean isCompilingShaders() {
        return this.scheduler != null && !this.scheduler.getCompletedFuture().isDone();
    }

    /**
     * Retrieves the active buffers for an existing shader program.
     *
     * @return The active buffers in the specified program
     * @since 2.3.0
     */
    public static int getActiveDynamicBuffers(class_5944 shaderInstance) {
        if (shaderInstance instanceof ShaderProgramImpl.Wrapper wrapper) {
            return wrapper.program().getActiveDynamicBuffers();
        }
        return ((ShaderInstanceExtension) shaderInstance).veil$getActiveBuffers();
    }

    @ApiStatus.Internal
    public static void markRendered(String shaderInstace) {
        if (VeilRenderSystem.renderer().getVanillaShaderCompiler().isCompilingShaders()) {
            LAST_FRAME_SHADERS.add(shaderInstace);
        }
    }

    @ApiStatus.Internal
    public static void clear() {
        LAST_FRAME_SHADERS.clear();
    }
}
