package foundry.veil.impl.client.render.framebuffer;

import com.google.common.base.Suppliers;
import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.systems.RenderSystem;
import foundry.veil.api.client.render.VeilRenderBridge;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.api.client.render.framebuffer.AdvancedFbo;
import foundry.veil.api.client.render.framebuffer.AdvancedFboAttachment;
import org.apache.commons.lang3.Validate;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.stream.IntStream;
import net.minecraft.class_276;
import net.minecraft.class_310;
import net.minecraft.class_6367;

import static org.lwjgl.opengl.ARBDirectStateAccess.glTextureParameteri;
import static org.lwjgl.opengl.GL11C.GL_OUT_OF_MEMORY;
import static org.lwjgl.opengl.GL30C.*;

/**
 * Default implementation of {@link AdvancedFbo}.
 *
 * @author Ocelot
 */
@ApiStatus.Internal
public abstract class AdvancedFboImpl implements AdvancedFbo {

    protected static final Map<Integer, String> ERRORS = Map.of(
            GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT, "GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT",
            GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT, "GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT",
            GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER, "GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER",
            GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER, "GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER",
            GL_FRAMEBUFFER_UNSUPPORTED, "GL_FRAMEBUFFER_UNSUPPORTED",
            GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE, "GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE",
            GL_FRAMEBUFFER_UNDEFINED, "GL_FRAMEBUFFER_UNDEFINED",
            GL_OUT_OF_MEMORY, "GL_OUT_OF_MEMORY"
    );

    public static final Supplier<AdvancedFbo> MAIN_WRAPPER = Suppliers.memoize(() -> VeilRenderBridge.wrap(class_310.method_1551()::method_1522));

    protected int id;
    protected final int width;
    protected final int height;
    protected final AdvancedFboAttachment[] colorAttachments;
    protected final AdvancedFboAttachment depthAttachment;
    protected final boolean hasStencil;
    protected final String debugLabel;
    protected final int clearMask;
    protected final int[] drawBuffers;
    protected int[] currentDrawBuffers;
    protected final Supplier<Wrapper> wrapper;

    public AdvancedFboImpl(int width, int height, AdvancedFboAttachment[] colorAttachments, @Nullable AdvancedFboAttachment depthAttachment, @Nullable String debugLabel) {
        this.id = -1;
        this.width = width;
        this.height = height;
        this.colorAttachments = colorAttachments;
        this.depthAttachment = depthAttachment;
        this.hasStencil = depthAttachment != null && (depthAttachment.getFormat() == GL_DEPTH24_STENCIL8 || depthAttachment.getFormat() == GL_DEPTH32F_STENCIL8);
        this.debugLabel = debugLabel;

        int mask = 0;
        if (this.hasColorAttachment(0)) {
            mask |= GL_COLOR_BUFFER_BIT;
        }
        if (this.hasDepthAttachment()) {
            mask |= GL_DEPTH_BUFFER_BIT;
        }
        if (this.hasStencilAttachment()) {
            mask |= GL_STENCIL_BUFFER_BIT;
        }
        this.clearMask = mask;
        this.drawBuffers = IntStream.range(0, this.colorAttachments.length)
                .map(i -> GL_COLOR_ATTACHMENT0 + i)
                .toArray();
        this.wrapper = Suppliers.memoize(() -> new Wrapper(this));
    }

    @Override
    public void bind(boolean setViewport) {
        GlStateManager._glBindFramebuffer(GL_FRAMEBUFFER, this.id);
        if (setViewport) {
            RenderSystem.viewport(0, 0, this.width, this.height);
        }
    }

    @Override
    public void bindDraw(boolean setViewport) {
        GlStateManager._glBindFramebuffer(GL_DRAW_FRAMEBUFFER, this.id);
        if (setViewport) {
            RenderSystem.viewport(0, 0, this.width, this.height);
        }
    }

    @Override
    public void free() {
        if (this.id == -1) {
            return;
        }
        glDeleteFramebuffers(this.id);
        this.id = -1;
        for (AdvancedFboAttachment attachment : this.colorAttachments) {
            attachment.free();
        }
        if (this.depthAttachment != null) {
            this.depthAttachment.free();
        }
    }

    @Override
    public int getId() {
        return this.id;
    }

    @Override
    public int getWidth() {
        return this.width;
    }

    @Override
    public int getHeight() {
        return this.height;
    }

    @Override
    public int getColorAttachments() {
        return this.colorAttachments.length;
    }

    @Override
    public int getClearMask() {
        return this.clearMask;
    }

    @Override
    public int[] getDrawBuffers() {
        return this.drawBuffers;
    }

    @Override
    public boolean hasColorAttachment(int attachment) {
        return attachment >= 0 && attachment < this.colorAttachments.length;
    }

    @Override
    public boolean hasDepthAttachment() {
        return this.depthAttachment != null;
    }

    @Override
    public boolean hasStencilAttachment() {
        return this.hasStencil;
    }

    @Override
    public AdvancedFboAttachment getColorAttachment(int attachment) {
        Validate.isTrue(this.hasColorAttachment(attachment), "Color attachment " + attachment + " does not exist.");
        return this.colorAttachments[attachment];
    }

    @Override
    public AdvancedFboAttachment getDepthAttachment() {
        return Objects.requireNonNull(this.depthAttachment, "Depth attachment does not exist.");
    }

    @Override
    public @Nullable String getDebugLabel() {
        return this.debugLabel;
    }

    @Override
    public Wrapper toRenderTarget() {
        return this.wrapper.get();
    }

    @ApiStatus.Internal
    public static Builder copy(class_276 parent) {
        if (parent instanceof Wrapper wrapper) {
            AdvancedFbo fbo = wrapper.fbo();
            return new Builder(fbo.getWidth(), fbo.getHeight()).addAttachments(fbo);
        }
        return new Builder(parent.field_1482, parent.field_1481).addAttachments(parent);
    }

    /**
     * A vanilla {@link class_276} wrapper of the {@link AdvancedFboImpl}.
     *
     * @author Ocelot
     * @see AdvancedFbo
     * @since 3.0.0
     */
    public static class Wrapper extends class_6367 {

        private final float[] clearChannels = new float[]{1.0F, 1.0F, 1.0F, 0.0F};

        private final AdvancedFboImpl fbo;

        private Wrapper(AdvancedFboImpl fbo) {
            super(fbo.width, fbo.height, fbo.hasDepthAttachment(), class_310.field_1703);
            this.fbo = fbo;
            this.field_1482 = this.fbo.getWidth();
            this.field_1481 = this.fbo.getHeight();
            this.field_1480 = this.field_1482;
            this.field_1477 = this.field_1481;
            this.field_1476 = this.fbo.id;
            this.field_1475 = this.fbo.isColorTextureAttachment(0) ? this.fbo.getColorTextureAttachment(0).method_4624() : 0;
            this.field_1474 = this.fbo.isDepthTextureAttachment() ? this.fbo.getDepthTextureAttachment().method_4624() : 0;
        }

        @Override
        public void method_1234(int width, int height, boolean onMac) {
        }

        @Override
        public void method_1238() {
            throw new UnsupportedOperationException("Cannot destroy advanced fbo from wrapper");
        }

        @Override
        public void method_1231(int width, int height, boolean onMac) {
        }

        @Override
        public void method_58226(int framebufferFilter) {
            this.field_1483 = framebufferFilter;
            if (VeilRenderSystem.directStateAccessSupported()) {
                for (int i = 0; i < this.fbo.getColorAttachments(); i++) {
                    int texture = this.fbo.getColorTextureAttachment(i).method_4624();
                    glTextureParameteri(texture, GL_TEXTURE_MIN_FILTER, framebufferFilter);
                    glTextureParameteri(texture, GL_TEXTURE_MAG_FILTER, framebufferFilter);
                    glTextureParameteri(texture, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
                    glTextureParameteri(texture, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
                }
            } else {
                for (int i = 0; i < this.fbo.getColorAttachments(); i++) {
                    this.fbo.getColorAttachment(i).bindAttachment();
                    GlStateManager._texParameter(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, framebufferFilter);
                    GlStateManager._texParameter(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, framebufferFilter);
                    GlStateManager._texParameter(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
                    GlStateManager._texParameter(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
                }
                GlStateManager._bindTexture(0);
            }
        }

        @Override
        public void method_1236(float red, float green, float blue, float alpha) {
            super.method_1236(red, green, blue, alpha);
            this.clearChannels[0] = red;
            this.clearChannels[1] = green;
            this.clearChannels[2] = blue;
            this.clearChannels[3] = alpha;
        }

        @Override
        public void method_1230(boolean clearError) {
            RenderSystem.assertOnRenderThreadOrInit();
            this.fbo.clear(this.clearChannels[0], this.clearChannels[1], this.clearChannels[2], this.clearChannels[3], this.fbo.getClearMask());
        }

        @Override
        public void method_35610() {
            if (this.fbo.hasColorAttachment(0)) {
                this.fbo.getColorAttachment(0).bindAttachment();
            }
        }

        @Override
        public void method_1242() {
            if (this.fbo.hasColorAttachment(0)) {
                this.fbo.getColorAttachment(0).unbindAttachment();
            }
        }

        @Override
        public void method_1235(boolean setViewport) {
            this.fbo.bind(setViewport);
        }

        /**
         * @return The backing advanced fbo
         */
        public AdvancedFboImpl fbo() {
            return this.fbo;
        }
    }
}
