package foundry.veil.api.client.render.shader;

import foundry.veil.Veil;
import foundry.veil.impl.client.render.shader.modifier.*;
import io.github.ocelot.glslprocessor.api.node.GlslTree;
import org.apache.commons.io.IOUtils;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.Reader;
import java.util.*;
import java.util.regex.Pattern;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3544;
import net.minecraft.class_3695;
import net.minecraft.class_4080;
import net.minecraft.class_5912;
import net.minecraft.class_7654;

/**
 * Manages modifications for both vanilla and Veil shader files.
 *
 * @author Ocelot
 */
public class ShaderModificationManager extends class_4080<ShaderModificationManager.Preparations> {

    public static final class_7654 MODIFIER_LISTER = new class_7654("pinwheel/shader_modifiers", ".txt");
    private static final Map<String, String> NEXT_STAGES = Map.of(
            "vsh", "gsh",
            "gsh", "fsh"
    );
    private static final Pattern OUT_PATTERN = Pattern.compile("out ");

    private Map<class_2960, List<ShaderModification>> shaders;
    private Map<ShaderModification, class_2960> names;

    public ShaderModificationManager() {
        this.shaders = Collections.emptyMap();
        this.names = Collections.emptyMap();
    }

    /**
     * Applies all shader modifiers to the specified shader source.
     *
     * @param shaderId The id of the shader to get modifiers for
     * @param tree     The shader source tree
     * @param flags    Additional flags for applying modifiers
     * @see ShaderModification
     */
    public void applyModifiers(class_2960 shaderId, GlslTree tree, int flags) {
        Collection<ShaderModification> modifiers = this.getModifiers(shaderId);
        if (modifiers.isEmpty()) {
            return;
        }

        try {
            VeilJobParameters parameters = new VeilJobParameters(this, shaderId, flags);
            for (ShaderModification modifier : modifiers) {
                modifier.inject(tree, parameters);
            }
        } catch (Exception e) {
            Veil.LOGGER.error("Failed to transform shader: {}", shaderId, e);
        }
    }

    /**
     * Retrieves all modifiers for the specified shader.
     *
     * @param shaderId The shader to get all modifiers for
     * @return The modifiers applied to the specified shader
     */
    public List<ShaderModification> getModifiers(class_2960 shaderId) {
        return this.shaders.getOrDefault(shaderId, Collections.emptyList());
    }

    /**
     * Retrieves the id of the specified modifier.
     *
     * @param modification The modification to get the id of
     * @return The id of that modification or <code>null</code> if unregistered
     */
    public @Nullable class_2960 getModifierId(ShaderModification modification) {
        return this.names.get(modification);
    }

    private @Nullable class_2960 getNextStage(class_2960 shader, class_5912 resourceProvider) {
        String[] parts = shader.method_12832().split("\\.");
        String extension = parts[parts.length - 1].toLowerCase(Locale.ROOT);

        while (extension != null) {
            extension = NEXT_STAGES.get(extension);

            class_2960 nextShader = class_2960.method_60655(shader.method_12836(), shader.method_12832().substring(0, shader.method_12832().length() - 3) + extension);
            if (resourceProvider.method_14486(nextShader).isPresent()) {
                return nextShader;
            }
        }
        return null;
    }

    @Override
    protected @NotNull Preparations method_18789(@NotNull class_3300 resourceManager, @NotNull class_3695 profilerFiller) {
        Map<class_2960, List<ShaderModification>> modifiers = new HashMap<>();
        Map<ShaderModification, class_2960> names = new HashMap<>();

        for (Map.Entry<class_2960, class_3298> entry : MODIFIER_LISTER.method_45113(resourceManager).entrySet()) {
            class_2960 file = entry.getKey();
            class_2960 id = MODIFIER_LISTER.method_45115(file);

            try {
                String[] parts = id.method_12832().split("/", 2);
                if (parts.length < 2) {
                    Veil.LOGGER.warn("Ignoring shader modifier {}. Expected format to be located in shader_modifiers/domain/shader_path.vsh.txt", file);
                    continue;
                }

                class_2960 shaderId = class_2960.method_60655(parts[0], parts[1]);
                try (Reader reader = entry.getValue().method_43039()) {
                    ShaderModification modification = ShaderModification.parse(IOUtils.toString(reader), shaderId.method_12832().endsWith(".vsh"));
                    List<ShaderModification> modifications = modifiers.computeIfAbsent(shaderId, name -> new LinkedList<>());

                    if (modification instanceof ReplaceShaderModification) {
                        // TODO This doesn't respect priority
                        modifications.clear();
                    }
                    if (modifications.size() != 1 || !(modifications.getFirst() instanceof ReplaceShaderModification)) {
                        modifications.add(modification);
                    }
                    names.put(modification, id);
                }
            } catch (Exception e) {
                Veil.LOGGER.error("Couldn't parse data file {} from {}", id, file, e);
            }
        }

        // Inject inputs to next shader stage
        for (Map.Entry<class_2960, List<ShaderModification>> entry : new HashMap<>(modifiers).entrySet()) {
            class_2960 nextStage = null;

            for (ShaderModification modification : entry.getValue()) {
                if (!(modification instanceof SimpleShaderModification simpleMod)) {
                    continue;
                }

                String output = simpleMod.getOutput();
                if (class_3544.method_15438(output)) {
                    continue;
                }

                if (nextStage == null) {
                    nextStage = this.getNextStage(entry.getKey(), resourceManager);
                }
                if (nextStage == null) {
                    // No need to inject in into next shader
                    break;
                }

                InputShaderModification input = new InputShaderModification(simpleMod.priority(), OUT_PATTERN.matcher(simpleMod.fillPlaceholders(simpleMod.getOutput())).replaceAll("in "));
                modifiers.computeIfAbsent(nextStage, unused -> new LinkedList<>()).add(input);
                names.put(input, names.get(simpleMod));
            }
        }
        modifiers.values().forEach(modifications -> modifications.sort(Comparator.comparingInt(ShaderModification::priority).thenComparing(names::get)));

        return new Preparations(modifiers, names);
    }

    @Override
    protected void apply(@NotNull Preparations preparations, @NotNull class_3300 resourceManager, @NotNull class_3695 profilerFiller) {
        this.shaders = Collections.unmodifiableMap(preparations.shaders);
        this.names = Collections.unmodifiableMap(preparations.names);
        Veil.LOGGER.info("Loaded {} shader modifications", this.names.size());
    }

    @ApiStatus.Internal
    public record Preparations(Map<class_2960, List<ShaderModification>> shaders,
                               Map<ShaderModification, class_2960> names) {
    }
}
