package foundry.veil.api.client.render.shader.compiler;

import foundry.veil.api.client.render.shader.program.ShaderProgram;
import foundry.veil.api.client.render.shader.uniform.ShaderUniform;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.system.NativeResource;

import java.util.Set;
import net.minecraft.class_2960;

import static org.lwjgl.opengl.GL20C.glDeleteShader;

/**
 * A shader instance that has additional pre-compiled data.
 * {@link #apply(ShaderProgram)} should be called after this shader is attached to a program.
 *
 * @param sourceFile             The source file this shader was compiled from or <code>null</code> if the shader has no file
 * @param id                     The OpenGL id of the shader
 * @param uniformBindings        The bindings set by the shader
 * @param definitionDependencies The shader pre-definitions this shader is dependent on
 * @param includes               All shader imports included in this file
 * @author Ocelot
 */
public record CompiledShader(@Nullable class_2960 sourceFile,
                             int id,
                             Object2IntMap<String> uniformBindings,
                             Set<String> definitionDependencies,
                             Set<class_2960> includes) implements NativeResource {

    /**
     * Applies the additional attributes of this shader to the specified program.
     */
    public void apply(ShaderProgram program) {
        for (Object2IntMap.Entry<String> entry : this.uniformBindings.object2IntEntrySet()) {
            ShaderUniform uniform = program.getUniform(entry.getKey());
            if (uniform != null) {
                uniform.setInt(entry.getIntValue());
            }
        }
    }

    @Override
    public void free() {
        glDeleteShader(this.id);
    }
}
